﻿//FileViewerWindow
//PanotiSoft 
//Markus Zerhusen
//20.09.2017

using System;
using System.IO;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;

namespace FileViewer
{
  public partial class FileViewerWindow: Form, IFileViewerProgressBar
  {
    #region Private Enum

    private enum FileFormat {
      Picture = 0,
      Animation,
      HelpReader,
      HelpWriter,
      CompressText,
      FormatText,
      PicturePaintCollection,
      PicturePaintPoject,
      ProtectedData,
      VideoThumbnailImageData,
    }

    #endregion

    #region Private Fields

    private Boolean MyRun;
    private String MyFilePath;
    private Int64 MyFileSize;

    private OpenFileDialog MyOpenFile;
    private SaveFileDialog MySaveFile;

    private FileViewerImageWindow MyImageViewer;
    private FileViewerTextWindow MyTextViewer;
    private FileViewerPassWindow MyPassViewer;
    private FileViewerProtectedWindow MyProtectedViewer;
    private FileViewerColumnWindow MyColumnViewer;

    private FileViewerObject MyViewerObject;

    #endregion


    #region Contructor

    public FileViewerWindow() {
      InitializeComponent();

      this.Icon = Properties.Resources.IconMain;

      this.ComboBoxFormat.SelectedIndex = 0;

      this.MyOpenFile = new OpenFileDialog();
      this.MyOpenFile.InitialDirectory = Directory.GetDirectoryRoot(Application.StartupPath);
      this.MyOpenFile.Filter = "2D Picture (*.tdp)|*.tdp|" +
                        "2D Animation (*.tda)|*.tda|" +
                        "2D Help Reader (*.tdhr)|*.tdhr|" +
                        "2D Help Writer (*.tdhw)|*.tdhw|" +
                        "2D Compress Text (*.tdct)|*.tdct|" +
                        "2D Format Text (*.tdft)|*.tdft|" +
                        "Picture Paint Collection (*.ppc)|*.ppc|" +
                        "Picture Paint Poject (*.ppp)|*.ppp|" +
                        "Protected Data (*.pdata)|*.pdata|" +
                        "Video Thumbnail Image Data (*.avi,*.mp4,...)|*.mp4;*.avi;*.wmv;*.mpeg|" + 
                        "All files (*.*)|*.*";
      this.MyOpenFile.FilterIndex = 10;

      this.MySaveFile = new SaveFileDialog();
      this.MySaveFile.InitialDirectory = Directory.GetDirectoryRoot(Application.StartupPath);
      this.MySaveFile.Filter = "Sound (*.wav)|*.wav|" +            
                        "All files (*.*)|*.*";
      this.MySaveFile.FilterIndex = 2;

      this.MyImageViewer = new FileViewerImageWindow();
      this.MyTextViewer = new FileViewerTextWindow();
      this.MyPassViewer = new FileViewerPassWindow();
      this.MyProtectedViewer = new FileViewerProtectedWindow();
      this.MyColumnViewer = new FileViewerColumnWindow();

    }

    #endregion


    #region Private Methods

    private Byte[] MakeDecompressMemory(Byte[] ByteArray, Int32 Count, Int32 ValueMax) {
      Int32 BitCountMax = (Int32) Math.Ceiling(Math.Log(ValueMax, 2));

      if(BitCountMax == 8) return ByteArray;

      Byte[] ValueArray = new Byte[Count];

      if(BitCountMax == 0) return ValueArray;

      Int32 BitCount = 0;
      Int32 Index = 0;

      for(Int32 i = 0; i < Count; i++) {
        Byte Value = ByteArray[Index];

        Value <<= BitCount;

        BitCount += BitCountMax;

        if(BitCount >= 8) {
          if(BitCount == 8) {
            BitCount = 0;

            Value >>= 8 - BitCountMax;

            Index++;
          } else {
            BitCount -= 8;

            Byte ValueNext = ByteArray[++Index];
            ValueNext >>= 8 - BitCount;

            Value >>= 8 - BitCountMax;
            Value |= ValueNext;
          }
        } else {
          Value >>= 8 - BitCountMax;
        }

        ValueArray[i] = Value;
      }
      return ValueArray;
    }

    private void UpdateComboBoxFormat(String Filename) {
      String Extension = Path.GetExtension(Filename);

      switch(Extension.ToLower()) {
        case ".tdp": {
          this.ComboBoxFormat.SelectedIndex = (Int32) FileFormat.Picture;
          break;
        }
        case ".tda": {
          this.ComboBoxFormat.SelectedIndex = (Int32) FileFormat.Animation;
          break;
        }
        case ".tdhr": {
          this.ComboBoxFormat.SelectedIndex = (Int32) FileFormat.HelpReader;
          break;
        }
        case ".tdhw": {
          this.ComboBoxFormat.SelectedIndex = (Int32) FileFormat.HelpWriter;
          break;
        }
        case ".tdct": {
          this.ComboBoxFormat.SelectedIndex = (Int32) FileFormat.CompressText;
          break;
        }
        case ".tdft": {
          this.ComboBoxFormat.SelectedIndex = (Int32) FileFormat.FormatText;
          break;
        }
        case ".ppc": {
          this.ComboBoxFormat.SelectedIndex = (Int32) FileFormat.PicturePaintCollection;
          break;
        }
        case ".ppp": {
          this.ComboBoxFormat.SelectedIndex = (Int32) FileFormat.PicturePaintPoject;
          break;
        }
        case ".pdata": {
          this.ComboBoxFormat.SelectedIndex = (Int32) FileFormat.ProtectedData;
          break;
        }
        case ".avi": {
          this.ComboBoxFormat.SelectedIndex = (Int32) FileFormat.VideoThumbnailImageData;
          break;
        }
        case ".mp4": {
          this.ComboBoxFormat.SelectedIndex = (Int32) FileFormat.VideoThumbnailImageData;
          break;
        }
        case ".wmv": {
          this.ComboBoxFormat.SelectedIndex = (Int32) FileFormat.VideoThumbnailImageData;
          break;
        }
      }
    }


    private void ShowAnimationItems(MemoryStream Stream) {
      try {
        BinaryReader Reader = new BinaryReader(Stream);

        List<String> InfoList = new List<String>();

        Int32 Frames = 0;

        while(Stream.Position < Stream.Length) {
          Byte ItemMode = Reader.ReadByte();

          switch(ItemMode) {
            case 0: { //Empty
              InfoList.Add("Mode:0  Frame:" + Frames);
              break;
            }
            case 1: { //Position
              Int16 ItemPosX = Reader.ReadInt16();
              Int16 ItemPosY = Reader.ReadInt16();

              InfoList.Add("Mode:1  Frame:" + Frames + "  X:" + ItemPosX + "  Y:" + ItemPosY);
              break;
            }
            case 2: { //Fast: Position, Size
              Int16 ItemPosX = Reader.ReadInt16();
              Int16 ItemPosY = Reader.ReadInt16();
              Int16 ItemWidth = Reader.ReadInt16();
              Int16 ItemHeight = Reader.ReadInt16();

              if(ItemWidth < 0 || ItemHeight < 0) throw new Exception("Wrong value:  frame: " + Frames + "  x: " + ItemPosX + "  y: " + ItemPosY + "  Width:" + ItemWidth + "  Height:" + ItemHeight);

              InfoList.Add("Mode:2  Frame:" + Frames + "  X:" + ItemPosX + "  Y:" + ItemPosY + "  Width:" + ItemWidth + "  Height:" + ItemHeight);
              break;
            }
            case 3: { //Quality: Position, Size
              Int16 ItemPosX = Reader.ReadInt16();
              Int16 ItemPosY = Reader.ReadInt16();
              Int16 ItemWidth = Reader.ReadInt16();
              Int16 ItemHeight = Reader.ReadInt16();

              if(ItemWidth < 0 || ItemHeight < 0) throw new Exception("Wrong value:  frame: " + Frames + "  x: " + ItemPosX + "  y: " + ItemPosY + "  Width:" + ItemWidth + "  Height:" + ItemHeight);

              InfoList.Add("Mode:3  Frame:" + Frames + "  X:" + ItemPosX + "  Y:" + ItemPosY + "  Width:" + ItemWidth + "  Height:" + ItemHeight);
              break;
            }
            case 4: { //HighQuality: Position, Size
              Int16 ItemPosX = Reader.ReadInt16();
              Int16 ItemPosY = Reader.ReadInt16();
              Int16 ItemWidth = Reader.ReadInt16();
              Int16 ItemHeight = Reader.ReadInt16();

              if(ItemWidth < 0 || ItemHeight < 0) throw new Exception("Wrong value:  frame: " + Frames + "  x: " + ItemPosX + "  y: " + ItemPosY + "  Width:" + ItemWidth + "  Height:" + ItemHeight);

              InfoList.Add("Mode:4  Frame:" + Frames + "  X:" + ItemPosX + "  Y:" + ItemPosY + "  Width:" + ItemWidth + "  Height:" + ItemHeight);
              break;
            }
            case 5: { //IndexPosition: Position, Index
              Int16 ItemPosX = Reader.ReadInt16();
              Int16 ItemPosY = Reader.ReadInt16();
              Int16 ItemIndex = Reader.ReadInt16();

              if(ItemIndex < 0) throw new Exception("Wrong value:  frame: " + Frames + "  x: " + ItemPosX + "  y: " + ItemPosY + "  index: " + ItemIndex);

              InfoList.Add("Mode:5  Frame:" + Frames + "  X:" + ItemPosX + "  Y:" + ItemPosY + "  Index:" + ItemIndex);
              break;
            }
            case 6: { //IndexFast: Position, Size, Index
              Int16 ItemPosX = Reader.ReadInt16();
              Int16 ItemPosY = Reader.ReadInt16();
              Int16 ItemWidth = Reader.ReadInt16();
              Int16 ItemHeight = Reader.ReadInt16();
              Int16 ItemIndex = Reader.ReadInt16();

              if(ItemWidth < 0 || ItemHeight < 0 || ItemIndex < 0) throw new Exception("Wrong value:  frame: " + Frames + "  x: " + ItemPosX + "  y: " + ItemPosY + "  Width:" + ItemWidth + "  Height:" + ItemHeight + "  Index:" + ItemIndex);

              InfoList.Add("Mode:6  Frame:" + Frames + "  X:" + ItemPosX + "  Y:" + ItemPosY + "  Width:" + ItemWidth + "  Height:" + ItemHeight + "  Index:" + ItemIndex);
              break;
            }
            case 7: { //IndexQuality: Position, Size, Index
              Int16 ItemPosX = Reader.ReadInt16();
              Int16 ItemPosY = Reader.ReadInt16();
              Int16 ItemWidth = Reader.ReadInt16();
              Int16 ItemHeight = Reader.ReadInt16();
              Int16 ItemIndex = Reader.ReadInt16();

              if(ItemWidth < 0 || ItemHeight < 0 || ItemIndex < 0) throw new Exception("Wrong value:  frame: " + Frames + "  x: " + ItemPosX + "  y: " + ItemPosY + "  Width:" + ItemWidth + "  Height:" + ItemHeight + "  Index:" + ItemIndex);

              InfoList.Add("Mode:7  Frame:" + Frames + "  X:" + ItemPosX + "  Y:" + ItemPosY + "  Width:" + ItemWidth + "  Height:" + ItemHeight + "  Index:" + ItemIndex);
              break;
            }
            case 8: { //IndexHighQuality: Position, Size, Index
              Int16 ItemPosX = Reader.ReadInt16();
              Int16 ItemPosY = Reader.ReadInt16();
              Int16 ItemWidth = Reader.ReadInt16();
              Int16 ItemHeight = Reader.ReadInt16();
              Int16 ItemIndex = Reader.ReadInt16();

              if(ItemWidth < 0 || ItemHeight < 0 || ItemIndex < 0) throw new Exception("Wrong value:  frame: " + Frames + "  x: " + ItemPosX + "  y: " + ItemPosY + "  Width:" + ItemWidth + "  Height:" + ItemHeight + "  Index:" + ItemIndex);

              InfoList.Add("Mode:8  Frame:" + Frames + "  X:" + ItemPosX + "  Y:" + ItemPosY + "  Width:" + ItemWidth + "  Height:" + ItemHeight + "  Index:" + ItemIndex);
              break;
            }
          }
          Frames++;
        }

        Stream.Position = 0;

        this.MyTextViewer.Run(this, "ItemMemory", null, InfoList.ToArray());
      } catch(Exception e) {
        MessageBox.Show(e.Message, "  Error: Reading Animation Data", MessageBoxButtons.OK, MessageBoxIcon.Error);
      }
    }

    private void ShowSaveRunFile(String DefaultName, Byte[] ByteArray) {
      this.MySaveFile.FileName = DefaultName;

      switch(this.MySaveFile.ShowDialog(this)) {
        case DialogResult.OK: {
          try {
            FileStream File = new FileStream(this.MySaveFile.FileName, FileMode.Create, FileAccess.ReadWrite);

            File.Write(ByteArray, 0, ByteArray.Length);
            File.Close();
          } catch(Exception e) {
            MessageBox.Show(e.Message, "  Error: Save File", MessageBoxButtons.OK, MessageBoxIcon.Error);
            return;
          }

          if(MessageBox.Show("Would you like to open the file?", "", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes) {
            try {
              Process.Start(new ProcessStartInfo(this.MySaveFile.FileName));
            } catch(Exception e) {
              MessageBox.Show(e.Message, "  Error: Open File", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
          }
          break;
        }
      }
    }

    private void ShowProtectedData(FileViewerProtectedDataObject DataObject) {
      String[] TextArray = new String[DataObject.ValueArray.Length];

      for(Int32 i = 0; i < DataObject.ValueArray.Length; i++) {
        TextArray[i] = DataObject.Name + ": " + i + "   0x" + DataObject.ReadArray[i].ToString("X2") + " XOR 0x" + DataObject.CodeArray[i].ToString("X2") + " = 0x" + DataObject.ValueArray[i].ToString("X2");
      }

      this.MyTextViewer.Run(this, DataObject.Title, null, TextArray);
    }

    private void ShowProtectedFile(FileViewerProtectedFileObject FileObject) {
      this.MySaveFile.FileName = FileObject.Name;

      switch(this.MySaveFile.ShowDialog(this)) {
        case DialogResult.OK: {
          if(!FileObject.Data.SaveFile(this.MySaveFile.FileName, FileObject.Position, FileObject.Size)) {
            if(File.Exists(this.MySaveFile.FileName)) {
              try {
                File.Delete(this.MySaveFile.FileName);
              } catch(Exception e) {
                MessageBox.Show(e.Message, "  Error: Delete File", MessageBoxButtons.OK, MessageBoxIcon.Error);
              }
            }
            return;
          }
          if(MessageBox.Show("Would you like to open the file?", "", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes) {
            try {
              Process.Start(new ProcessStartInfo(this.MySaveFile.FileName));
            } catch(Exception e) {
              MessageBox.Show(e.Message, "  Error: Open File", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
          }
          break;
        }
      }
    }

    private void ShowFormatText(FileViewerTDFormatTextObject TextObject) {
      if(TextObject.LetterInfoArray == null && TextObject.LetterSizeArray == null) {
        String Text = String.Empty;

        for(Int32 i = 0; i < TextObject.LetterArray.Length; i++) 
          Text += (Char) TextObject.LetterArray[i];

        this.MyTextViewer.Run(this, TextObject.Title, Text, null);
        return;
      }

      if(TextObject.LetterInfoArray != null && TextObject.FontLetterArray != null) {
        Int32 Index = 0;
        Int32 Length = TextObject.LetterInfoArray.Length / 5;

        String[] TextArray = new String[Length];

        for(Int32 i = 0; i < Length; i++) {
          String Info = String.Empty;

          Byte TypeID = TextObject.LetterInfoArray[Index];

          switch(TypeID & 0x03) {
            case 0: { //Letter
              Info = "L";
              break;
            }
            case 1: { //Seperator
              Info = "SE";
              break;
            }
            case 2: { //Space
              Info = "SP";
              break;
            }
            case 3: { //Control
              Info = "CT";
              break;
            }
          }

          Byte FontID = TextObject.LetterInfoArray[Index + 1];
          Byte ColorID = TextObject.LetterInfoArray[Index + 2];
          Byte LetterIDLow = TextObject.LetterInfoArray[Index + 3];
          Byte LetterIDHigh = TextObject.LetterInfoArray[Index + 4];

          UInt16 LetterID = (UInt16) (LetterIDHigh * 0x0100 + LetterIDLow);

          UInt16 Letter = TextObject.FontLetterArray[FontID][LetterID];

          if((TypeID & 0x04) == 0x04)
            Info += ",NL";

          if((TypeID & 0x03) == 0x03)
            TextArray[i] = i + ": 0x" + Letter.ToString("X2") + " " + Info + " F" + FontID + " C" + ColorID + " L" + LetterID;
          else
            TextArray[i] = i + ": '" + ((Char) Letter) + "' " + Info + " F" + FontID + " C" + ColorID + " L" + LetterID;

          Index += 5;
        }

        this.MyTextViewer.Run(this, TextObject.Title, null, TextArray);
        return;
      }

      if(TextObject.LetterInfoArray != null && TextObject.LetterSizeArray == null) {
        String[] TextArray = new String[TextObject.LetterArray.Length];
        Int32 Index = 0;

        for(Int32 i = 0; i < TextObject.LetterArray.Length; i++) {
          String Info = String.Empty;

          switch(TextObject.LetterInfoArray[Index] & 0x03) {
            case 0: { //Letter
              Info = "L";
              break;
            }
            case 1: { //Seperator
              Info = "SE";
              break;
            }
            case 2: { //Space
              Info = "SP";
              break;
            }
            case 3: { //Control
              Info = "CT";
              break;
            }
          }

          if((TextObject.LetterInfoArray[Index] & 0x04) == 0x04)
            Info += ",NL";

          if((TextObject.LetterInfoArray[Index] & 0x03) == 0x03)
            TextArray[i] = i + ": 0x" + TextObject.LetterArray[i].ToString("X2") + " " + Info + " F" + TextObject.LetterInfoArray[Index + 1] + " C" + TextObject.LetterInfoArray[Index + 2];
          else
            TextArray[i] = i + ": '" + ((Char) TextObject.LetterArray[i]) + "' " + Info + " F" + TextObject.LetterInfoArray[Index + 1] + " C" + TextObject.LetterInfoArray[Index + 2];
          
          Index += 3;
        }

        this.MyTextViewer.Run(this, TextObject.Title, null, TextArray);
        return;
      }

      if(TextObject.LetterSizeArray != null) {
        String[] TextArray = new String[TextObject.LetterArray.Length];
        Int32 Index = 0;

        for(Int32 i = 0; i < TextObject.LetterArray.Length; i++) {
          if((TextObject.LetterInfoArray[Index] & 0x03) == 0x03)
            TextArray[i] = i + ": 0x" + TextObject.LetterArray[i].ToString("X2") + " (" + TextObject.LetterSizeArray[Index] + "," + TextObject.LetterSizeArray[Index + 1] + "," + TextObject.LetterSizeArray[Index + 2] + ") ";
          else
            TextArray[i] = i + ": '" + ((Char) TextObject.LetterArray[i]) + "' (" + TextObject.LetterSizeArray[Index] + "," + TextObject.LetterSizeArray[Index + 1] + "," + TextObject.LetterSizeArray[Index + 2] + ") ";

          Index += 3;
        }

        this.MyTextViewer.Run(this, TextObject.Title, null, TextArray);
      }
    }

    private void ShowCompressText(FileViewerTDCompressTextObject TextObject) {
      if(TextObject.LetterArray != null && TextObject.LetterSizeArray == null) {
        String Text = String.Empty;

        for(Int32 i = 0; i < TextObject.LetterArray.Length; i++)
          Text += (Char) TextObject.LetterArray[i];

        this.MyTextViewer.Run(this, TextObject.Title, Text, null);
        return;
      }

      if(TextObject.LetterArray != null && TextObject.LetterSizeArray != null) {
        String[] TextArray = new String[TextObject.LetterArray.Length];
        Int32 Index = 0;

        for(Int32 i = 0; i < TextObject.LetterArray.Length; i++) {
          Char Letter = (Char) TextObject.LetterArray[i];

          switch(TextObject.LetterArray[i]) {
            case 0x0D: {
              TextArray[i] = i + ": 0x0D (" + TextObject.LetterSizeArray[Index] + "," + TextObject.LetterSizeArray[Index + 1] + "," + TextObject.LetterSizeArray[Index + 2] + ") ";
              break;
            }
            case 0x0A: {
              TextArray[i] = i + ": 0x0A (" + TextObject.LetterSizeArray[Index] + "," + TextObject.LetterSizeArray[Index + 1] + "," + TextObject.LetterSizeArray[Index + 2] + ") ";
              break;
            }
            default: {
              TextArray[i] = i + ": '" + Letter + "' (" + TextObject.LetterSizeArray[Index] + "," + TextObject.LetterSizeArray[Index + 1] + "," + TextObject.LetterSizeArray[Index + 2] + ") ";
              break;
            }
          }
          Index += 3;
        }

        this.MyTextViewer.Run(this, TextObject.Title, null, TextArray);
        return;
      }

      if(TextObject.ValueArray != null) {
        String[] TextArray = new String[TextObject.LetterLength];

        switch(TextObject.Type) {
          case FileViewerTDCompressTextType.Info: {
            Byte[] ValueArray;

            if(TextObject.UseInfo2Bit)
              ValueArray = this.MakeDecompressMemory(TextObject.ValueArray, TextObject.LetterLength, 4);
            else
              ValueArray = this.MakeDecompressMemory(TextObject.ValueArray, TextObject.LetterLength, 8);

            for(Int32 i = 0; i < TextObject.LetterLength; i++) {
              String Info = String.Empty;

              switch(ValueArray[i] & 0x03) {
                case 0: { //Letter
                  Info = "L";
                  break;
                }
                case 1: { //Seperator
                  Info = "SE";
                  break;
                }
                case 2: { //Space
                  Info = "SP";
                  break;
                }
                case 3: { //Control
                  Info = "CT";
                  break;
                }
              }

              if((ValueArray[i] & 0x04) == 0x04)
                Info += ",NL";

              TextArray[i] = i + ": " + ValueArray[i] + " " + Info;
            }
            break;
          }
          case FileViewerTDCompressTextType.Font: {
            Byte[] ValueArray = this.MakeDecompressMemory(TextObject.ValueArray, TextObject.LetterLength, TextObject.ValueMax);

            for(Int32 i = 0; i < TextObject.LetterLength; i++) 
              TextArray[i] = i + ": F" + ValueArray[i];
            break;
          }
          case FileViewerTDCompressTextType.Color: {
            Byte[] ValueArray = this.MakeDecompressMemory(TextObject.ValueArray, TextObject.LetterLength, TextObject.ValueMax);

            for(Int32 i = 0; i < TextObject.LetterLength; i++) 
              TextArray[i] = i + ": C" + ValueArray[i]; 
            break;
          }
          case FileViewerTDCompressTextType.Index: {
            if(TextObject.ValueMax > 256) {
              MessageBox.Show("Indexes for more than 256 different characters can not be displayed with this program.", "Information", MessageBoxButtons.OK, MessageBoxIcon.Information);
              return;
            }

            Byte[] ValueArray = this.MakeDecompressMemory(TextObject.ValueArray, TextObject.LetterLength, TextObject.ValueMax);

            if(TextObject.FontArray == null) {
              for(Int32 i = 0; i < TextObject.LetterLength; i++) {
                UInt16 LetterValue = TextObject.LetterList[0][ValueArray[i]];
                String Letter = String.Empty;

                switch(LetterValue) {
                  case 0x0D: {
                    Letter = ": 0x0D I";
                    break;
                  }
                  case 0x0A: {
                    Letter = ": 0x0A I";
                    break;
                  }
                  default: {
                    Letter = ": '" + (Char) LetterValue + "' I";
                    break;
                  }
                }

                TextArray[i] = i + Letter + ValueArray[i] + " F0";
              }
            } else {
              Byte[] FontArray = this.MakeDecompressMemory(TextObject.FontArray, TextObject.LetterLength, TextObject.FontCount);

              for(Int32 i = 0; i < TextObject.LetterLength; i++) {
                UInt16 LetterValue = TextObject.LetterList[FontArray[i]][ValueArray[i]];
                String Letter = String.Empty;

                switch(LetterValue) {
                  case 0x0D: {
                    Letter = ": 0x0D I";
                    break;
                  }
                  case 0x0A: {
                    Letter = ": 0x0A I";
                    break;
                  }
                  default: {
                    Letter = ": '" + (Char) LetterValue + "' I";
                    break;
                  }
                }

                TextArray[i] = i + Letter + ValueArray[i] + " F" + FontArray[i];
              }
            }
            break;
          }
        }

        this.MyTextViewer.Run(this, TextObject.Title, null, TextArray);
      }
    }

    private void ShowPicturePaintProject(FileViewerTDPicturePaintProjectObject ProjectObject) {
      String Title = null;
      ColumnHeader[] Columns = null;
      ListViewItem[] Items = null;
      Bitmap[] Bitmaps = null;

      switch(ProjectObject.Type) {
        case FileViewerTDPicturePaintProjectType.AlphaArray: {
          Byte[] AlphaArray = (Byte[]) ProjectObject.Value;

          String[] ColumnTexts = new String[] { "Index", "Value", "".PadRight(20), "Alpha", "" };

          Columns = new ColumnHeader[ColumnTexts.Length];

          for(Int32 i = 0; i < ColumnTexts.Length; i++) {
            Columns[i] = new ColumnHeader();
            Columns[i].Text = ColumnTexts[i];
          }

          for(Int32 i = 3; i < ColumnTexts.Length; i++)
            Columns[i].TextAlign = HorizontalAlignment.Right;

          Items = new ListViewItem[AlphaArray.Length];

          for(Int32 i = 0; i < AlphaArray.Length; i++) {
            Items[i] = new ListViewItem(new String[] { i.ToString(), "0x" + AlphaArray[i].ToString("X2"), "", AlphaArray[i].ToString(), "" });
            Items[i].UseItemStyleForSubItems = false;
            Items[i].SubItems[2].BackColor = Color.FromArgb(AlphaArray[i], AlphaArray[i], AlphaArray[i]);
          }

          Title = "AlphaArray (" + AlphaArray.Length + ")";
          break;
        }
        case FileViewerTDPicturePaintProjectType.AlphaPalette: {
          FileViewerTDPicturePaintProjectPalette[] AlphaPalette = (FileViewerTDPicturePaintProjectPalette[]) ProjectObject.Value;

          String[] ColumnTexts = new String[] { "Index", "Select", "Value", "".PadRight(20), "Alpha", "" };

          Columns = new ColumnHeader[ColumnTexts.Length];

          for(Int32 i = 0; i < ColumnTexts.Length; i++) {
            Columns[i] = new ColumnHeader();
            Columns[i].Text = ColumnTexts[i];
          }

          Columns[1].TextAlign = HorizontalAlignment.Center;

          for(Int32 i = 4; i < ColumnTexts.Length; i++)
            Columns[i].TextAlign = HorizontalAlignment.Right;

          Items = new ListViewItem[AlphaPalette.Length];

          for(Int32 i = 0; i < AlphaPalette.Length; i++) {
            Byte Alpha = (Byte) AlphaPalette[i].Value;

            Items[i] = new ListViewItem(new String[] { i.ToString(), AlphaPalette[i].Select.ToString(), "0x" + Alpha.ToString("X2"), "", Alpha.ToString(), "" });
            Items[i].UseItemStyleForSubItems = false;
            Items[i].SubItems[3].BackColor = Color.FromArgb(Alpha, Alpha, Alpha);
          }

          Title = "AlphaPalette (" + AlphaPalette.Length + ")";
          break;
        }
        case FileViewerTDPicturePaintProjectType.AlphaGradient: {
          FileViewerTDPicturePaintProjectGradient[] AlphaGradient = (FileViewerTDPicturePaintProjectGradient[]) ProjectObject.Value;

          String[] ColumnTexts = new String[] { "", "Index", "Type", "Begin", "Between", "Middle", "End", "Factor", "FactorBetween", "Length", "" };

          Columns = new ColumnHeader[ColumnTexts.Length];

          for(Int32 i = 0; i < ColumnTexts.Length; i++) {
            Columns[i] = new ColumnHeader();
            Columns[i].Text = ColumnTexts[i];
          }

          Columns[2].TextAlign = HorizontalAlignment.Center;

          Bitmaps = new Bitmap[AlphaGradient.Length];

          Items = new ListViewItem[AlphaGradient.Length];

          for(Int32 i = 0; i < AlphaGradient.Length; i++) {
            switch(AlphaGradient[i].Type) {
              case 0: { //Linear             
                Items[i] = new ListViewItem(new String[] { "", i.ToString(), AlphaGradient[i].Type.ToString(), "0x" + AlphaGradient[i].Begin.ToString("X2"), "", "", "0x" + AlphaGradient[i].End.ToString("X2"), "", "", AlphaGradient[i].Length.ToString(), "" });
                break;
              }
              case 1: { //Cosinus        
                Items[i] = new ListViewItem(new String[] { "", i.ToString(), AlphaGradient[i].Type.ToString(), "0x" + AlphaGradient[i].Begin.ToString("X2"), "", "", "0x" + AlphaGradient[i].End.ToString("X2"), AlphaGradient[i].Factor.ToString(), "", AlphaGradient[i].Length.ToString(), "" });
                break;
              }
              case 2: { //Sinus      
                Items[i] = new ListViewItem(new String[] { "", i.ToString(), AlphaGradient[i].Type.ToString(), "0x" + AlphaGradient[i].Begin.ToString("X2"), "", "", "0x" + AlphaGradient[i].End.ToString("X2"), AlphaGradient[i].Factor.ToString(), "", AlphaGradient[i].Length.ToString(), "" });
                break;
              }
              case 3: { //Round     
                Items[i] = new ListViewItem(new String[] { "", i.ToString(), AlphaGradient[i].Type.ToString(), "0x" + AlphaGradient[i].Begin.ToString("X2"), "", "0x" + AlphaGradient[i].Middle.ToString("X2"), "0x" + AlphaGradient[i].End.ToString("X2"), AlphaGradient[i].Factor.ToString(), "", AlphaGradient[i].Length.ToString(), "" });
                break;
              }
              case 4: { //RoundExtra    
                Items[i] = new ListViewItem(new String[] { "", i.ToString(), AlphaGradient[i].Type.ToString(), "0x" + AlphaGradient[i].Begin.ToString("X2"), "0x" + AlphaGradient[i].Between.ToString("X2"), "0x" + AlphaGradient[i].Middle.ToString("X2"), "0x" + AlphaGradient[i].End.ToString("X2"), AlphaGradient[i].Factor.ToString(), AlphaGradient[i].FactorBetween.ToString(), AlphaGradient[i].Length.ToString(), "" });
                break;
              }
              case 5: { //Mixer

                break;
              }
            }

            Items[i].ImageIndex = i;

            Bitmaps[i] = AlphaGradient[i].CreateAlphaBitmap();
          }

          Title = "AlphaGradient (" + AlphaGradient.Length + ")";
          break;
        }
        case FileViewerTDPicturePaintProjectType.ColorArray: {
          UInt32[] ColorArray = (UInt32[]) ProjectObject.Value;

          String[] ColumnTexts = new String[] { "Index", "Value", "".PadRight(4), "".PadRight(20), "Alpha", "   Red", "Green", "   Blue", "" };

          Columns = new ColumnHeader[ColumnTexts.Length];

          for(Int32 i = 0; i < ColumnTexts.Length; i++) {
            Columns[i] = new ColumnHeader();
            Columns[i].Text = ColumnTexts[i];
          }

          for(Int32 i = 4; i < ColumnTexts.Length; i++)
            Columns[i].TextAlign = HorizontalAlignment.Right;

          Items = new ListViewItem[ColorArray.Length];

          for(Int32 i = 0; i < ColorArray.Length; i++) {
            Color Color = Color.FromArgb((Int32) ColorArray[i]);

            Items[i] = new ListViewItem(new String[] { i.ToString(), "0x" + ColorArray[i].ToString("X8"), "", "", Color.A.ToString(), Color.R.ToString(), Color.G.ToString(), Color.B.ToString(), "" });
            Items[i].UseItemStyleForSubItems = false;
            Items[i].SubItems[2].BackColor = Color.FromArgb(Color.A, Color.A, Color.A);
            Items[i].SubItems[3].BackColor = Color.FromArgb((Int32) (0xFF000000 | ColorArray[i]));
          }

          Title = "ColorArray (" + ColorArray.Length + ")";
          break;
        }
        case FileViewerTDPicturePaintProjectType.ColorPalette: {
          FileViewerTDPicturePaintProjectPalette[] ColorPalette = (FileViewerTDPicturePaintProjectPalette[]) ProjectObject.Value;

          String[] ColumnTexts = new String[] { "Index", "Select", "Value", "".PadRight(4), "".PadRight(20), "Alpha", "   Red", "Green", "   Blue", "" };

          Columns = new ColumnHeader[ColumnTexts.Length];

          for(Int32 i = 0; i < ColumnTexts.Length; i++) {
            Columns[i] = new ColumnHeader();
            Columns[i].Text = ColumnTexts[i];
          }

          Columns[1].TextAlign = HorizontalAlignment.Center;

          for(Int32 i = 5; i < ColumnTexts.Length; i++)
            Columns[i].TextAlign = HorizontalAlignment.Right;

          Items = new ListViewItem[ColorPalette.Length];

          for(Int32 i = 0; i < ColorPalette.Length; i++) {
            Color Color = Color.FromArgb((Int32) ColorPalette[i].Value);

            Items[i] = new ListViewItem(new String[] { i.ToString(), ColorPalette[i].Select.ToString(), "0x" + ColorPalette[i].Value.ToString("X8"), "", "", Color.A.ToString(), Color.R.ToString(), Color.G.ToString(), Color.B.ToString(), "" });
            Items[i].UseItemStyleForSubItems = false;
            Items[i].SubItems[3].BackColor = Color.FromArgb(Color.A, Color.A, Color.A);
            Items[i].SubItems[4].BackColor = Color.FromArgb((Int32) (0xFF000000 | ColorPalette[i].Value));
          }

          Title = "ColorPalette (" + ColorPalette.Length + ")";
          break;
        }
        case FileViewerTDPicturePaintProjectType.ColorGradient: {
          FileViewerTDPicturePaintProjectGradient[] ColorGradient = (FileViewerTDPicturePaintProjectGradient[]) ProjectObject.Value;

          String[] ColumnTexts = new String[] { "", "Index", "Type", "Begin", "Between", "Middle", "End", "Factor", "FactorBetween", "Length", "" };

          Columns = new ColumnHeader[ColumnTexts.Length];

          for(Int32 i = 0; i < ColumnTexts.Length; i++) {
            Columns[i] = new ColumnHeader();
            Columns[i].Text = ColumnTexts[i];
          }

          Columns[2].TextAlign = HorizontalAlignment.Center;

          Bitmaps = new Bitmap[ColorGradient.Length];

          Items = new ListViewItem[ColorGradient.Length];

          for(Int32 i = 0; i < ColorGradient.Length; i++) {
            switch(ColorGradient[i].Type) {
              case 0: { //Linear             
                Items[i] = new ListViewItem(new String[] { "", i.ToString(), ColorGradient[i].Type.ToString(), "0x" + ColorGradient[i].Begin.ToString("X8"), "", "", "0x" + ColorGradient[i].End.ToString("X8"), "", "", ColorGradient[i].Length.ToString(), "" });
                break;
              }
              case 1: { //Cosinus        
                Items[i] = new ListViewItem(new String[] { "", i.ToString(), ColorGradient[i].Type.ToString(), "0x" + ColorGradient[i].Begin.ToString("X8"), "", "", "0x" + ColorGradient[i].End.ToString("X8"), ColorGradient[i].Factor.ToString(), "", ColorGradient[i].Length.ToString(), "" });
                break;
              }
              case 2: { //Sinus      
                Items[i] = new ListViewItem(new String[] { "", i.ToString(), ColorGradient[i].Type.ToString(), "0x" + ColorGradient[i].Begin.ToString("X8"), "", "", "0x" + ColorGradient[i].End.ToString("X8"), ColorGradient[i].Factor.ToString(), "", ColorGradient[i].Length.ToString(), "" });
                break;
              }
              case 3: { //Round     
                Items[i] = new ListViewItem(new String[] { "", i.ToString(), ColorGradient[i].Type.ToString(), "0x" + ColorGradient[i].Begin.ToString("X8"), "", "0x" + ColorGradient[i].Middle.ToString("X8"), "0x" + ColorGradient[i].End.ToString("X8"), ColorGradient[i].Factor.ToString(), "", ColorGradient[i].Length.ToString(), "" });
                break;
              }
              case 4: { //RoundExtra    
                Items[i] = new ListViewItem(new String[] { "", i.ToString(), ColorGradient[i].Type.ToString(), "0x" + ColorGradient[i].Begin.ToString("X8"), "0x" + ColorGradient[i].Between.ToString("X8"), "0x" + ColorGradient[i].Middle.ToString("X8"), "0x" + ColorGradient[i].End.ToString("X8"), ColorGradient[i].Factor.ToString(), ColorGradient[i].FactorBetween.ToString(), ColorGradient[i].Length.ToString(), "" });
                break;
              }
              case 5: { //Mixer
                
                break;
              }
            }

            Items[i].ImageIndex = i;

            Bitmaps[i] = ColorGradient[i].CreateColorBitmap();
          }

          Title = "ColorGradient (" + ColorGradient.Length + ")";
          break;
        }
        case FileViewerTDPicturePaintProjectType.GroupItem: {
          Int32[] ItemArray = (Int32[]) ProjectObject.Value;

          String[] ColumnTexts = new String[] { "Index", "PosX", "PosX", "Width", "Height", "Image", "" };

          Columns = new ColumnHeader[ColumnTexts.Length];

          for(Int32 i = 0; i < ColumnTexts.Length; i++) {
            Columns[i] = new ColumnHeader();
            Columns[i].Text = ColumnTexts[i];
          }

          Int32 ItemCount = ItemArray.Length / 5;
          Int32 Index = 0;

          Items = new ListViewItem[ItemCount];

          for(Int32 i = 0; i < ItemCount; i++) {
            Items[i] = new ListViewItem(new String[] { i.ToString(), ItemArray[Index].ToString(), ItemArray[Index + 1].ToString(), ItemArray[Index + 2].ToString(), ItemArray[Index + 3].ToString(), ItemArray[Index + 4].ToString() });
            Index += 5;
          }

          Title = "GroupItems (" + ItemCount + ")";
          break;
        }
      }

      if(Title == null || Columns == null || Items == null) return;

      this.MyColumnViewer.Run(this, Title, Columns, Items, Bitmaps);
    }

    private void ShowHelpWriter(FileViewerTDHelpWriterObject WriterObject) {
      switch(WriterObject.Type) {
        case FileViewerTDHelpWriterType.OutlineLetterIndex: {
          Int32[] IndexArray = (Int32[]) WriterObject.Value;

          String[] TextArray = new String[IndexArray.Length];

          for(Int32 i = 0; i < IndexArray.Length; i++)
            TextArray[i] = i + " = " + IndexArray[i];

          this.MyTextViewer.Run(this, "Outline letter index", null, TextArray);
          return;
        }
        case FileViewerTDHelpWriterType.OutlineInfoType: {
          FileViewerTDCompressTextObject TextObject = (FileViewerTDCompressTextObject) WriterObject.Value;

          Byte[] ValueArray = this.MakeDecompressMemory(TextObject.ValueArray, TextObject.LetterLength, 4);

          String[] TextArray = new String[TextObject.LetterLength];

          for(Int32 i = 0; i < TextObject.LetterLength; i++) {
            String Info = String.Empty;

            switch(ValueArray[i] & 0x03) {
              case 0: { //Letter
                Info = "L";
                break;
              }
              case 1: { //Seperator
                Info = "L,SE";
                break;
              }
              case 2: { //Space
                Info = "SP";
                break;
              }
              case 3: { //Control
                Info = "CT";
                break;
              }
            }

            TextArray[i] = i + ": " + ValueArray[i] + " " + Info;
          }

          this.MyTextViewer.Run(this, "Outline info type", null, TextArray);
          return;
        }
        case FileViewerTDHelpWriterType.OutlineInfoIndex: {
          FileViewerTDCompressTextObject TextObject = (FileViewerTDCompressTextObject) WriterObject.Value;

          if(TextObject.ValueMax > 256) {
            MessageBox.Show("Indexes for more than 256 different characters can not be displayed with this program.", "Information", MessageBoxButtons.OK, MessageBoxIcon.Information);
            return;
          }

          Byte[] ValueArray = this.MakeDecompressMemory(TextObject.ValueArray, TextObject.LetterLength, TextObject.ValueMax);

          String[] TextArray = new String[ValueArray.Length];

          for(Int32 i = 0; i < ValueArray.Length; i++)
            TextArray[i] = i + " = " + ValueArray[i];

          this.MyTextViewer.Run(this, "Outline letter info index", null, TextArray);
          return;
        }
        case FileViewerTDHelpWriterType.FieldTextColor: {
          UInt32[] ColorArray = (UInt32[]) WriterObject.Value;

          String[] TextArray = new String[ColorArray.Length];

          for(Int32 i = 0; i < ColorArray.Length; i++)
            TextArray[i] = i + " = 0x" + ColorArray[i].ToString("X8");

          this.MyTextViewer.Run(this, "Text color", null, TextArray);
          return;
        }
        case FileViewerTDHelpWriterType.FieldTextFont: {
          Int32[] IndexArray = (Int32[]) WriterObject.Value;

          String[] TextArray = new String[IndexArray.Length];

          for(Int32 i = 0; i < IndexArray.Length; i++)
            TextArray[i] = i + " = " + IndexArray[i];

          this.MyTextViewer.Run(this, "Text font index", null, TextArray);
          return;
        }
        case FileViewerTDHelpWriterType.FieldTextLetter: {
          Int32[] IndexArray = (Int32[]) WriterObject.Value;

          String[] TextArray = new String[IndexArray.Length];

          for(Int32 i = 0; i < IndexArray.Length; i++)
            TextArray[i] = i + " = " + IndexArray[i];

          this.MyTextViewer.Run(this, "Text letter index", null, TextArray);
          return;
        }
        case FileViewerTDHelpWriterType.FieldTextInfoType: {
          FileViewerTDCompressTextObject TextObject = (FileViewerTDCompressTextObject) WriterObject.Value;

          Byte[] ValueArray = this.MakeDecompressMemory(TextObject.ValueArray, TextObject.LetterLength, TextObject.ValueMax);

          String[] TextArray = new String[TextObject.LetterLength];

          for(Int32 i = 0; i < TextObject.LetterLength; i++) {
            String Info = String.Empty;

            switch(ValueArray[i] & 0x03) {
              case 0: { //Letter
                Info = "L";
                break;
              }
              case 1: { //Seperator
                Info = "L,SE";
                break;
              }
              case 2: { //Space
                Info = "SP";
                break;
              }
              case 3: { //Control
                Info = "CT";
                break;
              }
            }

            if((ValueArray[i] & 0x04) == 0x04)
              Info += ",NL";

            TextArray[i] = i + ": " + ValueArray[i] + " " + Info;
          }

          this.MyTextViewer.Run(this, "Text info type", null, TextArray);
          return;
        }
        case FileViewerTDHelpWriterType.FieldTextInfoFont: {
          FileViewerTDCompressTextObject TextObject = (FileViewerTDCompressTextObject) WriterObject.Value;

          Byte[] ValueArray = this.MakeDecompressMemory(TextObject.ValueArray, TextObject.LetterLength, TextObject.ValueMax);

          String[] TextArray = new String[TextObject.LetterLength];

          for(Int32 i = 0; i < TextObject.LetterLength; i++)
            TextArray[i] = i + ": F" + ValueArray[i];

          this.MyTextViewer.Run(this, "Text info font", null, TextArray);
          return;
        }
        case FileViewerTDHelpWriterType.FieldTextInfoColor: {
          FileViewerTDCompressTextObject TextObject = (FileViewerTDCompressTextObject) WriterObject.Value;

          Byte[] ValueArray = this.MakeDecompressMemory(TextObject.ValueArray, TextObject.LetterLength, TextObject.ValueMax);

          String[] TextArray = new String[TextObject.LetterLength];

          for(Int32 i = 0; i < TextObject.LetterLength; i++)
            TextArray[i] = i + ": C" + ValueArray[i];

          this.MyTextViewer.Run(this, "Text info color", null, TextArray);
          return;
        }
        case FileViewerTDHelpWriterType.FieldTextLetterIndex: {
          FileViewerTDCompressTextObject TextObject = (FileViewerTDCompressTextObject) WriterObject.Value;

          Byte[] ValueArray = this.MakeDecompressMemory(TextObject.ValueArray, TextObject.LetterLength, TextObject.ValueMax);

          String[] TextArray = new String[TextObject.LetterLength];

          for(Int32 i = 0; i < TextObject.LetterLength; i++)
            TextArray[i] = i + ": I" + ValueArray[i];

          this.MyTextViewer.Run(this, "Text letter index", null, TextArray);
          return;
        }
      }
    }

    #endregion


    #region Private Events

    private void ListViewFormat_ItemSelectionChanged(object sender, ListViewItemSelectionChangedEventArgs e) {
      if(e.Item == null || e.Item.Tag == null || !e.IsSelected) {
        this.ButtonShow.Enabled = false;

        this.MyViewerObject = null;
        return;
      }

      this.ButtonShow.Enabled = true;

      this.MyViewerObject = (FileViewerObject) e.Item.Tag;
    }


    private void FileViewerWindow_FormClosing(object sender, FormClosingEventArgs e) {
      this.MyRun = false;
    }

    private void FileViewerWindow_HelpButtonClicked(object sender, CancelEventArgs e) {
      try {
        Process.Start(Application.StartupPath + "\\FileViewer.pdf");
      } catch(Exception ex) {
        MessageBox.Show(ex.Message, "Can not open: FileViewer.pdf", MessageBoxButtons.OK, MessageBoxIcon.Error); 
      }
    }


    private void ButtonOpen_Click(object sender, EventArgs e) {
      switch(this.MyOpenFile.ShowDialog(this)) {
        case DialogResult.OK: {
          this.MyFilePath = this.MyOpenFile.FileName;

          this.TextBoxFile.Text = this.MyFilePath;

          this.UpdateComboBoxFormat(this.MyFilePath);

          this.ButtonShow.Enabled = false;
          this.ButtonRun.Enabled = true;

          this.ProgressBarPosition.Value = 0;

          this.ListViewFormat.Items.Clear();
          this.ListViewFormat.Refresh();

          this.MyOpenFile.InitialDirectory = Path.GetDirectoryName(this.MyFilePath);
          this.MyOpenFile.FileName = Path.GetFileName(this.MyFilePath);

          this.MyViewerObject = null;
          break;
        }
        case DialogResult.Cancel: {
          break;
        }
      }
    }

    private void ButtonShow_Click(object sender, EventArgs e) {
      switch(this.MyViewerObject.Type) {
        case FileViewerObjectType.Image: {
          this.MyImageViewer.Run(this, (Image) this.MyViewerObject.Value);
          break;
        }
        case FileViewerObjectType.Animation: {
          this.ShowAnimationItems((MemoryStream) this.MyViewerObject.Value);
          break;
        }
        case FileViewerObjectType.Sound: {
          this.ShowSaveRunFile("Sound.wav", (Byte[]) this.MyViewerObject.Value);
          break;
        }
        case FileViewerObjectType.ProtectedData: {
          this.ShowProtectedData((FileViewerProtectedDataObject) this.MyViewerObject.Value);
          break;
        }
        case FileViewerObjectType.ProtectedFile: {
          this.ShowProtectedFile((FileViewerProtectedFileObject) this.MyViewerObject.Value);
          break;
        }
        case FileViewerObjectType.ProtectedMemory: {
          this.MyProtectedViewer.Run(this, (FileViewerProtectedData) this.MyViewerObject.Value);
          break;
        }
        case FileViewerObjectType.FormatText: {
          this.ShowFormatText((FileViewerTDFormatTextObject) this.MyViewerObject.Value);
          break;
        }
        case FileViewerObjectType.CompressText: {
          this.ShowCompressText((FileViewerTDCompressTextObject) this.MyViewerObject.Value);
          break;
        }
        case FileViewerObjectType.PicturePaintFile: {
          FileViewerTDPicturePaintFileObject FileObject = (FileViewerTDPicturePaintFileObject) this.MyViewerObject.Value;

          this.ShowSaveRunFile(FileObject.FileName, FileObject.FileMemory);
          break;
        }
        case FileViewerObjectType.PicturePaintProject: {
          this.ShowPicturePaintProject((FileViewerTDPicturePaintProjectObject) this.MyViewerObject.Value);
          break;
        }
        case FileViewerObjectType.HelpWriter: {
          this.ShowHelpWriter((FileViewerTDHelpWriterObject) this.MyViewerObject.Value);
          break;
        }
      }
    }

    private void ButtonRun_Click(object sender, EventArgs e) {
      if(this.MyRun) {
        this.MyRun = false;

        this.ComboBoxFormat.Focus();

        this.ButtonRun.Enabled = false;

        Application.DoEvents();
        return;
      }

      this.ComboBoxFormat.Focus();

      this.MyRun = true;

      this.ButtonRun.Text = "Stop";
      this.ButtonOpen.Enabled = false;
      this.ComboBoxFormat.Enabled = false;

      this.ProgressBarPosition.Value = 0;

      this.ListViewFormat.Items.Clear();
      this.ListViewFormat.Refresh();

      this.ButtonShow.Enabled = false;

      Application.DoEvents();

      this.MyViewerObject = null;

      List<ListViewItem> ItemList = null;

      FileFormat Index = (FileFormat) this.ComboBoxFormat.SelectedIndex;

      switch(Index) {
        case FileFormat.Picture: { //2D Picture
          FileViewerTDPicture Item = new FileViewerTDPicture();

          ItemList = Item.Run(this, this.MyFilePath);
          break;
        }
        case FileFormat.Animation: { //2D Animation
          FileViewerTDAnimation Item = new FileViewerTDAnimation();

          ItemList = Item.Run(this, this.MyFilePath);
          break;
        }
        case FileFormat.HelpReader: { //2D Help Reader
          FileViewerTDHelpReader Item = new FileViewerTDHelpReader();

          ItemList = Item.Run(this, this.MyFilePath);
          break;
        }
        case FileFormat.HelpWriter: { //2D Help Writer
          FileViewerTDHelpWriter Item = new FileViewerTDHelpWriter();

          ItemList = Item.Run(this, this.MyFilePath);
          break;
        }
        case FileFormat.CompressText: { //2D Compress Text
          FileViewerTDCompressText Item = new FileViewerTDCompressText();

          ItemList = Item.Run(this, this.MyFilePath);
          break;
        }
        case FileFormat.FormatText: { //2D Format Text
          FileViewerTDFormatText Item = new FileViewerTDFormatText();
          
          ItemList = Item.Run(this, this.MyFilePath);
          break;
        }
        case FileFormat.PicturePaintCollection: { //Picture Paint Collection
          FileViewerTDPicturePaintCollection Item = new FileViewerTDPicturePaintCollection();

          ItemList = Item.Run(this, this.MyFilePath);
          break;
        }
        case FileFormat.PicturePaintPoject: { //Picture Paint Poject
          FileViewerTDPicturePaintProject Item = new FileViewerTDPicturePaintProject();

          ItemList = Item.Run(this, this.MyFilePath);
          break;
        }
        case FileFormat.ProtectedData: { //Protected Data
          FileViewerProtectedData Item = new FileViewerProtectedData();

          UInt16[] PassLetterArray = this.MyPassViewer.Run(this);

          ItemList = Item.Run(this, this.MyFilePath, PassLetterArray);
          break;
        }
        case FileFormat.VideoThumbnailImageData: { //Video Thumbnail Image Data 
          FileViewerVideoThumbnailImageData Item = new FileViewerVideoThumbnailImageData();

          ItemList = Item.Run(this, this.MyFilePath);
          break;
        }
      }

      if(this.IsDisposed) return;

      if(ItemList != null) {
        this.ListViewFormat.BeginUpdate();
        this.ListViewFormat.Items.AddRange(ItemList.ToArray());

        for(Int32 i = 0; i < 5; i++)
          this.ListViewFormat.Columns[i].AutoResize(ColumnHeaderAutoResizeStyle.ColumnContent);

        this.ListViewFormat.EndUpdate();
      }

      this.ButtonOpen.Enabled = true;
      this.ComboBoxFormat.Enabled = true;

      this.MyRun = false;

      this.ButtonRun.Text = "Run";
      this.ButtonRun.Enabled = true;
    }

    #endregion


    #region Public Methods: IFileViewerProgressBar

    public void BeginProgressBar(Int64 FileSize) {
      this.MyFileSize = FileSize;
    }
    
    public bool UpdateProgressBar(Int64 Position) {
      this.ProgressBarPosition.Value = (Int32) ((Position * 1000) / this.MyFileSize);

      Application.DoEvents();

      if(!this.MyRun) return false;

      return true;
    }
    
    #endregion    
  }
}
