﻿//FileViewerFormats
//PanotiSoft 
//Markus Zerhusen
//20.09.2017

using System;
using System.IO;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.Windows.Forms;
using System.Runtime.InteropServices;

namespace FileViewer {

  public interface IFileViewerProgressBar {
    void BeginProgressBar(Int64 Size);
    bool UpdateProgressBar(Int64 Position);
  }


  public enum FileViewerObjectType {
    Image = 0,
    Animation = 1,
    Sound = 2,
    ProtectedData = 3,
    ProtectedFile = 4,
    ProtectedMemory = 5,
    FormatText = 6,
    CompressText = 7,
    PicturePaintFile = 8,
    PicturePaintProject = 9,
    HelpWriter = 10,
  }

  public class FileViewerObject {
    #region Public Fields

    public FileViewerObjectType Type;
    public Object Value;

    #endregion

    #region Contructor

    public FileViewerObject(Image Image) {
      this.Type = FileViewerObjectType.Image;
      this.Value = Image;
    }

    public FileViewerObject(FileViewerObjectType Type, Object Value) {
      this.Type = Type;
      this.Value = Value;
    }

    #endregion
  }


  public class FileViewerTDPicture {
    #region Public Fields

    public IFileViewerProgressBar Parent;
    public List<ListViewItem> ItemList;
    public BinaryReader Reader;
    public FileStream Stream;
    public String FilePath;
    public Int64 Position;
    public Int64 Size;

    #endregion


    #region Contructor

    public FileViewerTDPicture() {
      this.ItemList = new List<ListViewItem>();
    }

    #endregion


    #region Private Methods

    private void AddItem(Int64 Size, String Type, String Name, String Value) {
      this.ItemList.Add(new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value }));

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }

    private void AddItem(Int64 Size, String Type, String Name, String Value, FileViewerObject Tag) {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value });

      Item.BackColor = Color.FromArgb(240, 255, 245);
      Item.Tag = Tag;

      this.ItemList.Add(Item);

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }


    private void AddItemError(String Value) {
      ListViewItem Item = new ListViewItem(new String[] { "".PadRight(10), "".PadRight(10), "".PadRight(10), "Error".PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }

    private void AddItemError(Exception e) {
      String Name = String.Empty;
      String Value = String.Empty;

      if(e.InnerException != null && e.InnerException.Message != null) {
        if(e.Message != null)
          Name = e.Message;

        Value = e.InnerException.Message;
      } else {
        if(e.Message != null)
          Value = e.Message;
      }

      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "ERROR".PadRight(10), Name.PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }


    private void AddItemBeginRead() {
      ListViewItem Item = new ListViewItem(new String[] { "0".PadRight(10), "0".PadRight(10), "".PadRight(10), "BeginRead".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }

    private void AddItemEndOfFile() {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "".PadRight(10), "EndOfFile".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }


    private void Close() {
      if(this.Stream != null) {
        this.Stream.Close();
        this.Stream = null;
      }

      if(this.Reader != null) {
        this.Reader.Close();
        this.Reader = null;
      }
    }

    #endregion

    #region Private Methods: Format

    private UInt32 ReadIDNumber() {
      UInt32 IDNumber = this.Reader.ReadUInt32();

      if(IDNumber != 0x49504454)
        throw new Exception("IDNumber", new Exception("0x" + IDNumber.ToString("X8")));

      this.AddItem(4, "UINT32", "IDNumber", "0x" + IDNumber.ToString("X8") + " hex");
      return IDNumber;
    }

    private Byte ReadVersion() {
      Byte Version = this.Reader.ReadByte();

      if(Version != 1)
        throw new Exception("Version", new Exception(Version.ToString()));

      this.AddItem(1, "BYTE", "Version", Version.ToString());
      return Version;
    }

    private Int64 ReadFileSize() {
      Int64 FileSize = this.Reader.ReadInt64();

      if(FileSize != this.Size)
        throw new Exception("FileSize", new Exception(FileSize.ToString()));

      this.AddItem(8, "INT64", "FileSize", FileSize.ToString("#,#") + " bytes");
      return FileSize;
    }

    private Int32 ReadThumbnailSize() {
      Int32 ThumbnailSize = this.Reader.ReadInt32();

      if(ThumbnailSize < 0 || ThumbnailSize > this.Size - this.Position - 4)
        throw new Exception("ThumbnailSize", new Exception(ThumbnailSize.ToString()));

      if(ThumbnailSize == 0)
        this.AddItem(4, "INT32", "ThumbnailSize", ThumbnailSize + " byte");
      else
        this.AddItem(4, "INT32", "ThumbnailSize", ThumbnailSize.ToString("#,#") + " bytes");
      return ThumbnailSize;
    }

    private void ReadThumbnailImage(Int32 ThumbnailSize) {
      try {
        Byte[] ThumbnailImageArray = this.Reader.ReadBytes(ThumbnailSize);

        Bitmap ThumbnailImage = ThumbnailImage = new Bitmap(new MemoryStream(ThumbnailImageArray));

        this.AddItem(ThumbnailSize, "MEMORY", "ThumbnailImage", "", new FileViewerObject(ThumbnailImage));
      } catch(Exception e) {
        throw new Exception("ThumbnailImage", e);
      }
    }

    private Int32 ReadImageWidth() {
      Int32 ImageWidth = this.Reader.ReadInt32();

      if(ImageWidth <= 0)
        throw new Exception("ImageWidth", new Exception(ImageWidth.ToString()));

      this.AddItem(4, "INT32", "ImageWidth", ImageWidth.ToString("#,#") + " pixel");
      return ImageWidth;
    }

    private Int32 ReadImageHeight() {
      Int32 ImageHeight = this.Reader.ReadInt32();

      if(ImageHeight <= 0)
        throw new Exception("ImageHeight", new Exception(ImageHeight.ToString()));

      this.AddItem(4, "INT32", "ImageHeight", ImageHeight.ToString("#,#") + " pixel");
      return ImageHeight;
    }

    private void ReadImageColors(Int32 ImageWidth, Int32 ImageHeight, Int64 ImageSize) {
      if(ImageSize > 16000L * 16000L || ImageWidth > 16000 || ImageHeight > 16000) {
        this.AddItem(ImageSize, "MEMORY", "ImageColors", "");
        return;
      }

      try {
        Bitmap Image = new Bitmap(ImageWidth, ImageHeight, PixelFormat.Format32bppArgb);

        BitmapData Data = Image.LockBits(new Rectangle(0, 0, ImageWidth, ImageHeight), ImageLockMode.ReadWrite, PixelFormat.Format32bppArgb);

        for(Int32 i = 0; i < ImageHeight; i++) {
          Byte[] ImageArray = this.Reader.ReadBytes(ImageWidth * 4);

          if(!this.Parent.UpdateProgressBar(this.Stream.Position)) throw new Exception("Reading was canceled.");

          System.Runtime.InteropServices.Marshal.Copy(ImageArray, 0, new IntPtr(Data.Scan0.ToInt64() + i * ImageWidth * 4L), ImageWidth * 4);
        }

        Image.UnlockBits(Data);

        this.AddItem(ImageSize, "MEMORY", "ImageColors", "", new FileViewerObject(Image));
      } catch(Exception e) {
        throw new Exception("ImageColors", e);
      }
    }


    private void ReadBegin() {
      UInt32 IDNumber = this.ReadIDNumber();

      Byte Version = this.ReadVersion();

      Int64 FileSize = this.ReadFileSize();

      Int32 ThumbnailSize = this.ReadThumbnailSize();

      if(ThumbnailSize > 0) this.ReadThumbnailImage(ThumbnailSize);

      Int32 ImageWidth = this.ReadImageWidth();
      Int32 ImageHeight = this.ReadImageHeight();

      Int64 ImageSize = 4L * ImageWidth * ImageHeight;

      if(ImageSize != this.Size - this.Position) throw new Exception("Wrong ImageSize");

      this.ReadImageColors(ImageWidth, ImageHeight, ImageSize);
    }

    #endregion


    #region Public Methods

    public List<ListViewItem> Run(IFileViewerProgressBar Parent, String FilePath) {
      this.Parent = Parent;

      try {
        this.Stream = new FileStream(FilePath, FileMode.Open, FileAccess.Read);
        this.Reader = new BinaryReader(this.Stream);

        this.Size = this.Stream.Length;

        this.Parent.BeginProgressBar(this.Size);

        this.AddItemBeginRead();

        this.ReadBegin();
      } catch(Exception e) {
        this.AddItemError(e);

        this.Close();
        return this.ItemList;
      }

      this.AddItemEndOfFile();

      this.Close();
      return this.ItemList;
    }

    #endregion

    #region Public Static Methods

    public static Bitmap LoadImage(MemoryStream Stream) {
      try {
        BinaryReader Reader = new BinaryReader(Stream);

        UInt32 IDNumber = Reader.ReadUInt32();

        if(IDNumber != 0x49504454) return null;

        Byte Version = Reader.ReadByte();

        if(Version != 1) return null;

        Int64 FileSize = Reader.ReadInt64();

        if(FileSize != Stream.Length) return null;

        Int32 ThumbnailSize = Reader.ReadInt32();

        if(ThumbnailSize < 0 || ThumbnailSize > Stream.Length - Stream.Position) return null;
        if(ThumbnailSize > 0) Stream.Position += ThumbnailSize;

        Int32 ImageWidth = Reader.ReadInt32();

        if(ImageWidth <= 0 || ImageWidth > 32000) return null;

        Int32 ImageHeight = Reader.ReadInt32();

        if(ImageHeight <= 0 || ImageHeight > 32000) return null;

        Int64 ImageSize = 4L * ImageWidth * ImageHeight;

        Bitmap Image = new Bitmap(ImageWidth, ImageHeight, PixelFormat.Format32bppArgb);

        BitmapData Data = Image.LockBits(new Rectangle(0, 0, ImageWidth, ImageHeight), ImageLockMode.ReadWrite, PixelFormat.Format32bppArgb);

        for(Int32 i = 0; i < ImageHeight; i++) {
          Byte[] ImageArray = Reader.ReadBytes(ImageWidth * 4);

          System.Runtime.InteropServices.Marshal.Copy(ImageArray, 0, new IntPtr(Data.Scan0.ToInt64() + i * ImageWidth * 4L), ImageWidth * 4);
        }

        Image.UnlockBits(Data);

        return Image;
      } catch(Exception) {

      }
      return null;
    }

    #endregion
  }

  public class FileViewerTDAnimation {
    #region Public Fields

    public IFileViewerProgressBar Parent;
    public List<ListViewItem> ItemList;
    public BinaryReader Reader;
    public FileStream Stream;
    public String FilePath;
    public Int64 Position;
    public Int64 Size;

    #endregion


    #region Contructor

    public FileViewerTDAnimation() {
      this.ItemList = new List<ListViewItem>();
    }

    #endregion


    #region Private Methods

    private void AddItem(Int64 Size, String Type, String Name, String Value) {
      this.ItemList.Add(new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value }));

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }

    private void AddItem(Int64 Size, String Type, String Name, String Value, FileViewerObject Tag) {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value });

      Item.BackColor = Color.FromArgb(240, 255, 245);
      Item.Tag = Tag;

      this.ItemList.Add(Item);

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }


    private void AddItemError(String Value) {
      ListViewItem Item = new ListViewItem(new String[] { "".PadRight(10), "".PadRight(10), "".PadRight(10), "Error".PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }

    private void AddItemError(Exception e) {
      String Name = String.Empty;
      String Value = String.Empty;

      if(e.InnerException != null && e.InnerException.Message != null) {
        if(e.Message != null)
          Name = e.Message;

        Value = e.InnerException.Message;
      } else {
        if(e.Message != null)
          Value = e.Message;
      }

      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "ERROR".PadRight(10), Name.PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }


    private void AddItemBeginRead() {
      ListViewItem Item = new ListViewItem(new String[] { "0".PadRight(10), "0".PadRight(10), "".PadRight(10), "BeginRead".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }

    private void AddItemEndOfFile() {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "".PadRight(10), "EndOfFile".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }


    private void Close() {
      if(this.Stream != null) {
        this.Stream.Close();
        this.Stream = null;
      }

      if(this.Reader != null) {
        this.Reader.Close();
        this.Reader = null;
      }
    }

    #endregion

    #region Private Methods: Format

    private UInt32 ReadIDNumber() {
      UInt32 IDNumber = this.Reader.ReadUInt32();

      if(IDNumber != 0x41504454)
        throw new Exception("IDNumber", new Exception("0x" + IDNumber.ToString("X8")));

      this.AddItem(4, "UINT32", "IDNumber", "0x" + IDNumber.ToString("X8") + " hex");
      return IDNumber;
    }

    private Int64 ReadFileSize() {
      Int64 FileSize = this.Reader.ReadInt64();

      if(FileSize != this.Size)
        throw new Exception("FileSize", new Exception(FileSize.ToString()));

      this.AddItem(8, "INT64", "FileSize", FileSize.ToString("#,#") + " bytes");
      return FileSize;
    }

    private Byte ReadVersion() {
      Byte Version = this.Reader.ReadByte();

      if(Version == 0 || Version > 1)
        throw new Exception("Version", new Exception(Version.ToString()));

      this.AddItem(1, "BYTE", "Version", Version.ToString());
      return Version;
    }

    private Int32 ReadThumbnailSize() {
      Int32 ThumbnailSize = this.Reader.ReadInt32();

      if(ThumbnailSize < 0 || ThumbnailSize > this.Size - this.Position - 4)
        throw new Exception("ThumbnailSize", new Exception(ThumbnailSize.ToString()));

      if(ThumbnailSize == 0)
        this.AddItem(4, "INT32", "ThumbnailSize", ThumbnailSize + " byte");
      else
        this.AddItem(4, "INT32", "ThumbnailSize", ThumbnailSize.ToString("#,#") + " bytes");
      return ThumbnailSize;
    }

    private void ReadThumbnailImage(Int32 ThumbnailSize) {
      try {
        Byte[] ThumbnailImageArray = this.Reader.ReadBytes(ThumbnailSize);

        Bitmap ThumbnailImage = ThumbnailImage = new Bitmap(new MemoryStream(ThumbnailImageArray));

        this.AddItem(ThumbnailSize, "MEMORY", "ThumbnailImage", "", new FileViewerObject(ThumbnailImage));
      } catch(Exception e) {
        throw new Exception("ThumbnailImage", e);
      }
    }

    private UInt32 ReadDisplayColor() {
      UInt32 DisplayColor = this.Reader.ReadUInt32();

      this.AddItem(4, "UINT32", "DisplayColor", "0x" + DisplayColor.ToString("X8") + " hex");
      return DisplayColor;
    }

    private Int32 ReadDisplayWidth() {
      Int32 DisplayWidth = this.Reader.ReadInt32();

      if(DisplayWidth <= 0 || DisplayWidth > 32000)
        throw new Exception("DisplayWidth", new Exception(DisplayWidth.ToString()));

      this.AddItem(4, "INT32", "DisplayWidth", DisplayWidth.ToString("#,#") + " pixel");
      return DisplayWidth;
    }

    private Int32 ReadDisplayHeight() {
      Int32 DisplayHeight = this.Reader.ReadInt32();

      if(DisplayHeight <= 0 || DisplayHeight > 32000)
        throw new Exception("DisplayHeight", new Exception(DisplayHeight.ToString()));

      this.AddItem(4, "INT32", "DisplayHeight", DisplayHeight.ToString("#,#") + " pixel");
      return DisplayHeight;
    }

    private Int32 ReadTimeTick() {
      Int32 TimeTick = this.Reader.ReadInt32();

      if(TimeTick <= 0 || TimeTick > 100000)
        throw new Exception("TimeTick", new Exception(TimeTick.ToString()));

      this.AddItem(4, "INT32", "TimeTick", TimeTick + " x 15 ms");
      return TimeTick;
    }

    private Int32 ReadFrameCount() {
      Int32 FrameCount = this.Reader.ReadInt32();

      if(FrameCount <= 0 || FrameCount > 100000)
        throw new Exception("FrameCount", new Exception(FrameCount.ToString()));

      this.AddItem(4, "INT32", "FrameCount", FrameCount.ToString("#,#") + " frames");
      return FrameCount;
    }


    private Int32 ReadImageCount() {
      Int32 ImageCount = this.Reader.ReadInt32();

      if(ImageCount <= 0 || ImageCount > 100000)
        throw new Exception("ImageCount", new Exception(ImageCount.ToString()));

      this.AddItem(4, "INT32", "ImageCount", ImageCount.ToString("#,#") + " images");
      return ImageCount;
    }

    private Int32 ReadImageNameLength() {
      Int32 ImageNameLength = this.Reader.ReadInt32();

      if(ImageNameLength < -1)
        throw new Exception("ImageNameLength", new Exception(ImageNameLength.ToString()));

      this.AddItem(4, "INT32", "ImageNameLength", ImageNameLength + " letters");
      return ImageNameLength;
    }

    private String ReadImageName(Int32 ImageNameLength) {
      Char[] LetterArray = new Char[ImageNameLength];

      for(Int32 i = 0; i < ImageNameLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String ImageName = new String(LetterArray);

      if(ImageName == null || ImageName == String.Empty)
        throw new Exception("ImageName", new Exception(ImageName));

      this.AddItem(4, "WCHAR[ ]", "ImageName", ImageName);
      return ImageName;
    }

    private Byte ReadImageMode() {
      Byte ImageMode = this.Reader.ReadByte();

      if(ImageMode > 2)
        throw new Exception("ImageMode", new Exception("0x" + ImageMode.ToString("X4")));

      this.AddItem(1, "BYTE", "ImageMode", "0x" + ImageMode.ToString("X4"));
      return ImageMode;
    }

    private Int32 ReadImageWidth() {
      Int32 ImageWidth = this.Reader.ReadInt32();

      if(ImageWidth <= 0 || ImageWidth > 32000)
        throw new Exception("ImageWidth", new Exception(ImageWidth.ToString()));

      this.AddItem(4, "INT32", "ImageWidth", ImageWidth.ToString("#,#") + " pixels");
      return ImageWidth;
    }

    private Int32 ReadImageHeight() {
      Int32 ImageHeight = this.Reader.ReadInt32();

      if(ImageHeight <= 0 || ImageHeight > 32000)
        throw new Exception("ImageHeight", new Exception(ImageHeight.ToString()));

      this.AddItem(4, "INT32", "ImageHeight", ImageHeight.ToString("#,#") + " pixels");
      return ImageHeight;
    }

    private Int32 ReadImageItemWidth(Int32 ImageWidth) {
      Int32 ImageItemWidth = this.Reader.ReadInt32();

      if(ImageItemWidth <= 0 || ImageItemWidth > ImageWidth)
        throw new Exception("ImageItemWidth", new Exception(ImageItemWidth.ToString()));

      this.AddItem(4, "INT32", "ImageItemWidth", ImageItemWidth.ToString("#,#") + " pixels");
      return ImageItemWidth;
    }

    private Int32 ReadImageItemCount(Int32 ImageWidth) {
      Int32 ImageItemCount = this.Reader.ReadInt32();

      if(ImageItemCount <= 0 || ImageItemCount > ImageWidth)
        throw new Exception("ImageItemCount", new Exception(ImageItemCount.ToString()));

      this.AddItem(4, "INT32", "ImageItemCount", ImageItemCount.ToString("#,#") + " pixels");
      return ImageItemCount;
    }

    private Int32 ReadImageMemorySize() {
      Int32 ImageMemorySize = this.Reader.ReadInt32();

      if(ImageMemorySize <= 0 || ImageMemorySize > 100000000)
        throw new Exception("ImageMemorySize", new Exception(ImageMemorySize.ToString()));

      this.AddItem(4, "INT32", "ImageMemorySize", ImageMemorySize.ToString("#,#") + " bytes");
      return ImageMemorySize;
    }

    private void ReadImageMemory(Int32 ImageMemorySize) {
      try {
        Byte[] ImageMemoryArray = this.Reader.ReadBytes(ImageMemorySize);

        Bitmap Image = new Bitmap(new MemoryStream(ImageMemoryArray));

        this.AddItem(ImageMemorySize, "MEMORY", "ImageMemory", "", new FileViewerObject(Image));
      } catch(Exception e) {
        throw new Exception("ImageMemory", e);
      }
    }


    private Int32 ReadItemCount() {
      Int32 ItemCount = this.Reader.ReadInt32();

      if(ItemCount <= 0 || ItemCount > 100000)
        throw new Exception("ItemCount", new Exception(ItemCount.ToString()));

      this.AddItem(4, "INT32", "ItemCount", ItemCount.ToString("#,#") + " items");
      return ItemCount;
    }

    private Int32 ReadItemImageIndex(Int32 ImageCount) {
      Int32 ItemImageIndex = this.Reader.ReadInt32();

      if(ItemImageIndex < 0 || ItemImageIndex >= ImageCount)
        throw new Exception("ItemImageIndex", new Exception(ItemImageIndex.ToString()));

      this.AddItem(4, "INT32", "ItemImageIndex", ItemImageIndex.ToString());
      return ItemImageIndex;
    }

    private Int32 ReadItemMemorySize(Int32 FrameCount) {
      Int32 ItemMemorySize = this.Reader.ReadInt32();

      if(ItemMemorySize <= 0 || ItemMemorySize > FrameCount * 11)
        throw new Exception("ItemMemorySize", new Exception(ItemMemorySize.ToString()));

      this.AddItem(4, "INT32", "ItemMemorySize", ItemMemorySize.ToString("#,#") + " bytes");
      return ItemMemorySize;
    }

    private void ReadItemMemory(Int32 ItemMemorySize) {
      Byte[] ItemMemoryArray = this.Reader.ReadBytes(ItemMemorySize);

      MemoryStream ItemStream = new MemoryStream(ItemMemoryArray);

      this.AddItem(ItemMemorySize, "MEMORY", "ItemMemory", "", new FileViewerObject(FileViewerObjectType.Animation, ItemStream));
    }


    private Int32 ReadSoundCount() {
      Int32 SoundCount = this.Reader.ReadInt32();

      if(SoundCount < 0 || SoundCount > 100)
        throw new Exception("SoundCount", new Exception(SoundCount.ToString()));

      this.AddItem(4, "INT32", "SoundCount", SoundCount + " sounds");
      return SoundCount;
    }

    private Int32 ReadSoundNameLength() {
      Int32 SoundNameLength = this.Reader.ReadInt32();

      if(SoundNameLength < -1)
        throw new Exception("SoundNameLength", new Exception(SoundNameLength.ToString()));

      this.AddItem(4, "INT32", "SoundNameLength", SoundNameLength + " letters");
      return SoundNameLength;
    }

    private String ReadSoundName(Int32 SoundNameLength) {
      Char[] LetterArray = new Char[SoundNameLength];

      for(Int32 i = 0; i < SoundNameLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String SoundName = new String(LetterArray);

      if(SoundName == null || SoundName == String.Empty)
        throw new Exception("SoundName", new Exception(SoundName));

      this.AddItem(4, "WCHAR[ ]", "SoundName", SoundName);
      return SoundName;
    }

    private Int64 ReadSoundDuration() {
      Int64 SoundDuration = this.Reader.ReadInt64();

      if(SoundDuration <= 0)
        throw new Exception("SoundDuration", new Exception(SoundDuration.ToString()));

      this.AddItem(8, "INT64", "SoundDuration", SoundDuration.ToString() + " * 100ns");
      return SoundDuration;
    }

    private Int32 ReadSoundMemorySize() {
      Int32 SoundMemorySize = this.Reader.ReadInt32();

      if(SoundMemorySize <= 0 || SoundMemorySize > 100000000)
        throw new Exception("SoundMemorySize", new Exception(SoundMemorySize.ToString()));

      this.AddItem(4, "INT32", "SoundMemorySize", SoundMemorySize.ToString("#,#") + " bytes");
      return SoundMemorySize;
    }

    private void ReadSoundMemory(Int32 SoundMemorySize) {
      Byte[] SoundMemoryArray = this.Reader.ReadBytes(SoundMemorySize);

      this.AddItem(SoundMemorySize, "MEMORY", "SoundMemory", "", new FileViewerObject(FileViewerObjectType.Sound, SoundMemoryArray));
    }


    private Int32 ReadSoundItemCount() {
      Int32 SoundItemCount = this.Reader.ReadInt32();

      if(SoundItemCount <= 0 || SoundItemCount > 100)
        throw new Exception("SoundItemCount", new Exception(SoundItemCount.ToString()));

      this.AddItem(4, "INT32", "SoundItemCount", SoundItemCount + " sound items");
      return SoundItemCount;
    }

    private Int32 ReadSoundItemIndex(Int32 SoundCount) {
      Int32 SoundItemIndex = this.Reader.ReadInt32();

      if(SoundItemIndex < 0 || SoundItemIndex >= SoundCount)
        throw new Exception("SoundItemIndex", new Exception(SoundItemIndex.ToString()));

      this.AddItem(4, "INT32", "SoundItemIndex", SoundItemIndex.ToString());
      return SoundItemIndex;
    }

    private Int32 ReadSoundItemRuns(Int32 Frames) {
      Int32 SoundItemRuns = this.Reader.ReadInt32();

      if(SoundItemRuns <= 0 || SoundItemRuns > Frames)
        throw new Exception("SoundItemRuns", new Exception(SoundItemRuns.ToString()));

      this.AddItem(4, "INT32", "SoundItemRuns", SoundItemRuns.ToString());
      return SoundItemRuns;
    }

    private Int32 ReadSoundItemFrame(Int32 Frames) {
      Int32 SoundItemFrame = this.Reader.ReadInt32();

      if(SoundItemFrame < 0 || SoundItemFrame > Frames)
        throw new Exception("SoundItemFrame", new Exception(SoundItemFrame.ToString()));

      this.AddItem(4, "INT32", "SoundItemFrame", SoundItemFrame.ToString());
      return SoundItemFrame;
    }


    private void ReadBegin() {
      UInt32 IDNumber = this.ReadIDNumber();

      Int64 FileSize = this.ReadFileSize();

      Byte Version = this.ReadVersion();

      Int32 ThumbnailSize = this.ReadThumbnailSize();

      if(ThumbnailSize > 0) this.ReadThumbnailImage(ThumbnailSize);

      UInt32 DisplayColor = this.ReadDisplayColor();

      Int32 DisplayWidth = this.ReadDisplayWidth();
      Int32 DisplayHeight = this.ReadDisplayHeight();

      Int32 TimeTick = this.ReadTimeTick();
      Int32 FrameCount = this.ReadFrameCount();

      //Images
      Int32 ImageCount = this.ReadImageCount();

      for(Int32 i = 0; i < ImageCount; i++) {
        Int32 ImageNameLength = this.ReadImageNameLength();

        if(ImageNameLength > 0) {
          String ImageName = this.ReadImageName(ImageNameLength);
        }

        Byte ImageMode = this.ReadImageMode();

        Int32 ImageWidth = this.ReadImageWidth();
        Int32 ImageHeight = this.ReadImageHeight();

        Int32 ImageItemWidth = this.ReadImageItemWidth(ImageWidth);
        Int32 ImageItemCount = this.ReadImageItemCount(ImageWidth);

        Int32 ImageMemorySize = this.ReadImageMemorySize();

        this.ReadImageMemory(ImageMemorySize);
      }

      //Items
      Int32 ItemCount = this.ReadItemCount();

      for(Int32 i = 0; i < ItemCount; i++) {
        Int32 ItemImageIndex = this.ReadItemImageIndex(ImageCount);
        Int32 ItemMemorySize = this.ReadItemMemorySize(FrameCount);

        this.ReadItemMemory(ItemMemorySize);
      }

      //Sounds
      Int32 SoundCount = this.ReadSoundCount();

      if(SoundCount == 0) {
        if(this.Position != this.Size) throw new Exception("Wrong FileSize");
        return;
      }

      for(Int32 i = 0; i < SoundCount; i++) {
        Int32 SoundNameLength = this.ReadSoundNameLength();

        if(SoundNameLength > 0) {
          String SoundName = this.ReadSoundName(SoundNameLength);
        }

        Int64 SoundDuration = this.ReadSoundDuration();
        Int32 SoundMemorySize = this.ReadSoundMemorySize();

        this.ReadSoundMemory(SoundMemorySize);
      }

      //SoundItems
      Int32 SoundItemCount = this.ReadSoundItemCount();

      for(Int32 i = 0; i < SoundItemCount; i++) {
        Int32 SoundItemIndex = this.ReadSoundItemIndex(SoundCount);
        Int32 SoundItemRuns = this.ReadSoundItemRuns(FrameCount);

        for(Int32 j = 0; j < SoundItemRuns; j++) {
          Int32 SoundItemFrame = this.ReadSoundItemFrame(FrameCount);
        }
      }

      if(this.Position != this.Size) throw new Exception("Wrong FileSize");
    }

    #endregion


    #region Public Methods

    public List<ListViewItem> Run(IFileViewerProgressBar Parent, String FilePath) {
      this.Parent = Parent;

      try {
        this.Stream = new FileStream(FilePath, FileMode.Open, FileAccess.Read);
        this.Reader = new BinaryReader(this.Stream);

        this.Size = this.Stream.Length;

        this.Parent.BeginProgressBar(this.Size);

        this.AddItemBeginRead();

        this.ReadBegin();
      } catch(Exception e) {
        this.AddItemError(e);

        this.Close();
        return this.ItemList;
      }

      this.AddItemEndOfFile();

      this.Close();
      return this.ItemList;
    }

    #endregion
  }


  public class FileViewerProtectedDataObject {
    #region Public Fields

    public String Title;
    public String Name;

    public Byte[] ReadArray;
    public Byte[] CodeArray;
    public Byte[] ValueArray;

    #endregion

    #region Contructor

    public FileViewerProtectedDataObject(String Title, String Name, Byte[] ReadArray, Byte[] CodeArray, Byte[] ValueArray) {
      this.Title = Title;
      this.Name = Name;
      this.ReadArray = ReadArray;
      this.CodeArray = CodeArray;
      this.ValueArray = ValueArray;
    }

    #endregion
  }

  public class FileViewerProtectedFileObject {
    #region Public Fields

    public String Name;

    public Int64 Position;
    public Int64 Size;

    public FileViewerProtectedData Data;

    #endregion

    #region Contructor

    public FileViewerProtectedFileObject(String Name, Int64 Position, Int64 Size, FileViewerProtectedData Data) {
      this.Name = Name;
      this.Position = Position;
      this.Size = Size;
      this.Data = Data;
    }

    #endregion
  }

  public class FileViewerProtectedData {
    #region Public Fields

    public IFileViewerProgressBar Parent;
    public List<ListViewItem> ItemList;
    public BinaryReader Reader;
    public FileStream Stream;
    public String FilePath;
    public Int64 Position;
    public Int64 Size;

    public Byte[] PassArray;

    public UInt64 CodeValue;
    public UInt64 CodeIndex;

    public Byte[] CodeValueMemory;
    public Byte[] CodePageMemory;

    public UInt32 ValueMemorySize;
    public UInt32 PageMemorySize;

    public Int32 FolderCount;

    #endregion


    #region Contructor

    public FileViewerProtectedData() {
      this.ItemList = new List<ListViewItem>();
    }

    #endregion


    #region Private Methods

    private void AddItem(String Name, String Value) {
      this.ItemList.Add(new ListViewItem(new String[] { "", "", "", Name, Value }));

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }

    private void AddItem(Int64 Size, String Type, String Name, String Value) {
      this.ItemList.Add(new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value }));

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }

    private void AddItem(Int64 Size, String Type, String Name, String Value, FileViewerObject Tag) {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value });

      Item.BackColor = Color.FromArgb(240, 255, 245);
      Item.Tag = Tag;

      this.ItemList.Add(Item);

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }


    private void AddItemError(String Value) {
      ListViewItem Item = new ListViewItem(new String[] { "".PadRight(10), "".PadRight(10), "".PadRight(10), "Error".PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }

    private void AddItemError(Exception e) {
      String Name = String.Empty;
      String Value = String.Empty;

      if(e.InnerException != null && e.InnerException.Message != null) {
        if(e.Message != null)
          Name = e.Message;

        Value = e.InnerException.Message;
      } else {
        if(e.Message != null)
          Value = e.Message;
      }

      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "ERROR".PadRight(10), Name.PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }


    private void AddItemBeginRead() {
      ListViewItem Item = new ListViewItem(new String[] { "0".PadRight(10), "0".PadRight(10), "".PadRight(10), "BeginRead".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }

    private void AddItemEndOfFile() {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "".PadRight(10), "EndOfFile".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }


    private void Close() {
      if(this.Stream != null) {
        this.Stream.Close();
        this.Stream = null;
      }

      if(this.Reader != null) {
        this.Reader.Close();
        this.Reader = null;
      }
    }


    private String GetSizeString(Int64 Size) {
      if(Size == 0) return "0 byte";
      return Size.ToString("#,#") + " bytes";
    }

    private String GetValueString(Int64 Value) {
      if(Value == 0) return "0";
      return Value.ToString("#,#");
    }

    #endregion

    #region Private Methods: Format

    private UInt32 ReadIDNumber() {
      UInt32 IDNumber = this.Reader.ReadUInt32();

      this.AddItem(4, "UINT32", "IDNumber", "0x" + IDNumber.ToString("X8") + " hex");
      return IDNumber;
    }

    private UInt32 ReadLoopMax(Boolean UseFileIdentifier) {
      UInt32 LoopMax = this.Reader.ReadUInt32();

      if(UseFileIdentifier) {
        this.AddItem(4, "UINT32", "LoopMax", this.GetValueString(LoopMax));
        return LoopMax;
      }

      this.AddItem(4, "UINT32", "LoopMax", "0x" + LoopMax.ToString("X2") + " hex");
      return UInt32.MaxValue / 2;
    }

    private Byte ReadPassVersion(Boolean UseFileIdentifier) {
      Byte Value = this.Reader.ReadByte();

      if(UseFileIdentifier) {
        if((Value & 0x7F) != 1)
          throw new Exception("PassVersion", new Exception("0x" + Value.ToString("X2") + " hex"));

        if((Value & 0x80) == 0x80) {
          this.AddItem(1, "BYTE", "PassVersion", "0x" + Value.ToString("X2") + " hex");
          return Value;
        }
      }

      this.AddItem(1, "BYTE", "PassVersion", "0x" + Value.ToString("X2") + " hex");
      return Value;
    }

    private Byte ReadFileCount(Boolean UseFileIdentifier) {
      Byte FileCount = this.Reader.ReadByte();

      if(UseFileIdentifier)
        this.AddItem(1, "BYTE", "FileCount", FileCount.ToString());
      else
        this.AddItem(1, "BYTE", "FileCount", "0x" + FileCount.ToString("X2") + " hex");

      return FileCount;
    }

    private Byte ReadFactorCount(Boolean UsePassword) {
      Byte FactorCount = this.Reader.ReadByte();

      if(UsePassword) FactorCount ^= this.PassArray[0];

      FactorCount = (Byte) (FactorCount | 0x80);

      this.AddItem(1, "BYTE", "FactorCount", FactorCount.ToString());
      return FactorCount;
    }

    private Byte[] ReadFactorMemory(Boolean UsePassword, Int32 FactorCount) {
      Byte[] FactorMemory = this.Reader.ReadBytes(FactorCount);
      Byte[] PassMemory = new Byte[FactorCount];
      Byte[] FactorPassMemory = new Byte[FactorCount];

      Array.Copy(FactorMemory, FactorPassMemory, FactorCount);

      if(UsePassword) {
        Array.Copy(this.PassArray, 1, PassMemory, 0, FactorCount);

        for(Int32 i = 0; i < FactorCount; i++)
          FactorPassMemory[i] ^= PassMemory[i];
      }

      this.AddItem(FactorCount, "BYTE[]", "FactorMemory", "", new FileViewerObject(FileViewerObjectType.ProtectedData, new FileViewerProtectedDataObject("FactorMemory", "Data", FactorMemory, PassMemory, FactorPassMemory)));
      return FactorPassMemory;
    }

    private Byte[] ReadReferenceMemory(Boolean UsePassword, Int32 FactorCount) {
      Byte[] ReferenceMemory = this.Reader.ReadBytes(FactorCount);
      Byte[] PassMemory = new Byte[FactorCount];
      Byte[] ReferencePassMemory = new Byte[FactorCount];

      Array.Copy(ReferenceMemory, ReferencePassMemory, FactorCount);

      if(UsePassword) {
        Array.Copy(this.PassArray, FactorCount + 1, PassMemory, 0, FactorCount);

        for(Int32 i = 0; i < FactorCount; i++)
          ReferencePassMemory[i] ^= PassMemory[i];
      }

      this.AddItem(FactorCount, "BYTE[]", "ReferenceMemory", "", new FileViewerObject(FileViewerObjectType.ProtectedData, new FileViewerProtectedDataObject("ReferenceMemory", "Data", ReferenceMemory, PassMemory, ReferencePassMemory)));
      return ReferencePassMemory;
    }


    private void MakeCodeValueIndex(Byte FactorCount, UInt32 LoopMax, Byte[] FactorMemory, Byte[] ReferenceMemory) {
      this.CodeValue = 0;
      this.CodeIndex = 1;

      Int32 Index = 0;
      Int32 UpdateIndex = 0;

      LoopMax++;

      Int64 ProgressIndex = 0;

      this.Parent.BeginProgressBar((Int64) LoopMax);

      for(UInt64 i = 1; i <= LoopMax; i++) {
        for(UInt64 j = 1; j <= FactorCount; j++)
          this.CodeValue += (FactorMemory[j - 1] + (this.CodeValue + 1) * i) * j;

        this.CodeIndex++;
        this.CodeValue /= FactorCount;

        if((Byte) this.CodeValue == ReferenceMemory[Index]) {
          Index++;

          if(Index == FactorCount) {
            this.Parent.BeginProgressBar(this.Size);
            this.AddItem(0, "INFO", "LoopCount", this.GetValueString((Int64) CodeIndex));
            return;
          }
        } else {
          Index = 0;
        }

        UpdateIndex++;
        ProgressIndex++;

        if(UpdateIndex == 10000) {
          UpdateIndex = 0;

          if(!this.Parent.UpdateProgressBar(ProgressIndex))
            throw new Exception("Reading was canceled.");
        }
      }

      throw new Exception("LoopCount", new Exception(this.CodeIndex.ToString()));
    }

    private Byte[] MakeCodeMemory(Byte FactorCount, Byte[] FactorMemory, Byte[] ReferenceMemory, UInt32 Count) {
      Byte[] ByteArray = new Byte[(Int32) Count];

      Int32 Index = 0;
      Int32 UpdateIndex = 0;

      Int64 ProgressIndex = 0;

      this.Parent.BeginProgressBar((Int64) Count);

      for(UInt64 i = this.CodeIndex; i < this.CodeIndex + Count; i++) {
        for(UInt64 j = 1; j <= FactorCount; j++)
          this.CodeValue += (FactorMemory[j - 1] + (this.CodeValue + 1) * i) * j;

        this.CodeValue /= FactorCount;

        ByteArray[Index++] = (Byte) this.CodeValue;

        UpdateIndex++;
        ProgressIndex++;

        if(UpdateIndex == 10000) {
          UpdateIndex = 0;

          if(!this.Parent.UpdateProgressBar(ProgressIndex))
            throw new Exception("Reading was canceled.");
        }
      }

      this.CodeIndex += Count;

      this.Parent.BeginProgressBar(this.Size);
      return ByteArray;
    }


    private UInt32 ReadCodeValueCount(Byte FactorCount, Byte[] FactorMemory, Byte[] ReferenceMemory) {
      Byte[] ReadMemory = this.Reader.ReadBytes(4);
      Byte[] CodeMemory = this.MakeCodeMemory(FactorCount, FactorMemory, ReferenceMemory, 4);
      Byte[] ValueMemory = new Byte[4];

      UInt32 CodeValueCount = 0;
      UInt32[] FactorArray = new UInt32[] { 0x00000001, 0x00000100, 0x00010000, 0x01000000 };

      Array.Copy(ReadMemory, ValueMemory, 4);

      for(Int32 i = 0; i < 4; i++) {
        ValueMemory[i] ^= CodeMemory[i];
        CodeValueCount += FactorArray[i] * ValueMemory[i];
      }

      if(CodeValueCount >= 500000) {
        this.AddItem(4, "UINT32", "CodeValueCount", this.GetValueString(CodeValueCount), new FileViewerObject(FileViewerObjectType.ProtectedData, new FileViewerProtectedDataObject("CodeValueCount", "Value", ReadMemory, CodeMemory, ValueMemory)));
        return CodeValueCount;
      }

      throw new Exception("CodeValueCount", new Exception(CodeValueCount.ToString()));
    }

    private UInt32 ReadCodePageCount(Byte FactorCount, Byte[] FactorMemory, Byte[] ReferenceMemory) {
      Byte[] ReadMemory = this.Reader.ReadBytes(4);
      Byte[] CodeMemory = this.MakeCodeMemory(FactorCount, FactorMemory, ReferenceMemory, 4);
      Byte[] ValueMemory = new Byte[4];

      UInt32 CodePageCount = 0;
      UInt32[] FactorArray = new UInt32[] { 0x00000001, 0x00000100, 0x00010000 };

      Array.Copy(ReadMemory, ValueMemory, 4);

      for(Int32 i = 0; i < 3; i++) {
        ValueMemory[i] ^= CodeMemory[i];
        CodePageCount += FactorArray[i] * ValueMemory[i];
      }

      ValueMemory[3] ^= CodeMemory[3];

      Byte FileVersion = ValueMemory[3];

      if(FileVersion == 1) {
        if(CodePageCount == 0 || (CodePageCount >= 500000 && CodePageCount <= 1000000)) {
          this.AddItem(4, "UINT32", "CodePageCount", this.GetValueString(CodePageCount) + ", FileVersion = " + FileVersion, new FileViewerObject(FileViewerObjectType.ProtectedData, new FileViewerProtectedDataObject("CodePageCount", "Value", ReadMemory, CodeMemory, ValueMemory)));
          return CodePageCount;
        }
      }
      throw new Exception("CodePageCount", new Exception(CodePageCount.ToString() + ", FileVersion = " + FileVersion));
    }


    private BinaryReader ReadMemoryCodeValue(UInt32 Count) {
      Byte[] ByteArray = this.Reader.ReadBytes((Int32) Count);

      UInt64 CodePosition = (UInt64) this.Position;

      UInt32 CodeValuePosition = (UInt32) (CodePosition - CodePosition / this.ValueMemorySize * this.ValueMemorySize);
      UInt32 CodeValueCount = this.ValueMemorySize - CodeValuePosition;

      UInt32 Index = 0;

      while(true) {
        if(CodeValueCount > Count - Index) CodeValueCount = Count - Index;

        for(UInt32 i = 0; i < CodeValueCount; i++)
          ByteArray[Index++] ^= this.CodeValueMemory[CodeValuePosition++];

        if(Index == Count) break;

        CodeValuePosition = 0;
        CodeValueCount = this.ValueMemorySize;
      }
      return new BinaryReader(new MemoryStream(ByteArray));
    }

    private BinaryReader ReadMemoryCodePage(UInt32 Count) {
      Byte[] ByteArray = this.Reader.ReadBytes((Int32) Count);

      UInt64 CodePosition = (UInt64) this.Position;
      UInt32 CodePagePosition = (UInt32) (CodePosition / this.ValueMemorySize);

      UInt32 CodeValuePosition = (UInt32) (CodePosition - CodePagePosition * this.ValueMemorySize);
      UInt32 CodeValueCount = this.ValueMemorySize - CodeValuePosition;

      UInt32 Index = 0;

      while(true) {
        if(CodeValueCount > Count - Index) CodeValueCount = Count - Index;

        Byte CodePage = this.CodePageMemory[CodePagePosition];

        for(UInt32 i = 0; i < CodeValueCount; i++)
          ByteArray[Index++] ^= (Byte) (CodePosition++ * this.CodeValueMemory[CodeValuePosition++] + CodePage);

        if(Index == Count) break;

        CodePagePosition++;

        CodeValuePosition = 0;
        CodeValueCount = this.ValueMemorySize;
      }
      return new BinaryReader(new MemoryStream(ByteArray));
    }

    private BinaryReader ReadMemory(UInt32 Count) {
      if(this.CodePageMemory == null)
        return this.ReadMemoryCodeValue(Count);

      return this.ReadMemoryCodePage(Count);
    }


    private void ReadMemoryCodeValue(Byte[] ByteArray, UInt32 Count) {
      this.Stream.Read(ByteArray, 0, (Int32) Count);

      UInt64 CodePosition = (UInt64) this.Position;

      UInt32 CodeValuePosition = (UInt32) (CodePosition - CodePosition / this.ValueMemorySize * this.ValueMemorySize);
      UInt32 CodeValueCount = this.ValueMemorySize - CodeValuePosition;

      UInt32 Index = 0;

      while(true) {
        if(CodeValueCount > Count - Index) CodeValueCount = Count - Index;

        for(UInt32 i = 0; i < CodeValueCount; i++)
          ByteArray[Index++] ^= this.CodeValueMemory[CodeValuePosition++];

        if(Index == Count) break;

        CodeValuePosition = 0;
        CodeValueCount = this.ValueMemorySize;
      }
    }

    private void ReadMemoryCodePage(Byte[] ByteArray, UInt32 Count) {
      this.Stream.Read(ByteArray, 0, (Int32) Count);

      UInt64 CodePosition = (UInt64) this.Position;
      UInt32 CodePagePosition = (UInt32) (CodePosition / this.ValueMemorySize);

      UInt32 CodeValuePosition = (UInt32) (CodePosition - CodePagePosition * this.ValueMemorySize);
      UInt32 CodeValueCount = this.ValueMemorySize - CodeValuePosition;

      UInt32 Index = 0;

      while(true) {
        if(CodeValueCount > Count - Index) CodeValueCount = Count - Index;

        Byte CodePage = this.CodePageMemory[CodePagePosition];

        for(UInt32 i = 0; i < CodeValueCount; i++)
          ByteArray[Index++] ^= (Byte) (CodePosition++ * this.CodeValueMemory[CodeValuePosition++] + CodePage);

        if(Index == Count) break;

        CodePagePosition++;

        CodeValuePosition = 0;
        CodeValueCount = this.ValueMemorySize;
      }
    }

    private void ReadMemory(Byte[] ByteArray, UInt32 Count) {
      if(this.CodePageMemory == null) {
        this.ReadMemoryCodeValue(ByteArray, Count);
        return;
      }

      this.ReadMemoryCodePage(ByteArray, Count);
    }


    private Byte[] CreateMemoryCodeValue(UInt32 Count) {
      Byte[] ByteArray = new Byte[Count];

      UInt64 CodePosition = (UInt64) this.Position;

      UInt32 CodeValuePosition = (UInt32) (CodePosition - CodePosition / this.ValueMemorySize * this.ValueMemorySize);
      UInt32 CodeValueCount = this.ValueMemorySize - CodeValuePosition;

      UInt32 Index = 0;

      while(true) {
        if(CodeValueCount > Count - Index) CodeValueCount = Count - Index;

        for(UInt32 i = 0; i < CodeValueCount; i++)
          ByteArray[Index++] = this.CodeValueMemory[CodeValuePosition++];

        if(Index == Count) break;

        CodeValuePosition = 0;
        CodeValueCount = this.ValueMemorySize;
      }
      return ByteArray;
    }

    private Byte[] CreateMemoryCodePage(UInt32 Count) {
      Byte[] ByteArray = new Byte[Count];

      UInt64 CodePosition = (UInt64) this.Position;
      UInt32 CodePagePosition = (UInt32) (CodePosition / this.ValueMemorySize);

      UInt32 CodeValuePosition = (UInt32) (CodePosition - CodePagePosition * this.ValueMemorySize);
      UInt32 CodeValueCount = this.ValueMemorySize - CodeValuePosition;

      UInt32 Index = 0;

      while(true) {
        if(CodeValueCount > Count - Index) CodeValueCount = Count - Index;

        Byte CodePage = this.CodePageMemory[CodePagePosition];

        for(UInt32 i = 0; i < CodeValueCount; i++)
          ByteArray[Index++] = (Byte) (CodePosition++ * this.CodeValueMemory[CodeValuePosition++] + CodePage);

        if(Index == Count) break;

        CodePagePosition++;

        CodeValuePosition = 0;
        CodeValueCount = this.ValueMemorySize;
      }
      return ByteArray;
    }

    private Byte[] CreateMemory(UInt32 Count) {
      if(this.CodePageMemory == null)
        return this.CreateMemoryCodeValue(Count);

      return this.CreateMemoryCodePage(Count);
    }


    private UInt16 ReadHeaderSize() {
      UInt16 HeaderSize = this.ReadMemory(2).ReadUInt16();

      if(HeaderSize < 4 || HeaderSize > 535)
        throw new Exception("HeaderSize", new Exception(this.GetSizeString(HeaderSize)));

      this.AddItem(2, "UINT16", "HeaderSize", this.GetSizeString(HeaderSize));
      return HeaderSize;
    }

    private Byte ReadHeaderFlags(BinaryReader HeaderReader) {
      Byte Flags = HeaderReader.ReadByte();

      this.AddItem(1, "BYTE", "HeaderFlags", "0x" + Flags.ToString("X2") + " hex");
      return Flags;
    }


    private Int32 ReadNameSize(String Name, BinaryReader HeaderReader, Byte HeaderFlags) {
      Int32 NameSize = 0;

      if((HeaderFlags & 0x10) == 0x10) {
        NameSize = HeaderReader.ReadUInt16();

        if(NameSize == 0 || NameSize > 520)
          throw new Exception(Name, new Exception(this.GetSizeString(NameSize)));

        this.AddItem(2, "UINT16", Name, this.GetSizeString(NameSize));
      } else {
        NameSize = HeaderReader.ReadByte();

        if(NameSize == 0)
          throw new Exception(Name, new Exception(this.GetSizeString(NameSize)));

        this.AddItem(1, "BYTE", Name, this.GetSizeString(NameSize));
      }
      return NameSize;
    }

    private String ReadName(String Name, Int32 Size, BinaryReader HeaderReader, Byte HeaderFlags) {
      String Text = String.Empty;

      if((HeaderFlags & 0x20) == 0x20) {
        if(Size / 2 * 2 != Size || Size > 520)
          throw new Exception(Name, new Exception(this.GetSizeString(Size)));

        for(Int32 i = 0; i < Size / 2; i++)
          Text += (Char) HeaderReader.ReadUInt16();

        this.AddItem(Size, "UINT16[]", Name, Text);
      } else {
        if(Size > 260)
          throw new Exception(Name, new Exception(this.GetSizeString(Size)));

        for(Int32 i = 0; i < Size; i++)
          Text += (Char) HeaderReader.ReadByte();

        this.AddItem(Size, "BYTE[]", Name, Text);
      }
      return Text;
    }


    private Int32 ReadFolderIndex(BinaryReader HeaderReader, Byte HeaderFlags) {
      Int32 FolderIndex = 0;

      if((HeaderFlags & 0x40) == 0x40) {
        FolderIndex = HeaderReader.ReadInt32();

        if(FolderIndex < -1 || FolderIndex >= this.FolderCount)
          throw new Exception("FolderIndex", new Exception(FolderIndex.ToString()));

        this.AddItem(4, "INT32", "FolderIndex", FolderIndex.ToString());
      } else {
        FolderIndex = HeaderReader.ReadSByte();

        if(FolderIndex < -1 || FolderIndex >= this.FolderCount)
          throw new Exception("FolderIndex", new Exception(FolderIndex.ToString()));

        this.AddItem(1, "INT8", "FolderIndex", FolderIndex.ToString());
      }
      return FolderIndex;
    }

    private void ReadFolder(BinaryReader HeaderReader, Byte HeaderFlags) {
      Int32 FolderIndex = this.ReadFolderIndex(HeaderReader, HeaderFlags);

      Int32 FolderNameSize = this.ReadNameSize("FolderNameSize", HeaderReader, HeaderFlags);

      String FolderName = this.ReadName("FolderName", FolderNameSize, HeaderReader, HeaderFlags);

      this.FolderCount++;
    }


    private UInt64 ReadFileSize(BinaryReader HeaderReader, Byte HeaderFlags) {
      UInt64 FileSize = 0;

      switch(HeaderFlags & 0x03) {
        case 0: {
          FileSize = HeaderReader.ReadByte();

          this.AddItem(1, "BYTE", "FileSize", this.GetSizeString((Int64) FileSize));
          break;
        }
        case 1: {
          FileSize = HeaderReader.ReadUInt16();

          this.AddItem(2, "UINT16", "FileSize", this.GetSizeString((Int64) FileSize));
          break;
        }
        case 2: {
          FileSize = HeaderReader.ReadUInt32();

          this.AddItem(4, "UINT32", "FileSize", this.GetSizeString((Int64) FileSize));
          break;
        }
        case 3: {
          FileSize = HeaderReader.ReadUInt64();

          this.AddItem(8, "UINT64", "FileSize", this.GetSizeString((Int64) FileSize));
          break;
        }
      }
      return FileSize;
    }

    private void ReadFileMemory(String FileName, UInt64 FileSize) {
      this.AddItem((Int64) FileSize, "MEMORY", "FileMemory", "", new FileViewerObject(FileViewerObjectType.ProtectedFile, new FileViewerProtectedFileObject(FileName, this.Position, (Int64) FileSize, this)));

      this.Position += (Int64) FileSize;

      if(this.Position > this.Size) throw new Exception("Wrong FileSize");

      this.Stream.Position = this.Position;
    }

    private void ReadFile(BinaryReader HeaderReader, Byte HeaderFlags) {
      Int32 FolderIndex = this.ReadFolderIndex(HeaderReader, HeaderFlags);

      if(FolderIndex == -1)
        throw new Exception("FolderIndex", new Exception(FolderIndex.ToString()));

      UInt64 FileSize = this.ReadFileSize(HeaderReader, HeaderFlags);

      Int32 FileNameSize = this.ReadNameSize("FileNameSize", HeaderReader, HeaderFlags);

      String FileName = this.ReadName("FileName", FileNameSize, HeaderReader, HeaderFlags);

      if(this.Position + (Int64) FileSize > this.Size)
        throw new Exception("FileSize", new Exception(this.GetSizeString((Int64) FileSize)));

      this.ReadFileMemory(FileName, FileSize);
    }


    private void MakeImageMemory(String Name, Byte[] ByteArray) {
      Int32 ImageSize = (Int32) Math.Pow(ByteArray.Length, 0.5);

      Bitmap Image = new Bitmap(ImageSize, ImageSize, PixelFormat.Format24bppRgb);

      Int32 Index = 0;

      for(Int32 Y = 0; Y < ImageSize; Y++) {
        for(Int32 X = 0; X < ImageSize; X++)
          Image.SetPixel(X, Y, Color.FromArgb(ByteArray[Index++], 0, 0));
      }

      MemoryStream Stream = new MemoryStream();
      Image.Save(Stream, ImageFormat.Png);

      this.AddItem(0, "MEMORY", Name, "PNG: " + this.GetSizeString(Stream.Length), new FileViewerObject(Image));
    }

    private void MakePassArray(UInt16[] PassLetterArray) {
      if(PassLetterArray == null || PassLetterArray.Length == 0)
        this.PassArray = new Byte[0];
      else {
        this.PassArray = new Byte[515];

        Byte PassSpace = 0;
        Int32 PassIndex = 0;

        for(Int32 i = 0; i <= 513; i++) {
          PassSpace += (Byte) PassLetterArray[PassIndex];

          if(PassLetterArray[PassIndex] > 255) {
            this.PassArray[i++] = (Byte) PassLetterArray[PassIndex];
            this.PassArray[i] = (Byte) (PassLetterArray[PassIndex++] / 0x0100);
          } else {
            this.PassArray[i] = (Byte) PassLetterArray[PassIndex++];
          }

          if(PassIndex == PassLetterArray.Length) {
            this.PassArray[++i] = PassSpace;
            PassIndex = 0;
          }
        }
      }
    }


    private void ReadBegin() {
      UInt32 IDNumber = this.ReadIDNumber();

      Boolean UseFileIdentifier = IDNumber == 0x54414450 ? true : false;

      UInt32 LoopMax = this.ReadLoopMax(UseFileIdentifier);

      Byte PassVersion = this.ReadPassVersion(UseFileIdentifier);

      Boolean UsePassword = this.PassArray.Length > 0 ? true : false;

      if(UsePassword && UseFileIdentifier && (PassVersion & 0x80) != 0x80)
        UsePassword = false;

      Byte FileCount = this.ReadFileCount(UseFileIdentifier);
      Byte FactorCount = this.ReadFactorCount(UsePassword);

      Byte[] FactorMemory = this.ReadFactorMemory(UsePassword, FactorCount);
      Byte[] ReferenceMemory = this.ReadReferenceMemory(UsePassword, FactorCount);

      this.MakeCodeValueIndex(FactorCount, LoopMax, FactorMemory, ReferenceMemory);

      this.ValueMemorySize = this.ReadCodeValueCount(FactorCount, FactorMemory, ReferenceMemory);
      this.PageMemorySize = this.ReadCodePageCount(FactorCount, FactorMemory, ReferenceMemory);

      this.CodeValueMemory = this.MakeCodeMemory(FactorCount, FactorMemory, ReferenceMemory, this.ValueMemorySize);

      this.MakeImageMemory("CodeValueMemory", this.CodeValueMemory);

      if(this.PageMemorySize > 0) {
        this.CodePageMemory = this.MakeCodeMemory(FactorCount, FactorMemory, ReferenceMemory, this.PageMemorySize);

        this.MakeImageMemory("CodePageMemory", this.CodePageMemory);
      }

      this.FolderCount = 0;

      while(true) {
        UInt16 HeaderSize = this.ReadHeaderSize();

        BinaryReader HeaderReader = this.ReadMemory(HeaderSize);

        Byte HeaderFlags = this.ReadHeaderFlags(HeaderReader);

        if((HeaderFlags & 0x80) == 0x80)
          this.ReadFolder(HeaderReader, HeaderFlags);
        else
          this.ReadFile(HeaderReader, HeaderFlags);

        if(this.Position == this.Size) break;
      }

      this.AddItem(0, "MEMORY", "ProtectedMemory", "", new FileViewerObject(FileViewerObjectType.ProtectedMemory, this));
    }

    #endregion


    #region Public Methods

    public List<ListViewItem> Run(IFileViewerProgressBar Parent, String FilePath, UInt16[] PassLetterArray) {
      this.Parent = Parent;
      this.FilePath = FilePath;

      this.MakePassArray(PassLetterArray);

      try {
        this.Stream = new FileStream(FilePath, FileMode.Open, FileAccess.Read);
        this.Reader = new BinaryReader(this.Stream);

        this.Size = this.Stream.Length;

        this.Parent.BeginProgressBar(this.Size);

        this.AddItemBeginRead();

        this.ReadBegin();
      } catch(Exception e) {
        this.AddItemError(e);

        this.Close();
        return this.ItemList;
      }

      this.AddItemEndOfFile();

      this.Parent.UpdateProgressBar(this.Size);

      this.Close();
      return this.ItemList;
    }


    public Boolean SaveFile(String Name, Int64 Position, Int64 Size) {
      FileStream Writer = null;

      try {
        if(Size > 0) {
          this.Stream = new FileStream(this.FilePath, FileMode.Open, FileAccess.Read);
          this.Stream.Position = Position;

          this.Size = this.Stream.Length;
          this.Position = Position;
        }

        Writer = new FileStream(Name, FileMode.Create, FileAccess.ReadWrite);

        if(Size > 0) {
          this.Parent.BeginProgressBar(Size);
          this.Parent.UpdateProgressBar(0);

          Byte[] ByteArray = new Byte[66000];

          while(Size > 0) {
            Int32 Count = 66000;

            if(Count > Size) Count = (Int32) Size;

            this.ReadMemory(ByteArray, (UInt32) Count);

            this.Position += Count;

            Writer.Write(ByteArray, 0, Count);

            this.Parent.UpdateProgressBar(Writer.Position);

            Size -= Count;
          }
        }
      } catch(Exception e) {
        this.Close();

        if(Writer != null)
          Writer.Close();

        MessageBox.Show(e.Message, "  Error: Save File", MessageBoxButtons.OK, MessageBoxIcon.Error);
        return false;
      }

      this.Close();

      if(Writer != null) {
        Writer.Flush();
        Writer.Close();
      }
      return true;
    }


    public Byte[] CreateMemory(UInt64 Position, UInt32 Size) {
      this.Position = (Int64) Position;

      return this.CreateMemory(Size);
    }

    #endregion
  }


  public class FileViewerTDFormatTextObject {
    #region Public Fields

    public String Title;

    public UInt16[] LetterArray;
    public Byte[] LetterInfoArray;
    public Int16[] LetterSizeArray;
    public UInt16[][] FontLetterArray;

    #endregion

    #region Contructor

    public FileViewerTDFormatTextObject(String Title, UInt16[] LetterArray, Byte[] LetterInfoArray, Int16[] LetterSizeArray) {
      this.Title = Title;
      this.LetterArray = LetterArray;
      this.LetterInfoArray = LetterInfoArray;
      this.LetterSizeArray = LetterSizeArray;
    }

    public FileViewerTDFormatTextObject(String Title, Byte[] LetterInfoArray, UInt16[][] FontLetterArray) {
      this.Title = Title;
      this.LetterInfoArray = LetterInfoArray;
      this.FontLetterArray = FontLetterArray;
    }

    #endregion
  }

  public class FileViewerTDFormatText {
    #region Public Fields

    public IFileViewerProgressBar Parent;
    public List<ListViewItem> ItemList;
    public BinaryReader Reader;
    public FileStream Stream;
    public String FilePath;
    public Int64 Position;
    public Int64 Size;

    #endregion


    #region Contructor

    public FileViewerTDFormatText() {
      this.ItemList = new List<ListViewItem>();
    }

    #endregion


    #region Private Methods

    private void AddItem(Int64 Size, String Type, String Name, String Value) {
      this.ItemList.Add(new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value }));

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }

    private void AddItem(Int64 Size, String Type, String Name, String Value, FileViewerObject Tag) {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value });

      Item.BackColor = Color.FromArgb(240, 255, 245);
      Item.Tag = Tag;

      this.ItemList.Add(Item);

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }


    private void AddItemError(String Value) {
      ListViewItem Item = new ListViewItem(new String[] { "".PadRight(10), "".PadRight(10), "".PadRight(10), "Error".PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }

    private void AddItemError(Exception e) {
      String Name = String.Empty;
      String Value = String.Empty;

      if(e.InnerException != null && e.InnerException.Message != null) {
        if(e.Message != null)
          Name = e.Message;

        Value = e.InnerException.Message;
      } else {
        if(e.Message != null)
          Value = e.Message;
      }

      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "ERROR".PadRight(10), Name.PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }


    private void AddItemBeginRead() {
      ListViewItem Item = new ListViewItem(new String[] { "0".PadRight(10), "0".PadRight(10), "".PadRight(10), "BeginRead".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }

    private void AddItemEndOfFile() {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "".PadRight(10), "EndOfFile".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }


    private void Close() {
      if(this.Stream != null) {
        this.Stream.Close();
        this.Stream = null;
      }

      if(this.Reader != null) {
        this.Reader.Close();
        this.Reader = null;
      }
    }

    #endregion

    #region Private Methods: Format

    private UInt32 ReadIDNumber() {
      UInt32 IDNumber = this.Reader.ReadUInt32();

      if(IDNumber != 0x57544454) //TDTW
        throw new Exception("IDNumber", new Exception("0x" + IDNumber.ToString("X8")));

      this.AddItem(4, "UINT32", "IDNumber", "0x" + IDNumber.ToString("X8") + " hex");
      return IDNumber;
    }

    private Byte ReadVersion() {
      Byte Version = this.Reader.ReadByte();

      if(Version == 0 || Version > 2)
        throw new Exception("Version", new Exception(Version.ToString()));

      this.AddItem(1, "BYTE", "Version", Version.ToString());
      return Version;
    }

    private Byte ReadAlignment() {
      Byte Alignment = this.Reader.ReadByte();

      //Left = 0, Center = 1, Right = 2, Justified = 3

      if(Alignment > 3)
        throw new Exception("Alignment", new Exception(Alignment.ToString()));

      this.AddItem(1, "BYTE", "Alignment", Alignment.ToString());
      return Alignment;
    }

    private Byte ReadFlags() {
      Byte Flags = this.Reader.ReadByte();

      //Separator = 0x01, TextWidth = 0x02, Compress = 0x04, Thumbnail = 0x08

      if(Flags > 0x0F)
        throw new Exception("Flags", new Exception("0x" + Flags.ToString("X2")));

      this.AddItem(1, "BYTE", "Flags", "0x" + Flags.ToString("X2"));
      return Flags;
    }


    private Int32 ReadThumbnailSize() {
      Int32 ThumbnailSize = this.Reader.ReadInt32();

      if(ThumbnailSize <= 0 || ThumbnailSize > this.Size - this.Position - 4)
        throw new Exception("ThumbnailSize", new Exception(ThumbnailSize.ToString()));

      this.AddItem(4, "INT32", "ThumbnailSize", ThumbnailSize.ToString("#,#") + " bytes");
      return ThumbnailSize;
    }

    private void ReadThumbnailImage(Int32 ThumbnailSize) {
      try {
        Byte[] ThumbnailImageArray = this.Reader.ReadBytes(ThumbnailSize);

        Bitmap ThumbnailImage = ThumbnailImage = new Bitmap(new MemoryStream(ThumbnailImageArray));

        this.AddItem(ThumbnailSize, "MEMORY", "ThumbnailImage", "", new FileViewerObject(ThumbnailImage));
      } catch(Exception e) {
        throw new Exception("ThumbnailImage", e);
      }
    }


    private Char ReadSeparator() {
      Char Separator = (Char) this.Reader.ReadUInt16();

      if(Char.IsControl(Separator))
        throw new Exception("Separator", new Exception("0x" + ((UInt16) Separator).ToString("X4")));

      this.AddItem(2, "WCHAR", "Separator", "0x" + ((UInt16) Separator).ToString("X4"));
      return Separator;
    }

    private Int32 ReadTextWidth() {
      Int32 TextWidth = this.Reader.ReadInt32();

      if(TextWidth < 0 || TextWidth > 100000)
        throw new Exception("TextWidth", new Exception(TextWidth.ToString()));

      this.AddItem(4, "INT32", "TextWidth", TextWidth + " pixel");
      return TextWidth;
    }

    private Double ReadDocumentFactor() {
      Double DocumentFactor = this.Reader.ReadDouble();

      if(DocumentFactor < 1 || DocumentFactor > 4)
        throw new Exception("DocumentFactor", new Exception(DocumentFactor.ToString()));

      this.AddItem(8, "DOUBLE", "DocumentFactor", DocumentFactor.ToString());
      return DocumentFactor;
    }

    private Int32 ReadLetterLength() {
      Int32 LetterLength = this.Reader.ReadInt32();

      if(LetterLength < 0 || LetterLength > this.Size - this.Position - 4)
        throw new Exception("LetterLength", new Exception(LetterLength.ToString()));

      this.AddItem(4, "INT32", "LetterLength", LetterLength.ToString());
      return LetterLength;
    }


    private Byte ReadFontCount() {
      Byte FontCount = this.Reader.ReadByte();

      if(FontCount == 0)
        throw new Exception("FontCount", new Exception(FontCount.ToString()));

      this.AddItem(1, "BYTE", "FontCount", FontCount.ToString());
      return FontCount;
    }

    private Int32 ReadFontNameLength() {
      Int32 FontNameLength = this.Reader.ReadInt32();

      if(FontNameLength <= 0)
        throw new Exception("FontNameLength", new Exception(FontNameLength.ToString()));

      this.AddItem(4, "INT32", "FontNameLength", FontNameLength + " letters");
      return FontNameLength;
    }

    private String ReadFontName(Int32 FontNameLength) {
      Char[] LetterArray = new Char[FontNameLength];

      for(Int32 i = 0; i < FontNameLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String FontName = new String(LetterArray);

      if(FontName == null || FontName == String.Empty)
        throw new Exception("FontName", new Exception("Empty"));

      this.AddItem(FontNameLength * 2, "WCHAR[ ]", "FontName", FontName);
      return FontName;
    }

    private FontStyle ReadFontStyle() {
      Byte Style = this.Reader.ReadByte();

      if(Style > 0x0F)
        throw new Exception("FontStyle", new Exception("0x" + Style.ToString("X2")));

      this.AddItem(1, "BYTE", "FontStyle", ((FontStyle) Style).ToString());
      return (FontStyle) Style;
    }

    private Single ReadFontSize() {
      Single FontSize = this.Reader.ReadSingle();

      if(FontSize < 1 || FontSize > 100000)
        throw new Exception("FontSize", new Exception(FontSize.ToString()));

      this.AddItem(4, "FLOAT", "FontSize", FontSize.ToString());
      return FontSize;
    }


    private Byte ReadColorCount() {
      Byte ColorCount = this.Reader.ReadByte();

      if(ColorCount == 0)
        throw new Exception("ColorCount", new Exception(ColorCount.ToString()));

      this.AddItem(1, "BYTE", "ColorCount", ColorCount.ToString());
      return ColorCount;
    }

    private UInt32 ReadColorValue() {
      UInt32 ColorValue = this.Reader.ReadUInt32();

      this.AddItem(4, "UINT32", "ColorValue", "0x" + ColorValue.ToString("X8") + " hex");
      return ColorValue;
    }


    private Int32 ReadLetterCount() {
      Int32 LetterCount = this.Reader.ReadInt32();

      if(LetterCount <= 0)
        throw new Exception("LetterCount", new Exception(LetterCount.ToString()));

      this.AddItem(4, "INT32", "LetterCount", LetterCount + " letters");
      return LetterCount;
    }

    private UInt16[] ReadLetterMemory(Int32 LetterLength) {
      UInt16[] LetterMemory = new UInt16[LetterLength];

      for(Int32 i = 0; i < LetterLength; i++)
        LetterMemory[i] = this.Reader.ReadUInt16();

      this.AddItem(LetterLength * 2, "UINT16[ ]", "LetterMemory", (LetterLength * 2).ToString("#,#") + " bytes", new FileViewerObject(FileViewerObjectType.FormatText, new FileViewerTDFormatTextObject("Letter", LetterMemory, null, null)));
      return LetterMemory;
    }

    private Byte[] ReadLetterInfoMemoryV1(UInt16[] LetterArray) {
      Int32 ByteSize = LetterArray.Length * 3;
      Byte[] LetterInfoMemory = this.Reader.ReadBytes(ByteSize);

      this.AddItem(ByteSize, "BYTE[ ]", "LetterInfoMemory", ByteSize.ToString("#,#") + " bytes", new FileViewerObject(FileViewerObjectType.FormatText, new FileViewerTDFormatTextObject("LetterInfo", LetterArray, LetterInfoMemory, null)));
      return LetterInfoMemory;
    }

    private Byte[] ReadLetterInfoMemoryV2(Int32 LetterLength, UInt16[][] FontLetterArray) {
      Int32 ByteSize = LetterLength * 5;
      Byte[] LetterInfoMemory = this.Reader.ReadBytes(ByteSize);

      this.AddItem(ByteSize, "BYTE[ ]", "LetterInfoMemory", ByteSize.ToString("#,#") + " bytes", new FileViewerObject(FileViewerObjectType.FormatText, new FileViewerTDFormatTextObject("LetterInfo", LetterInfoMemory, FontLetterArray)));
      return LetterInfoMemory;
    }

    private Int16[] ReadLetterSizeMemory(UInt16[] LetterArray, Byte[] LetterInfoArray) {
      Int32 LetterLength = LetterArray.Length;
      Int16[] LetterSizeMemory = new Int16[LetterLength * 3];

      for(Int32 i = 0; i < LetterLength * 3; i++)
        LetterSizeMemory[i] = this.Reader.ReadInt16();

      this.AddItem(LetterLength * 6, "INT16[ ]", "LetterSizeMemory", (LetterLength * 6).ToString("#,#") + " bytes", new FileViewerObject(FileViewerObjectType.FormatText, new FileViewerTDFormatTextObject("LetterSize", LetterArray, LetterInfoArray, LetterSizeMemory)));
      return LetterSizeMemory;
    }


    private void ReadDataV1() {
      Byte Alignment = this.ReadAlignment();
      Byte Flags = this.ReadFlags();

      //CompressText
      if((Flags & 0x04) == 0x04)
        throw new Exception("Bad Format");

      //Thumbnail
      if((Flags & 0x08) == 0x08) {
        Int32 ThumbnailSize = this.ReadThumbnailSize();

        this.ReadThumbnailImage(ThumbnailSize);
      }

      Char Separator = '-';

      //Separator
      if((Flags & 0x01) == 0x01)
        Separator = this.ReadSeparator();

      Int32 TextWidth = 0;

      //TextWidth
      if((Flags & 0x02) == 0x02)
        TextWidth = this.ReadTextWidth();

      Int32 LetterLength = this.ReadLetterLength();

      if(LetterLength == 0) {
        if(this.Position != this.Size) throw new Exception("Wrong FileSize");
        return;
      }

      Byte FontCount = this.ReadFontCount();

      for(Int32 i = 0; i < FontCount; i++) {
        Int32 FontNameLength = this.ReadFontNameLength();
        String FontName = this.ReadFontName(FontNameLength);

        //Style: Regular, Bold, Italic, Underline, Strikeout
        FontStyle FontStyle = this.ReadFontStyle();

        //GraphicsUnit: Pixel
        Single FontSize = this.ReadFontSize();
      }

      Byte ColorCount = this.ReadColorCount();

      for(Int32 i = 0; i < ColorCount; i++) {
        UInt32 Color = this.ReadColorValue();
      }

      UInt16[] LetterArray = this.ReadLetterMemory(LetterLength);

      //Info(0): 0x00 Letter, 0x01 Seperator, 0x02 Space, 0x03 Control, 0x04 NewLine (Flag)
      //Info(1): Font index
      //Info(2): Color index
      Byte[] LetterInfoArray = this.ReadLetterInfoMemoryV1(LetterArray);

      //Size(0): Width A
      //Size(1): Width B
      //Size(2): Width C
      Int16[] LetterSizeArray = this.ReadLetterSizeMemory(LetterArray, LetterInfoArray);

      if(this.Position != this.Size) throw new Exception("Wrong FileSize");
    }

    private void ReadDataV2() {
      Byte Alignment = this.ReadAlignment();
      Byte Flags = this.ReadFlags();

      //CompressText
      if((Flags & 0x04) == 0x04)
        throw new Exception("Bad Format");

      //Thumbnail
      if((Flags & 0x08) == 0x08) {
        Int32 ThumbnailSize = this.ReadThumbnailSize();

        this.ReadThumbnailImage(ThumbnailSize);
      }

      Char Separator = '-';

      //Separator
      if((Flags & 0x01) == 0x01)
        Separator = this.ReadSeparator();

      Int32 TextWidth = 0;

      //TextWidth
      if((Flags & 0x02) == 0x02)
        TextWidth = this.ReadTextWidth();

      //DocumentFactor
      Double DocumentFactor = this.ReadDocumentFactor();

      Int32 LetterLength = this.ReadLetterLength();

      if(LetterLength == 0) {
        if(this.Position != this.Size) throw new Exception("Wrong FileSize");
        return;
      }

      List<UInt16[]> LetterList = new List<UInt16[]>();

      Byte FontCount = this.ReadFontCount();

      for(Int32 i = 0; i < FontCount; i++) {
        Int32 FontNameLength = this.ReadFontNameLength();
        String FontName = this.ReadFontName(FontNameLength);

        //Style: Regular, Bold, Italic, Underline, Strikeout
        FontStyle FontStyle = this.ReadFontStyle();

        //GraphicsUnit: Pixel
        Single FontSize = this.ReadFontSize();

        //Character
        Int32 LetterCount = this.ReadLetterCount();
        UInt16[] LetterArray = this.ReadLetterMemory(LetterCount);

        LetterList.Add(LetterArray);
      }

      Byte ColorCount = this.ReadColorCount();

      for(Int32 i = 0; i < ColorCount; i++) {
        UInt32 Color = this.ReadColorValue();
      }

      //Info(0): 0x00 Letter, 0x01 Seperator, 0x02 Space, 0x03 Control, 0x04 NewLine (Flag)
      //Info(1): Font index
      //Info(2): Color index
      //Info(3): Letter index
      Byte[] LetterInfoArray = this.ReadLetterInfoMemoryV2(LetterLength, LetterList.ToArray());

      if(this.Position != this.Size) throw new Exception("Wrong FileSize");
    }


    private void ReadBegin() {
      UInt32 IDNumber = this.ReadIDNumber();

      Byte Version = this.ReadVersion();

      switch(Version) {
        case 1: {
          this.ReadDataV1();
          return;
        }
        case 2: {
          this.ReadDataV2();
          return;
        }
      }
    }

    #endregion


    #region Public Methods

    public List<ListViewItem> Run(IFileViewerProgressBar Parent, String FilePath) {
      this.Parent = Parent;

      try {
        this.Stream = new FileStream(FilePath, FileMode.Open, FileAccess.Read);
        this.Reader = new BinaryReader(this.Stream);

        this.Size = this.Stream.Length;

        this.Parent.BeginProgressBar(this.Size);

        this.AddItemBeginRead();

        this.ReadBegin();
      } catch(Exception e) {
        this.AddItemError(e);

        this.Close();
        return this.ItemList;
      }

      this.AddItemEndOfFile();

      this.Close();
      return this.ItemList;
    }

    #endregion
  }


  public enum FileViewerTDCompressTextType {
    Info = 0,
    Font = 1,
    Color = 2,
    Index = 3,
  }

  public class FileViewerTDCompressTextObject {
    #region Public Fields

    public Boolean UseInfo2Bit;

    public FileViewerTDCompressTextType Type;

    public Byte[] ValueArray;
    public Int32 ValueMax;

    public String Title;

    public UInt16[] LetterArray;
    public Int16[] LetterSizeArray;
    public Int32 LetterLength;

    public List<UInt16[]> LetterList;
    public Byte[] FontArray;
    public Int32 FontCount;

    #endregion

    #region Contructor

    public FileViewerTDCompressTextObject(String Title, UInt16[] LetterArray) {
      this.Title = Title;
      this.LetterArray = LetterArray;
    }

    public FileViewerTDCompressTextObject(String Title, UInt16[] LetterArray, Int16[] LetterSizeArray) {
      this.Title = Title;
      this.LetterArray = LetterArray;
      this.LetterSizeArray = LetterSizeArray;
    }

    public FileViewerTDCompressTextObject(String Title, FileViewerTDCompressTextType Type, Byte[] ValueArray, Int32 ValueMax, Int32 LetterLength) {
      this.Title = Title;
      this.Type = Type;
      this.ValueArray = ValueArray;
      this.ValueMax = ValueMax;
      this.LetterLength = LetterLength;
    }

    public FileViewerTDCompressTextObject(String Title, FileViewerTDCompressTextType Type, Byte[] ValueArray, Int32 ValueMax, Int32 LetterLength, List<UInt16[]> LetterList, Byte[] FontArray, Int32 FontCount) {
      this.Title = Title;
      this.Type = Type;
      this.ValueArray = ValueArray;
      this.ValueMax = ValueMax;
      this.LetterLength = LetterLength;
      this.LetterList = LetterList;
      this.FontArray = FontArray;
      this.FontCount = FontCount;
    }

    public FileViewerTDCompressTextObject(Boolean UseInfo2Bit, String Title, FileViewerTDCompressTextType Type, Byte[] ValueArray, Int32 ValueMax, Int32 LetterLength) {
      this.UseInfo2Bit = UseInfo2Bit;
      this.Title = Title;
      this.Type = Type;
      this.ValueArray = ValueArray;
      this.ValueMax = ValueMax;
      this.LetterLength = LetterLength;
    }

    #endregion
  }

  public class FileViewerTDCompressText {
    #region Public Fields

    public IFileViewerProgressBar Parent;
    public List<ListViewItem> ItemList;
    public BinaryReader Reader;
    public FileStream Stream;
    public String FilePath;
    public Int64 Position;
    public Int64 Size;

    #endregion


    #region Contructor

    public FileViewerTDCompressText() {
      this.ItemList = new List<ListViewItem>();
    }

    #endregion


    #region Private Methods

    private void AddItem(Int64 Size, String Type, String Name, String Value) {
      this.ItemList.Add(new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value }));

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }

    private void AddItem(Int64 Size, String Type, String Name, String Value, FileViewerObject Tag) {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value });

      Item.BackColor = Color.FromArgb(240, 255, 245);
      Item.Tag = Tag;

      this.ItemList.Add(Item);

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }


    private void AddItemError(String Value) {
      ListViewItem Item = new ListViewItem(new String[] { "".PadRight(10), "".PadRight(10), "".PadRight(10), "Error".PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }

    private void AddItemError(Exception e) {
      String Name = String.Empty;
      String Value = String.Empty;

      if(e.InnerException != null && e.InnerException.Message != null) {
        if(e.Message != null)
          Name = e.Message;

        Value = e.InnerException.Message;
      } else {
        if(e.Message != null)
          Value = e.Message;
      }

      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "ERROR".PadRight(10), Name.PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }


    private void AddItemBeginRead() {
      ListViewItem Item = new ListViewItem(new String[] { "0".PadRight(10), "0".PadRight(10), "".PadRight(10), "BeginRead".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }

    private void AddItemEndOfFile() {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "".PadRight(10), "EndOfFile".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }


    private void Close() {
      if(this.Stream != null) {
        this.Stream.Close();
        this.Stream = null;
      }

      if(this.Reader != null) {
        this.Reader.Close();
        this.Reader = null;
      }
    }

    #endregion

    #region Private Methods: Format

    private UInt32 ReadIDNumber() {
      UInt32 IDNumber = this.Reader.ReadUInt32();

      if(IDNumber != 0x57544454) //TDTW
        throw new Exception("IDNumber", new Exception("0x" + IDNumber.ToString("X8")));

      this.AddItem(4, "UINT32", "IDNumber", "0x" + IDNumber.ToString("X8") + " hex");
      return IDNumber;
    }

    private Byte ReadVersion() {
      Byte Version = this.Reader.ReadByte();

      if(Version == 0 || Version > 2)
        throw new Exception("Version", new Exception(Version.ToString()));

      this.AddItem(1, "BYTE", "Version", Version.ToString());
      return Version;
    }

    private Byte ReadAlignment() {
      Byte Alignment = this.Reader.ReadByte();

      //Left = 0, Center = 1, Right = 2, Justified = 3

      if(Alignment > 3)
        throw new Exception("Alignment", new Exception(Alignment.ToString()));

      this.AddItem(1, "BYTE", "Alignment", Alignment.ToString());
      return Alignment;
    }

    private Byte ReadFlags() {
      Byte Flags = this.Reader.ReadByte();

      //Separator = 0x01, TextWidth = 0x02, Compress = 0x04, Thumbnail = 0x08

      if(Flags > 0x0F)
        throw new Exception("Flags", new Exception("0x" + Flags.ToString("X2")));

      this.AddItem(1, "BYTE", "Flags", "0x" + Flags.ToString("X2"));
      return Flags;
    }


    private Int32 ReadThumbnailSize() {
      Int32 ThumbnailSize = this.Reader.ReadInt32();

      if(ThumbnailSize <= 0 || ThumbnailSize > this.Size - this.Position - 4)
        throw new Exception("ThumbnailSize", new Exception(ThumbnailSize.ToString()));

      this.AddItem(4, "INT32", "ThumbnailSize", ThumbnailSize.ToString("#,#") + " bytes");
      return ThumbnailSize;
    }

    private void ReadThumbnailImage(Int32 ThumbnailSize) {
      try {
        Byte[] ThumbnailImageArray = this.Reader.ReadBytes(ThumbnailSize);

        Bitmap ThumbnailImage = ThumbnailImage = new Bitmap(new MemoryStream(ThumbnailImageArray));

        this.AddItem(ThumbnailSize, "MEMORY", "ThumbnailImage", "", new FileViewerObject(ThumbnailImage));
      } catch(Exception e) {
        throw new Exception("ThumbnailImage", e);
      }
    }


    private Char ReadSeparator() {
      Char Separator = (Char) this.Reader.ReadUInt16();

      if(Char.IsControl(Separator))
        throw new Exception("Separator", new Exception("0x" + ((UInt16) Separator).ToString("X4")));

      this.AddItem(2, "WCHAR", "Separator", "0x" + ((UInt16) Separator).ToString("X4"));
      return Separator;
    }

    private Int32 ReadTextWidth() {
      Int32 TextWidth = this.Reader.ReadInt32();

      if(TextWidth < 0 || TextWidth > 100000)
        throw new Exception("TextWidth", new Exception(TextWidth.ToString()));

      this.AddItem(4, "INT32", "TextWidth", TextWidth + " pixel");
      return TextWidth;
    }

    private Double ReadDocumentFactor() {
      Double DocumentFactor = this.Reader.ReadDouble();

      if(DocumentFactor < 1 || DocumentFactor > 4)
        throw new Exception("DocumentFactor", new Exception(DocumentFactor.ToString()));

      this.AddItem(8, "DOUBLE", "DocumentFactor", DocumentFactor.ToString());
      return DocumentFactor;
    }

    private Int32 ReadLetterLength() {
      Int32 LetterLength = this.Reader.ReadInt32();

      if(LetterLength < 0 || LetterLength > this.Size - this.Position - 4)
        throw new Exception("LetterLength", new Exception(LetterLength.ToString()));

      this.AddItem(4, "INT32", "LetterLength", LetterLength.ToString());
      return LetterLength;
    }


    private Byte ReadFontCount() {
      Byte FontCount = this.Reader.ReadByte();

      if(FontCount == 0)
        throw new Exception("FontCount", new Exception(FontCount.ToString()));

      this.AddItem(1, "BYTE", "FontCount", FontCount.ToString());
      return FontCount;
    }

    private Int32 ReadFontNameLength() {
      Int32 FontNameLength = this.Reader.ReadInt32();

      if(FontNameLength <= 0)
        throw new Exception("FontNameLength", new Exception(FontNameLength.ToString()));

      this.AddItem(4, "INT32", "FontNameLength", FontNameLength + " letters");
      return FontNameLength;
    }

    private String ReadFontName(Int32 FontNameLength) {
      Char[] LetterArray = new Char[FontNameLength];

      for(Int32 i = 0; i < FontNameLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String FontName = new String(LetterArray);

      if(FontName == null || FontName == String.Empty)
        throw new Exception("FontName", new Exception("Empty"));

      this.AddItem(FontNameLength * 2, "WCHAR[ ]", "FontName", FontName);
      return FontName;
    }

    private FontStyle ReadFontStyle() {
      Byte Style = this.Reader.ReadByte();

      if(Style > 0x0F)
        throw new Exception("FontStyle", new Exception("0x" + Style.ToString("X2")));

      this.AddItem(1, "BYTE", "FontStyle", ((FontStyle) Style).ToString());
      return (FontStyle) Style;
    }

    private Single ReadFontSize() {
      Single FontSize = this.Reader.ReadSingle();

      if(FontSize < 1 || FontSize > 100000)
        throw new Exception("FontSize", new Exception(FontSize.ToString()));

      this.AddItem(4, "FLOAT", "FontSize", FontSize.ToString());
      return FontSize;
    }


    private Int32 ReadLetterCount() {
      Int32 LetterCount = this.Reader.ReadInt32();

      if(LetterCount <= 0)
        throw new Exception("LetterCount", new Exception(LetterCount.ToString()));

      this.AddItem(4, "INT32", "LetterCount", LetterCount + " letters");
      return LetterCount;
    }

    private UInt16[] ReadLetterMemory(Int32 LetterCount) {
      UInt16[] LetterMemory = new UInt16[LetterCount];

      for(Int32 i = 0; i < LetterCount; i++)
        LetterMemory[i] = this.Reader.ReadUInt16();

      this.AddItem(LetterCount * 2, "UINT16[ ]", "LetterMemory", (LetterCount * 2).ToString("#,#") + " bytes", new FileViewerObject(FileViewerObjectType.CompressText, new FileViewerTDCompressTextObject("Letter", LetterMemory)));
      return LetterMemory;
    }


    private Int32 ReadLetterSizeCount(Int32 LetterCount) {
      Int32 LetterSizeCount = this.Reader.ReadInt32();

      if(LetterSizeCount <= 0 || LetterSizeCount != LetterCount * 6)
        throw new Exception("LetterSizeCount", new Exception(LetterSizeCount.ToString()));

      this.AddItem(4, "INT32", "LetterSizeCount", LetterSizeCount.ToString());
      return LetterSizeCount;
    }

    private Int16[] ReadLetterSizeMemory(UInt16[] LetterMemory, Int32 LetterSizeCount) {
      Int32 LetterSize = LetterSizeCount / 2;
      Int16[] LetterSizeMemory = new Int16[LetterSize];

      for(Int32 i = 0; i < LetterSize; i++)
        LetterSizeMemory[i] = this.Reader.ReadInt16();

      this.AddItem(LetterSizeCount, "INT16[ ]", "LetterSizeMemory", LetterSizeCount.ToString("#,#") + " bytes", new FileViewerObject(FileViewerObjectType.CompressText, new FileViewerTDCompressTextObject("LetterSize", LetterMemory, LetterSizeMemory)));
      return LetterSizeMemory;
    }


    private Byte ReadColorCount() {
      Byte ColorCount = this.Reader.ReadByte();

      if(ColorCount == 0)
        throw new Exception("ColorCount", new Exception(ColorCount.ToString()));

      this.AddItem(1, "BYTE", "ColorCount", ColorCount.ToString());
      return ColorCount;
    }

    private UInt32 ReadColorValue() {
      UInt32 ColorValue = this.Reader.ReadUInt32();

      this.AddItem(4, "UINT32", "ColorValue", "0x" + ColorValue.ToString("X8") + " hex");
      return ColorValue;
    }


    private Int32 ReadLetterInfoSize() {
      Int32 LetterInfoSize = this.Reader.ReadInt32();

      if(LetterInfoSize <= 0)
        throw new Exception("LetterInfoSize", new Exception(LetterInfoSize.ToString()));

      this.AddItem(4, "INT32", "LetterInfoSize", LetterInfoSize.ToString());
      return LetterInfoSize;
    }

    private Byte[] ReadLetterInfoMemory(Boolean UseInfo2Bit, Int32 LetterInfoSize, Int32 LetterLength) {
      Byte[] LetterInfoMemory = this.Reader.ReadBytes(LetterInfoSize);

      this.AddItem(LetterInfoSize, "BYTE[ ]", "LetterInfoMemory", LetterInfoSize.ToString("#,#") + " bytes", new FileViewerObject(FileViewerObjectType.CompressText, new FileViewerTDCompressTextObject(UseInfo2Bit, "LetterInfo", FileViewerTDCompressTextType.Info, LetterInfoMemory, 8, LetterLength)));
      return LetterInfoMemory;
    }


    private Int32 ReadLetterFontSize() {
      Int32 LetterFontSize = this.Reader.ReadInt32();

      if(LetterFontSize <= 0)
        throw new Exception("LetterFontSize", new Exception(LetterFontSize.ToString()));

      this.AddItem(4, "INT32", "LetterFontSize", LetterFontSize.ToString());
      return LetterFontSize;
    }

    private Byte[] ReadLetterFontMemory(Int32 LetterFontSize, Int32 FontCount, Int32 LetterLength) {
      Byte[] LetterFontMemory = this.Reader.ReadBytes(LetterFontSize);

      this.AddItem(LetterFontSize, "BYTE[ ]", "LetterFontMemory", LetterFontSize.ToString("#,#") + " bytes", new FileViewerObject(FileViewerObjectType.CompressText, new FileViewerTDCompressTextObject(false, "LetterFont", FileViewerTDCompressTextType.Font, LetterFontMemory, FontCount, LetterLength)));
      return LetterFontMemory;
    }


    private Int32 ReadLetterColorSize() {
      Int32 LetterColorSize = this.Reader.ReadInt32();

      if(LetterColorSize <= 0)
        throw new Exception("LetterColorSize", new Exception(LetterColorSize.ToString()));

      this.AddItem(4, "INT32", "LetterColorSize", LetterColorSize.ToString());
      return LetterColorSize;
    }

    private Byte[] ReadLetterColorMemory(Int32 LetterColorSize, Int32 ColorCount, Int32 LetterLength) {
      Byte[] LetterColorMemory = this.Reader.ReadBytes(LetterColorSize);

      this.AddItem(LetterColorSize, "BYTE[ ]", "LetterColorMemory", LetterColorSize.ToString("#,#") + " bytes", new FileViewerObject(FileViewerObjectType.CompressText, new FileViewerTDCompressTextObject(false, "LetterColor", FileViewerTDCompressTextType.Color, LetterColorMemory, ColorCount, LetterLength)));
      return LetterColorMemory;
    }


    private Int32 ReadLetterMaxCount() {
      Int32 LetterMaxCount = this.Reader.ReadInt32();

      if(LetterMaxCount < 1 || LetterMaxCount > 65536)
        throw new Exception("LetterMaxCount", new Exception(LetterMaxCount.ToString()));

      this.AddItem(4, "INT32", "LetterMaxCount", LetterMaxCount.ToString());
      return LetterMaxCount;
    }


    private Int32 ReadLetterIndexSize() {
      Int32 LetterIndexSize = this.Reader.ReadInt32();

      if(LetterIndexSize <= 0)
        throw new Exception("LetterIndexSize", new Exception(LetterIndexSize.ToString()));

      this.AddItem(4, "INT32", "LetterIndexSize", LetterIndexSize.ToString());
      return LetterIndexSize;
    }

    private Byte[] ReadLetterIndexMemory(Int32 LetterIndexSize, Int32 IndexCount, Int32 LetterLength, List<UInt16[]> LetterList, Byte[] FontArray, Int32 FontCount) {
      Byte[] LetterIndexMemory = this.Reader.ReadBytes(LetterIndexSize);

      this.AddItem(LetterIndexSize, "BYTE[ ]", "LetterIndexMemory", LetterIndexSize.ToString("#,#") + " bytes", new FileViewerObject(FileViewerObjectType.CompressText, new FileViewerTDCompressTextObject("LetterIndex", FileViewerTDCompressTextType.Index, LetterIndexMemory, IndexCount, LetterLength, LetterList, FontArray, FontCount)));
      return LetterIndexMemory;
    }


    private void ReadDataV1() {
      Byte Alignment = this.ReadAlignment();
      Byte Flags = this.ReadFlags();

      //CompressText
      if((Flags & 0x04) != 0x04)
        throw new Exception("Bad Format");

      //Thumbnail
      if((Flags & 0x08) == 0x08) {
        Int32 ThumbnailSize = this.ReadThumbnailSize();

        this.ReadThumbnailImage(ThumbnailSize);
      }

      Char Separator = '-';

      //Separator
      if((Flags & 0x01) == 0x01)
        Separator = this.ReadSeparator();

      Int32 TextWidth = 0;

      //TextWidth
      if((Flags & 0x02) == 0x02)
        TextWidth = this.ReadTextWidth();

      Int32 LetterLength = this.ReadLetterLength();

      if(LetterLength == 0) {
        if(this.Position != this.Size) throw new Exception("Wrong FileSize");
        return;
      }

      List<UInt16[]> LetterList = new List<UInt16[]>();

      Byte FontCount = this.ReadFontCount();

      for(Int32 i = 0; i < FontCount; i++) {
        Int32 FontNameLength = this.ReadFontNameLength();
        String FontName = this.ReadFontName(FontNameLength);

        //Style: Regular, Bold, Italic, Underline, Strikeout
        FontStyle FontStyle = this.ReadFontStyle();

        //GraphicsUnit: Pixel
        Single FontSize = this.ReadFontSize();

        //Character
        Int32 LetterCount = this.ReadLetterCount();
        UInt16[] LetterArray = this.ReadLetterMemory(LetterCount);

        LetterList.Add(LetterArray);

        //Width a,b,c
        Int32 LetterSizeCount = this.ReadLetterSizeCount(LetterCount);
        Int16[] LetterSizeArray = this.ReadLetterSizeMemory(LetterArray, LetterSizeCount);
      }

      Byte ColorCount = this.ReadColorCount();

      for(Int32 i = 0; i < ColorCount; i++) {
        UInt32 Color = this.ReadColorValue();
      }


      Int32 LetterInfoSize = this.ReadLetterInfoSize();

      //InfoValue 0..7 = 3Bit
      Byte[] LetterInfoMemory = this.ReadLetterInfoMemory(false, LetterInfoSize, LetterLength);

      Byte[] LetterFontMemory = null;

      //FontIndex 0..0 = 0Bit
      if(FontCount > 1) {
        Int32 LetterFontSize = this.ReadLetterFontSize();

        //FontIndex 0..1 = 1Bit
        //FontIndex 0..3 = 2Bit
        //FontIndex 0..7 = 3Bit
        //FontIndex 0..15 = 4Bit
        //FontIndex 0..31 = 5Bit
        //FontIndex 0..63 = 6Bit
        //FontIndex 0..127 = 7Bit
        //FontIndex 0..255 = 8Bit

        LetterFontMemory = this.ReadLetterFontMemory(LetterFontSize, FontCount, LetterLength);
      }

      //ColorIndex 0 = 0Bit
      if(ColorCount > 1) {
        Int32 LetterColorSize = this.ReadLetterColorSize();

        //ColorIndex 0..1 = 1Bit
        //ColorIndex 0..3 = 2Bit
        //ColorIndex 0..7 = 3Bit
        //ColorIndex 0..15 = 4Bit
        //ColorIndex 0..31 = 5Bit
        //ColorIndex 0..63 = 6Bit
        //ColorIndex 0..127 = 7Bit
        //ColorIndex 0..255 = 8Bit

        Byte[] LetterColorMemory = this.ReadLetterColorMemory(LetterColorSize, ColorCount, LetterLength);
      }

      Int32 LetterMaxCount = this.ReadLetterMaxCount();

      //LetterIndex 0 = 0Bit
      if(LetterMaxCount > 1) {
        Int32 LetterIndexSize = this.ReadLetterIndexSize();

        //LetterIndex 0..1 = 1Bit
        //LetterIndex 0..3 = 2Bit
        //LetterIndex 0..7 = 3Bit
        //LetterIndex 0..15 = 4Bit
        //LetterIndex 0..31 = 5Bit
        //LetterIndex 0..63 = 6Bit
        //LetterIndex 0..127 = 7Bit
        //LetterIndex 0..255 = 8Bit

        Byte[] LetterIndexMemory = this.ReadLetterIndexMemory(LetterIndexSize, LetterMaxCount, LetterLength, LetterList, LetterFontMemory, FontCount);
      }

      if(this.Position != this.Size) throw new Exception("Wrong FileSize");
    }

    private void ReadDataV2() {
      Byte Alignment = this.ReadAlignment();
      Byte Flags = this.ReadFlags();

      //CompressText
      if((Flags & 0x04) != 0x04)
        throw new Exception("Bad Format");

      //Thumbnail
      if((Flags & 0x08) == 0x08) {
        Int32 ThumbnailSize = this.ReadThumbnailSize();

        this.ReadThumbnailImage(ThumbnailSize);
      }

      Char Separator = '-';

      //Separator
      if((Flags & 0x01) == 0x01)
        Separator = this.ReadSeparator();

      Int32 TextWidth = 0;

      //TextWidth
      if((Flags & 0x02) == 0x02)
        TextWidth = this.ReadTextWidth();

      //DocumentFactor
      Double DocumentFactor = this.ReadDocumentFactor();

      Int32 LetterLength = this.ReadLetterLength();

      if(LetterLength == 0) {
        if(this.Position != this.Size) throw new Exception("Wrong FileSize");
        return;
      }

      List<UInt16[]> LetterList = new List<UInt16[]>();

      Byte FontCount = this.ReadFontCount();

      for(Int32 i = 0; i < FontCount; i++) {
        Int32 FontNameLength = this.ReadFontNameLength();
        String FontName = this.ReadFontName(FontNameLength);

        //Style: Regular, Bold, Italic, Underline, Strikeout
        FontStyle FontStyle = this.ReadFontStyle();

        //GraphicsUnit: Pixel
        Single FontSize = this.ReadFontSize();

        //Character
        Int32 LetterCount = this.ReadLetterCount();
        UInt16[] LetterArray = this.ReadLetterMemory(LetterCount);

        LetterList.Add(LetterArray);
      }

      Byte ColorCount = this.ReadColorCount();

      for(Int32 i = 0; i < ColorCount; i++) {
        UInt32 Color = this.ReadColorValue();
      }

      Int32 LetterInfoSize = this.ReadLetterInfoSize();

      //InfoValue 0..3 = 2Bit
      Byte[] LetterInfoMemory = this.ReadLetterInfoMemory(true, LetterInfoSize, LetterLength);

      Byte[] LetterFontMemory = null;

      //FontIndex 0..0 = 0Bit
      if(FontCount > 1) {
        Int32 LetterFontSize = this.ReadLetterFontSize();

        //FontIndex 0..1 = 1Bit
        //FontIndex 0..3 = 2Bit
        //FontIndex 0..7 = 3Bit
        //FontIndex 0..15 = 4Bit
        //FontIndex 0..31 = 5Bit
        //FontIndex 0..63 = 6Bit
        //FontIndex 0..127 = 7Bit
        //FontIndex 0..255 = 8Bit

        LetterFontMemory = this.ReadLetterFontMemory(LetterFontSize, FontCount, LetterLength);
      }

      //ColorIndex 0 = 0Bit
      if(ColorCount > 1) {
        Int32 LetterColorSize = this.ReadLetterColorSize();

        //ColorIndex 0..1 = 1Bit
        //ColorIndex 0..3 = 2Bit
        //ColorIndex 0..7 = 3Bit
        //ColorIndex 0..15 = 4Bit
        //ColorIndex 0..31 = 5Bit
        //ColorIndex 0..63 = 6Bit
        //ColorIndex 0..127 = 7Bit
        //ColorIndex 0..255 = 8Bit

        Byte[] LetterColorMemory = this.ReadLetterColorMemory(LetterColorSize, ColorCount, LetterLength);
      }

      Int32 LetterMaxCount = this.ReadLetterMaxCount();

      //LetterIndex 0 = 0Bit
      if(LetterMaxCount > 1) {
        Int32 LetterIndexSize = this.ReadLetterIndexSize();

        //LetterIndex 0..1 = 1Bit
        //LetterIndex 0..3 = 2Bit
        //LetterIndex 0..7 = 3Bit
        //LetterIndex 0..15 = 4Bit
        //LetterIndex 0..31 = 5Bit
        //LetterIndex 0..63 = 6Bit
        //LetterIndex 0..127 = 7Bit
        //LetterIndex 0..255 = 8Bit

        Byte[] LetterIndexMemory = this.ReadLetterIndexMemory(LetterIndexSize, LetterMaxCount, LetterLength, LetterList, LetterFontMemory, FontCount);
      }

      if(this.Position != this.Size) throw new Exception("Wrong FileSize");
    }


    private void ReadBegin() {
      UInt32 IDNumber = this.ReadIDNumber();

      Byte Version = this.ReadVersion();

      switch(Version) {
        case 1: {
          this.ReadDataV1();
          return;
        }
        case 2: {
          this.ReadDataV2();
          return;
        }
      }
    }

    #endregion


    #region Public Methods

    public List<ListViewItem> Run(IFileViewerProgressBar Parent, String FilePath) {
      this.Parent = Parent;

      try {
        this.Stream = new FileStream(FilePath, FileMode.Open, FileAccess.Read);
        this.Reader = new BinaryReader(this.Stream);

        this.Size = this.Stream.Length;

        this.Parent.BeginProgressBar(this.Size);

        this.AddItemBeginRead();

        this.ReadBegin();
      } catch(Exception e) {
        this.AddItemError(e);

        this.Close();
        return this.ItemList;
      }

      this.AddItemEndOfFile();

      this.Close();
      return this.ItemList;
    }

    #endregion
  }


  public class FileViewerTDPicturePaintFileObject {
    #region Public Fields

    public String FileName;

    public Byte[] FileMemory;

    #endregion

    #region Contructor

    public FileViewerTDPicturePaintFileObject(String FileName, Byte[] FileMemory) {
      this.FileName = FileName;
      this.FileMemory = FileMemory;
    }

    #endregion
  }

  public class FileViewerTDPicturePaintCollection {
    #region Public Fields

    public IFileViewerProgressBar Parent;
    public List<ListViewItem> ItemList;
    public BinaryReader Reader;
    public FileStream Stream;
    public String FilePath;
    public Int64 Position;
    public Int64 Size;

    #endregion


    #region Contructor

    public FileViewerTDPicturePaintCollection() {
      this.ItemList = new List<ListViewItem>();
    }

    #endregion


    #region Private Methods

    private void AddItem(Int64 Size, String Type, String Name, String Value) {
      this.ItemList.Add(new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value }));

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }

    private void AddItem(Int64 Size, String Type, String Name, String Value, FileViewerObject Tag) {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value });

      Item.BackColor = Color.FromArgb(240, 255, 245);
      Item.Tag = Tag;

      this.ItemList.Add(Item);

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }


    private void AddItemError(String Value) {
      ListViewItem Item = new ListViewItem(new String[] { "".PadRight(10), "".PadRight(10), "".PadRight(10), "Error".PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }

    private void AddItemError(Exception e) {
      String Name = String.Empty;
      String Value = String.Empty;

      if(e.InnerException != null && e.InnerException.Message != null) {
        if(e.Message != null)
          Name = e.Message;

        Value = e.InnerException.Message;
      } else {
        if(e.Message != null)
          Value = e.Message;
      }

      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "ERROR".PadRight(10), Name.PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }


    private void AddItemBeginRead() {
      ListViewItem Item = new ListViewItem(new String[] { "0".PadRight(10), "0".PadRight(10), "".PadRight(10), "BeginRead".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }

    private void AddItemEndOfFile() {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "".PadRight(10), "EndOfFile".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }


    private void Close() {
      if(this.Stream != null) {
        this.Stream.Close();
        this.Stream = null;
      }

      if(this.Reader != null) {
        this.Reader.Close();
        this.Reader = null;
      }
    }

    #endregion

    #region Private Methods: Format

    private UInt32 ReadIDNumber() {
      UInt32 IDNumber = this.Reader.ReadUInt32();

      if(IDNumber != 0x43504454) //TDPC
        throw new Exception("IDNumber", new Exception("0x" + IDNumber.ToString("X8")));

      this.AddItem(4, "UINT32", "IDNumber", "0x" + IDNumber.ToString("X8") + " hex");
      return IDNumber;
    }

    private Int64 ReadFileSize() {
      Int64 FileSize = this.Reader.ReadInt64();

      if(FileSize != this.Size)
        throw new Exception("FileSize", new Exception(FileSize.ToString()));

      this.AddItem(8, "INT64", "FileSize", FileSize.ToString("#,#") + " bytes");
      return FileSize;
    }

    private Byte ReadVersion() {
      Byte Version = this.Reader.ReadByte();

      if(Version != 1)
        throw new Exception("Version", new Exception(Version.ToString()));

      this.AddItem(1, "BYTE", "Version", Version.ToString());
      return Version;
    }


    private Int32 ReadFileCount() {
      Int32 FileCount = this.Reader.ReadInt32();

      if(FileCount < 0)
        throw new Exception("FileCount", new Exception(FileCount.ToString()));

      this.AddItem(4, "INT32", "FileCount", FileCount.ToString());
      return FileCount;
    }

    private Byte ReadFileType() {
      Byte FileType = this.Reader.ReadByte();

      //Image = 0, Video = 1, Sound = 2, Gif = 3, Animation = 4, Various = 5 

      if(FileType > 5)
        throw new Exception("FileType", new Exception(FileType.ToString()));

      String InfoText = String.Empty;

      switch(FileType) {
        case 0: {
          InfoText = " (Image)";
          break;
        }
        case 1: {
          InfoText = " (Video)";
          break;
        }
        case 2: {
          InfoText = " (Sound)";
          break;
        }
        case 3: {
          InfoText = " (Gif)";
          break;
        }
        case 4: {
          InfoText = " (Animation)";
          break;
        }
        case 5: {
          InfoText = " (Various)";
          break;
        }
      }

      this.AddItem(1, "BYTE", "FileType", FileType + InfoText);
      return FileType;
    }

    private Int32 ReadFileNameLength() {
      Int32 FileNameLength = this.Reader.ReadInt32();

      if(FileNameLength <= 0)
        throw new Exception("FileNameLength", new Exception(FileNameLength.ToString()));

      this.AddItem(4, "INT32", "FileNameLength", FileNameLength + " letters");
      return FileNameLength;
    }

    private String ReadFileName(Int32 FileNameLength) {
      Char[] LetterArray = new Char[FileNameLength];

      for(Int32 i = 0; i < FileNameLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String FileName = new String(LetterArray);

      if(FileName == null || FileName == String.Empty)
        throw new Exception("FileName", new Exception("Empty"));

      this.AddItem(FileNameLength * 2, "WCHAR[ ]", "FileName", FileName);
      return FileName;
    }

    private Int32 ReadFileExtentionLength() {
      Int32 FileExtentionLength = this.Reader.ReadInt32();

      if(FileExtentionLength < 0)
        throw new Exception("FileExtentionLength", new Exception(FileExtentionLength.ToString()));

      this.AddItem(4, "INT32", "FileExtentionLength", FileExtentionLength + " letters");
      return FileExtentionLength;
    }

    private String ReadFileExtention(Int32 FileExtentionLength) {
      if(FileExtentionLength == 0) {
        this.AddItem(0, "WCHAR[ ]", "FileExtention", "Empty");
        return String.Empty;
      }

      Char[] LetterArray = new Char[FileExtentionLength];

      for(Int32 i = 0; i < FileExtentionLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String FileExtention = new String(LetterArray);

      if(FileExtention == null || FileExtention == String.Empty)
        throw new Exception("FileExtention", new Exception("Empty"));

      this.AddItem(FileExtentionLength * 2, "WCHAR[ ]", "FileExtention", FileExtention);
      return FileExtention;
    }


    private Byte ReadImageFormat() {
      Byte ImageFormat = this.Reader.ReadByte();

      //Unknown = 0, Bmp = 1, Emf = 2, Exif = 3,
      //Gif = 4, Icon = 5, Jpeg = 6, Png = 7,
      //Tiff = 8, Wmf = 9, Tdp = 10,

      if(ImageFormat > 10)
        throw new Exception("ImageFormat", new Exception(ImageFormat.ToString()));

      String InfoText = String.Empty;

      switch(ImageFormat) {
        case 0: {
          break;
        }
        case 1: {
          InfoText = " (bmp)";
          break;
        }
        case 2: {
          InfoText = " (emf)";
          break;
        }
        case 3: {
          InfoText = " (exif)";
          break;
        }
        case 4: {
          InfoText = " (gif)";
          break;
        }
        case 5: {
          InfoText = " (icon)";
          break;
        }
        case 6: {
          InfoText = " (jpeg)";
          break;
        }
        case 7: {
          InfoText = " (png)";
          break;
        }
        case 8: {
          InfoText = " (tiff)";
          break;
        }
        case 9: {
          InfoText = " (wmf)";
          break;
        }
        case 10: {
          InfoText = " (tdp)";
          break;
        }
      }

      this.AddItem(1, "BYTE", "ImageFormat", ImageFormat + InfoText);
      return ImageFormat;
    }

    private Int32 ReadImageWidth() {
      Int32 ImageWidth = this.Reader.ReadInt32();

      if(ImageWidth < 0 || ImageWidth > 1000000)
        throw new Exception("ImageWidth", new Exception(ImageWidth.ToString()));

      this.AddItem(4, "INT32", "ImageWidth", ImageWidth + " pixel");
      return ImageWidth;
    }

    private Int32 ReadImageHeight() {
      Int32 ImageHeight = this.Reader.ReadInt32();

      if(ImageHeight < 0 || ImageHeight > 1000000)
        throw new Exception("ImageHeight", new Exception(ImageHeight.ToString()));

      this.AddItem(4, "INT32", "ImageHeight", ImageHeight + " pixel");
      return ImageHeight;
    }


    private Int64 ReadPlayerDuration() {
      Int64 PlayerDuration = this.Reader.ReadInt64();

      if(PlayerDuration < 0)
        throw new Exception("PlayerDuration", new Exception(PlayerDuration.ToString()));

      this.AddItem(8, "INT64", "PlayerDuration", new TimeSpan(PlayerDuration).ToString());
      return PlayerDuration;
    }


    private Int32 ReadFileMemorySize() {
      Int32 FileMemorySize = this.Reader.ReadInt32();

      if(FileMemorySize < 0 || FileMemorySize > this.Size - this.Position - 4)
        throw new Exception("FileMemorySize", new Exception(FileMemorySize.ToString()));

      this.AddItem(4, "INT32", "FileMemorySize", FileMemorySize.ToString("#,#") + " bytes");
      return FileMemorySize;
    }

    private Byte[] ReadFileMemory(Int32 FileMemorySize, String FileName, String FileExtention) {
      Byte[] FileMemory = this.Reader.ReadBytes(FileMemorySize);

      if(FileExtention != null && FileExtention != String.Empty)
        FileName += FileExtention;

      this.AddItem(FileMemorySize, "BYTE[ ]", "FileMemory", FileMemorySize.ToString("#,#") + " bytes", new FileViewerObject(FileViewerObjectType.PicturePaintFile, new FileViewerTDPicturePaintFileObject(FileName, FileMemory)));
      return FileMemory;
    }


    private void ReadBegin() {
      UInt32 IDNumber = this.ReadIDNumber();

      Int64 FileSize = this.ReadFileSize();

      Byte Version = this.ReadVersion();

      Int32 FileCount = this.ReadFileCount();

      if(FileCount == 0) {
        if(this.Position != this.Size) throw new Exception("Wrong FileSize");
        return;
      }

      for(Int32 i = 0; i < FileCount; i++) {
        Byte FileType = this.ReadFileType();

        Int32 FileNameLength = this.ReadFileNameLength();
        String FileName = this.ReadFileName(FileNameLength);

        Int32 FileExtentionLength = this.ReadFileExtentionLength();
        String FileExtention = this.ReadFileExtention(FileExtentionLength);

        Byte ImageFormat = this.ReadImageFormat();
        Int32 ImageWidth = this.ReadImageWidth();
        Int32 ImageHeight = this.ReadImageHeight();

        Int64 PlayerDuration = this.ReadPlayerDuration();

        Int32 FileMemorySize = this.ReadFileMemorySize();
        Byte[] FileMemory = this.ReadFileMemory(FileMemorySize, FileName, FileExtention);
      }

      if(this.Position != this.Size) throw new Exception("Wrong FileSize");
    }

    #endregion


    #region Public Methods

    public List<ListViewItem> Run(IFileViewerProgressBar Parent, String FilePath) {
      this.Parent = Parent;

      try {
        this.Stream = new FileStream(FilePath, FileMode.Open, FileAccess.Read);
        this.Reader = new BinaryReader(this.Stream);

        this.Size = this.Stream.Length;

        this.Parent.BeginProgressBar(this.Size);

        this.AddItemBeginRead();

        this.ReadBegin();
      } catch(Exception e) {
        this.AddItemError(e);

        this.Close();
        return this.ItemList;
      }

      this.AddItemEndOfFile();

      this.Close();
      return this.ItemList;
    }

    #endregion


    #region Public Static Methods

    public static Dictionary<String, MemoryStream> Load(String FilePath) {
      //Dictionary: System.Collections.Generic
      //FileStream, BinaryReader, MemoryStream: System.IO

      try {
        FileStream Stream = new FileStream(FilePath, FileMode.Open, FileAccess.Read);
        BinaryReader Reader = new BinaryReader(Stream);

        if(Reader.ReadUInt32() != 0x43504454) return null;
        if(Reader.ReadInt64() != Stream.Length) return null;

        Dictionary<String, MemoryStream> FileDictionary = new Dictionary<String, MemoryStream>();

        Int32 FileCount = Reader.ReadInt32();

        for(Int32 i = 0; i < FileCount; i++) {
          Stream.Position++;

          Int32 FileNameLength = Reader.ReadInt32();

          if(FileNameLength < 1 || FileNameLength > 260) return null;

          String FileName = String.Empty;

          while(FileNameLength > 0) {
            FileName += (Char) Reader.ReadUInt16();
            FileNameLength--;
          }

          Int32 FileExtentionLength = Reader.ReadInt32();

          if(FileExtentionLength < 0 || FileExtentionLength > 260) return null;

          Stream.Position += FileExtentionLength * 2 + 1 + 4 + 4 + 8;

          Int32 FileSize = Reader.ReadInt32();

          if(FileSize < 1 || Stream.Position + FileSize > Stream.Length) return null;

          FileDictionary[FileName] = new MemoryStream(Reader.ReadBytes(FileSize));
        }

        if(Stream.Position != Stream.Length) return null;

        Stream.Close();
        return FileDictionary;
      } catch(Exception) {

      }
      return null;
    }

    #endregion
  }


  public enum FileViewerTDPicturePaintProjectType {
    AlphaArray = 0,
    AlphaPalette = 1,
    AlphaGradient = 2,
    ColorArray = 3,
    ColorPalette = 4,
    ColorGradient = 5,
    GroupItem = 6,
  }

  public class FileViewerTDPicturePaintProjectObject {
    #region Public Fields

    public FileViewerTDPicturePaintProjectType Type;

    public Object Value;

    #endregion

    #region Contructor

    public FileViewerTDPicturePaintProjectObject(FileViewerTDPicturePaintProjectType Type, Object Value) {
      this.Type = Type;
      this.Value = Value;
    }

    #endregion
  }

  public class FileViewerTDPicturePaintProjectPalette {
    #region Public Fields

    public Byte Select;
    public UInt32 Value;

    #endregion

    #region Contructor

    public FileViewerTDPicturePaintProjectPalette(Byte Select, Byte Alpha) {
      this.Select = Select;
      this.Value = Alpha;
    }

    public FileViewerTDPicturePaintProjectPalette(Byte Select, UInt32 Color) {
      this.Select = Select;
      this.Value = Color;
    }

    #endregion
  }

  public class FileViewerTDPicturePaintProjectGradient {
    #region Public Fields

    public Byte Type;
    public UInt32 Begin;
    public UInt32 Between;
    public UInt32 Middle;
    public UInt32 End;
    public Int32 Factor;
    public Int32 FactorBetween;
    public Int32 Length;

    public String MathString;
    public UInt32[] ValueArray;

    #endregion

    #region Contructor

    public FileViewerTDPicturePaintProjectGradient(Byte Type) {
      this.Type = Type;
    }

    public FileViewerTDPicturePaintProjectGradient(Byte Type, Byte Begin, Byte Between, Byte Middle, Byte End, Int32 Factor, Int32 FactorBetween, Int32 Length) {
      this.Type = Type;
      this.Begin = Begin;
      this.Between = Between;
      this.Middle = Middle;
      this.End = End;
      this.Factor = Factor;
      this.FactorBetween = FactorBetween;
      this.Length = Length;
    }

    public FileViewerTDPicturePaintProjectGradient(Byte Type, UInt32 Begin, UInt32 Between, UInt32 Middle, UInt32 End, Int32 Factor, Int32 FactorBetween, Int32 Length) {
      this.Type = Type;
      this.Begin = Begin;
      this.Between = Between;
      this.Middle = Middle;
      this.End = End;
      this.Factor = Factor;
      this.FactorBetween = FactorBetween;
      this.Length = Length;
    }

    #endregion


    #region Public Methods

    public void CreateSineColor(UInt32[] ColorArray, Int32 Index, Int32 Count, Double Factor, Color Begin, Color End) {
      if(ColorArray == null || ColorArray.Length == 0) return;

      if(ColorArray.Length == 1) {
        ColorArray[0] = (UInt32) Begin.ToArgb();
        return;
      }

      double A = End.A - Begin.A;
      double R = End.R - Begin.R;
      double G = End.G - Begin.G;
      double B = End.B - Begin.B;

      double dCount = (Count - 1) * (Count - 1);

      for(Int32 i = 0; i < Count; i++) {
        double Value = Math.Pow(1.0 - i * i / dCount, Factor);

        ColorArray[Index + Count - 1 - i] = (UInt32) Color.FromArgb(Convert.ToInt32(Begin.A + Value * A), Convert.ToInt32(Begin.R + Value * R), Convert.ToInt32(Begin.G + Value * G), Convert.ToInt32(Begin.B + Value * B)).ToArgb();
      }
    }

    public void CreateSineColor(UInt32[] ColorArray, Int32 Index, Int32 Count, Double Factor, Double FactorBegin, Color Begin, Color Middle, Color End) {
      if(ColorArray == null || Index < 0 || Count < 1 || ColorArray.Length < Index + Count) return;

      if(Count < 4) {
        ColorArray[Index] = (UInt32) Begin.ToArgb();
        if(Count > 1) ColorArray[Index + 1] = (UInt32) Middle.ToArgb();
        if(Count > 2) ColorArray[Index + 2] = (UInt32) End.ToArgb();
        return;
      }

      if(FactorBegin < 0) FactorBegin = 0;
      if(FactorBegin > 1.0) FactorBegin = 1.0;

      Int32 CountA = Convert.ToInt32(Count * FactorBegin);
      Int32 CountB = Count - CountA;

      double dCount = (Count - 1) * (Count - 1);

      if(CountA < 3) {
        if(CountA > 0) ColorArray[Index] = (UInt32) Begin.ToArgb();
        if(CountA == 2) ColorArray[Index + 1] = (UInt32) Middle.ToArgb();

        Index += CountA;
      } else {
        double dRef = 1.0 / Math.Pow(1.0 - CountB * CountB / dCount, Factor);

        double A = Middle.A - Begin.A;
        double R = Middle.R - Begin.R;
        double G = Middle.G - Begin.G;
        double B = Middle.B - Begin.B;

        for(Int32 i = Count - 1; i >= CountB; i--, Index++) {
          double Value = Math.Pow(1.0 - i * i / dCount, Factor) * dRef;

          ColorArray[Index] = (UInt32) Color.FromArgb(Convert.ToInt32(Begin.A + Value * A), Convert.ToInt32(Begin.R + Value * R), Convert.ToInt32(Begin.G + Value * G), Convert.ToInt32(Begin.B + Value * B)).ToArgb();
        }
      }

      if(CountB < 3) {
        if(CountB > 0) ColorArray[Index + CountB - 1] = (UInt32) End.ToArgb();
        if(CountB == 2) ColorArray[Index + CountB - 2] = (UInt32) Middle.ToArgb();
      } else {
        double dRef = Math.Pow(1.0 - (CountB - 1) * (CountB - 1) / dCount, Factor);

        double A = End.A - Middle.A;
        double R = End.R - Middle.R;
        double G = End.G - Middle.G;
        double B = End.B - Middle.B;

        for(Int32 i = CountB - 1; i >= 0; i--, Index++) {
          double Value = (Math.Pow(1.0 - i * i / dCount, Factor) - dRef) * 1.0 / (1.0 - dRef);

          ColorArray[Index] = (UInt32) Color.FromArgb(Convert.ToInt32(Middle.A + Value * A), Convert.ToInt32(Middle.R + Value * R), Convert.ToInt32(Middle.G + Value * G), Convert.ToInt32(Middle.B + Value * B)).ToArgb();
        }
      }
    }

    public void CreateCosineColor(UInt32[] ColorArray, Int32 Index, Int32 Count, Double Factor, Color Begin, Color End) {
      if(ColorArray == null || ColorArray.Length == 0) return;

      if(ColorArray.Length == 1) {
        ColorArray[0] = (UInt32) Begin.ToArgb();
        return;
      }

      double A = End.A - Begin.A;
      double R = End.R - Begin.R;
      double G = End.G - Begin.G;
      double B = End.B - Begin.B;

      double dCount = (Count - 1) * (Count - 1);

      for(Int32 i = 0; i < Count; i++) {
        double Value = Math.Pow(1.0 - i * i / dCount, Factor);

        ColorArray[Index + i] = (UInt32) Color.FromArgb(Convert.ToInt32(End.A - Value * A), Convert.ToInt32(End.R - Value * R), Convert.ToInt32(End.G - Value * G), Convert.ToInt32(End.B - Value * B)).ToArgb();
      }
    }


    public UInt32[] CreateLinearColor(Int32 Count, Color Begin, Color End) {
      if(Count < 1) return new UInt32[0];

      UInt32[] ColorArray = new UInt32[Count];

      if(ColorArray.Length == 1) {
        ColorArray[0] = (UInt32) Begin.ToArgb();
        return ColorArray;
      }

      double A = End.A - Begin.A;
      double R = End.R - Begin.R;
      double G = End.G - Begin.G;
      double B = End.B - Begin.B;

      for(Int32 i = 0; i < Count; i++) {
        double Value;

        if(i == Count - 1)
          Value = 1.0;
        else
          Value = ((double) i) / (Count - 1);

        ColorArray[i] = (UInt32) Color.FromArgb(Convert.ToInt32(Begin.A + Value * A), Convert.ToInt32(Begin.R + Value * R), Convert.ToInt32(Begin.G + Value * G), Convert.ToInt32(Begin.B + Value * B)).ToArgb();
      }
      return ColorArray;
    }

    public UInt32[] CreateSineColor(Int32 Count, Double Factor, Color Begin, Color End) {
      if(Count < 1) return new UInt32[0];

      UInt32[] ColorArray = new UInt32[Count];

      if(Count == 1) {
        ColorArray[0] = (UInt32) Begin.ToArgb();
        return ColorArray;
      }

      double A = End.A - Begin.A;
      double R = End.R - Begin.R;
      double G = End.G - Begin.G;
      double B = End.B - Begin.B;

      double dCount = (Count - 1) * (Count - 1);

      for(Int32 i = 0; i < Count; i++) {
        double Value = Math.Pow(1.0 - i * i / dCount, Factor);

        ColorArray[Count - 1 - i] = (UInt32) Color.FromArgb(Convert.ToInt32(Begin.A + Value * A), Convert.ToInt32(Begin.R + Value * R), Convert.ToInt32(Begin.G + Value * G), Convert.ToInt32(Begin.B + Value * B)).ToArgb();
      }
      return ColorArray;
    }

    public UInt32[] CreateCosineColor(Int32 Count, Double Factor, Color Begin, Color End) {
      if(Count < 1) return new UInt32[0];

      UInt32[] ColorArray = new UInt32[Count];

      if(Count == 1) {
        ColorArray[0] = (UInt32) Begin.ToArgb();
        return ColorArray;
      }

      double A = End.A - Begin.A;
      double R = End.R - Begin.R;
      double G = End.G - Begin.G;
      double B = End.B - Begin.B;

      double dCount = (Count - 1) * (Count - 1);

      for(Int32 i = 0; i < Count; i++) {
        double Value = Math.Pow(1.0 - i * i / dCount, Factor);

        ColorArray[i] = (UInt32) Color.FromArgb(Convert.ToInt32(End.A - Value * A), Convert.ToInt32(End.R - Value * R), Convert.ToInt32(End.G - Value * G), Convert.ToInt32(End.B - Value * B)).ToArgb();
      }
      return ColorArray;
    }


    public UInt32[] CreateRoundColor(Int32 Count, Double Factor, Color Begin, Color Middle, Color End) {
      if(Count < 1) return new UInt32[0];

      UInt32[] ColorArray = new UInt32[Count];

      if(Count < 4) {
        if(Count == 1) {
          ColorArray[0] = (UInt32) Middle.ToArgb();
          return ColorArray;
        }
        if(Count == 2) {
          ColorArray[0] = (UInt32) Middle.ToArgb();
          ColorArray[1] = (UInt32) End.ToArgb();
          return ColorArray;
        }


        ColorArray[0] = (UInt32) Begin.ToArgb();
        ColorArray[1] = (UInt32) Middle.ToArgb();
        ColorArray[2] = (UInt32) End.ToArgb();
        return ColorArray;
      }

      Int32 SizeA = Count / 2;
      Int32 SizeB = Count - SizeA;

      this.CreateSineColor(ColorArray, 0, SizeA, Factor, Begin, Middle);
      this.CreateCosineColor(ColorArray, SizeA, SizeB, Factor, Middle, End);
      return ColorArray;
    }

    public UInt32[] CreateRoundColor(Int32 Count, Double Factor, Double FactorBetween, Color Begin, Color Between, Color Middle, Color End) {
      if(Count < 1) return new UInt32[0];

      UInt32[] ColorArray = new UInt32[Count];

      if(Count < 5) {
        if(Count == 1) {
          ColorArray[0] = (UInt32) Middle.ToArgb();
          return ColorArray;
        }
        if(Count == 2) {
          ColorArray[0] = (UInt32) Middle.ToArgb();
          ColorArray[1] = (UInt32) End.ToArgb();
          return ColorArray;
        }

        if(Count == 3) {
          ColorArray[0] = (UInt32) Begin.ToArgb();
          ColorArray[1] = (UInt32) Middle.ToArgb();
          ColorArray[2] = (UInt32) End.ToArgb();
          return ColorArray;
        }

        ColorArray[0] = (UInt32) Begin.ToArgb();
        ColorArray[1] = (UInt32) Between.ToArgb();
        ColorArray[2] = (UInt32) Middle.ToArgb();
        ColorArray[3] = (UInt32) End.ToArgb();
        return ColorArray;
      }

      Int32 SizeA = Count / 2;
      Int32 SizeB = Count - SizeA;

      this.CreateSineColor(ColorArray, 0, SizeA, Factor, FactorBetween, Begin, Between, Middle);
      this.CreateCosineColor(ColorArray, SizeA, SizeB, Factor, Middle, End);
      return ColorArray;
    }


    public Bitmap CreateAlphaBitmap() {
      Bitmap Image = new Bitmap(75, 16, PixelFormat.Format24bppRgb);

      UInt32[] ColorArray = null;

      switch(this.Type) {
        case 0: {
          ColorArray = this.CreateLinearColor(this.Length, Color.FromArgb((Int32) this.Begin * 0x01000000), Color.FromArgb((Int32) this.End * 0x01000000));
          break;
        }
        case 1: {
          ColorArray = this.CreateSineColor(this.Length, this.Factor / 100.0, Color.FromArgb((Int32) this.Begin * 0x01000000), Color.FromArgb((Int32) this.End * 0x01000000));
          break;
        }
        case 2: {
          ColorArray = this.CreateCosineColor(this.Length, this.Factor / 100.0, Color.FromArgb((Int32) this.Begin * 0x01000000), Color.FromArgb((Int32) this.End * 0x01000000));
          break;
        }
        case 3: {
          ColorArray = this.CreateRoundColor(this.Length, this.Factor / 100.0, Color.FromArgb((Int32) this.Begin * 0x01000000), Color.FromArgb((Int32) this.Middle * 0x01000000), Color.FromArgb((Int32) this.End * 0x01000000));
          break;
        }
        case 4: {
          ColorArray = this.CreateRoundColor(this.Length, this.Factor / 100.0, this.FactorBetween / 100.0, Color.FromArgb((Int32) this.Begin * 0x01000000), Color.FromArgb((Int32) this.Between * 0x01000000), Color.FromArgb((Int32) this.Middle * 0x01000000), Color.FromArgb((Int32) this.End * 0x01000000));
          break;
        }
        default: {
          return Image;
        }
      }

      if(ColorArray == null || ColorArray.Length < 2) return Image;

      Graphics Graphics = Graphics.FromImage(Image);

      Int32 ColorCount = ColorArray.Length - 1;
      Int32 Count = Image.Width - 1;

      for(Int32 i = 0; i <= Count; i++) {
        UInt32 AlphaColor = ColorArray[i * ColorCount / Count];
        Byte Alpha = (Byte) (AlphaColor / 0x01000000);

        Graphics.DrawLine(new Pen(Color.FromArgb(Alpha, Alpha, Alpha)), new Point(i, 0), new Point(i, 16));
        Graphics.DrawRectangle(new Pen(Color.Gray), new Rectangle(0, 0, 74, 15));
      }

      Graphics.Dispose();
      return Image;
    }

    public Bitmap CreateColorBitmap() {
      Bitmap Image = new Bitmap(75, 16, PixelFormat.Format24bppRgb);

      UInt32[] ColorArray = null;

      Boolean ShowAlpha = false;

      if(this.Begin < 0xFF000000) ShowAlpha = true;
      if(this.End < 0xFF000000) ShowAlpha = true;

      switch(this.Type) {
        case 0: {
          ColorArray = this.CreateLinearColor(this.Length, Color.FromArgb((Int32) this.Begin), Color.FromArgb((Int32) this.End));
          break;
        }
        case 1: {
          ColorArray = this.CreateSineColor(this.Length, this.Factor / 100.0, Color.FromArgb((Int32) this.Begin), Color.FromArgb((Int32) this.End));
          break;
        }
        case 2: {
          ColorArray = this.CreateCosineColor(this.Length, this.Factor / 100.0, Color.FromArgb((Int32) this.Begin), Color.FromArgb((Int32) this.End));
          break;
        }
        case 3: {
          if(this.Middle < 0xFF000000) ShowAlpha = true;

          ColorArray = this.CreateRoundColor(this.Length, this.Factor / 100.0, Color.FromArgb((Int32) this.Begin), Color.FromArgb((Int32) this.Middle), Color.FromArgb((Int32) this.End));
          break;
        }
        case 4: {
          if(this.Middle < 0xFF000000) ShowAlpha = true;
          if(this.Between < 0xFF000000) ShowAlpha = true;

          ColorArray = this.CreateRoundColor(this.Length, this.Factor / 100.0, this.FactorBetween / 100.0, Color.FromArgb((Int32) this.Begin), Color.FromArgb((Int32) this.Between), Color.FromArgb((Int32) this.Middle), Color.FromArgb((Int32) this.End));
          break;
        }
        default: {
          return Image;
        }
      }

      if(ColorArray == null || ColorArray.Length < 2) return Image;

      Graphics Graphics = Graphics.FromImage(Image);

      Int32 ColorCount = ColorArray.Length - 1;
      Int32 Count = Image.Width - 1;

      if(ShowAlpha) {
        for(Int32 i = 0; i <= Count; i++) {
          UInt32 AlphaColor = ColorArray[i * ColorCount / Count];
          Byte Alpha = (Byte) (AlphaColor / 0x01000000);

          Graphics.DrawLine(new Pen(Color.FromArgb(Alpha, Alpha, Alpha)), new Point(i, 0), new Point(i, 4));
          Graphics.DrawLine(new Pen(Color.FromArgb((Int32) (0xFF000000 | AlphaColor))), new Point(i, 4), new Point(i, 16));
          Graphics.DrawRectangle(new Pen(Color.Gray), new Rectangle(0, 0, 74, 15));
        }
      } else {
        for(Int32 i = 0; i <= Count; i++) {
          UInt32 AlphaColor = ColorArray[i * ColorCount / Count];

          Graphics.DrawLine(new Pen(Color.FromArgb((Int32) (0xFF000000 | AlphaColor))), new Point(i, 0), new Point(i, 16));
          Graphics.DrawRectangle(new Pen(Color.Gray), new Rectangle(0, 0, 74, 15));
        }
      }

      Graphics.Dispose();
      return Image;
    }

    #endregion
  }

  public class FileViewerTDPicturePaintProject {
    #region Public Fields

    public IFileViewerProgressBar Parent;
    public List<ListViewItem> ItemList;
    public BinaryReader Reader;
    public FileStream Stream;
    public String FilePath;
    public Int64 Position;
    public Int64 Size;

    #endregion


    #region Contructor

    public FileViewerTDPicturePaintProject() {
      this.ItemList = new List<ListViewItem>();
    }

    #endregion


    #region Private Methods

    private void AddItem(Int64 Size, String Type, String Name, String Value) {
      this.ItemList.Add(new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value }));

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }

    private void AddItem(Int64 Size, String Type, String Name, String Value, FileViewerObject Tag) {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value });

      Item.BackColor = Color.FromArgb(240, 255, 245);
      Item.Tag = Tag;

      this.ItemList.Add(Item);

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }


    private void AddItemError(String Value) {
      ListViewItem Item = new ListViewItem(new String[] { "".PadRight(10), "".PadRight(10), "".PadRight(10), "Error".PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }

    private void AddItemError(Exception e) {
      String Name = String.Empty;
      String Value = String.Empty;

      if(e.InnerException != null && e.InnerException.Message != null) {
        if(e.Message != null)
          Name = e.Message;

        Value = e.InnerException.Message;
      } else {
        if(e.Message != null)
          Value = e.Message;
      }

      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "ERROR".PadRight(10), Name.PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }


    private void AddItemBeginRead() {
      ListViewItem Item = new ListViewItem(new String[] { "0".PadRight(10), "0".PadRight(10), "".PadRight(10), "BeginRead".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }

    private void AddItemEndOfFile() {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "".PadRight(10), "EndOfFile".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }


    private void Close() {
      if(this.Stream != null) {
        this.Stream.Close();
        this.Stream = null;
      }

      if(this.Reader != null) {
        this.Reader.Close();
        this.Reader = null;
      }
    }

    #endregion

    #region Private Methods: Format

    private UInt32 ReadIDNumber() {
      UInt32 IDNumber = this.Reader.ReadUInt32();

      if(IDNumber != 0x50504454) //TDPP
        throw new Exception("IDNumber", new Exception("0x" + IDNumber.ToString("X8")));

      this.AddItem(4, "UINT32", "IDNumber", "0x" + IDNumber.ToString("X8") + " hex");
      return IDNumber;
    }

    private Int64 ReadFileSize() {
      Int64 FileSize = this.Reader.ReadInt64();

      if(FileSize != this.Size)
        throw new Exception("FileSize", new Exception(FileSize.ToString()));

      this.AddItem(8, "INT64", "FileSize", FileSize.ToString("#,#") + " bytes");
      return FileSize;
    }

    private Byte ReadVersion() {
      Byte Version = this.Reader.ReadByte();

      if(Version != 1)
        throw new Exception("Version", new Exception(Version.ToString()));

      this.AddItem(1, "BYTE", "Version", Version.ToString());
      return Version;
    }

    private Int32 ReadPreviewImageSize() {
      Int32 PreviewImageSize = this.Reader.ReadInt32();

      if(PreviewImageSize < 0 || PreviewImageSize > this.Size - this.Position - 4)
        throw new Exception("PreviewImageSize", new Exception(PreviewImageSize.ToString()));

      if(PreviewImageSize == 0)
        this.AddItem(4, "INT32", "PreviewImageSize", "0 byte");
      else
        this.AddItem(4, "INT32", "PreviewImageSize", PreviewImageSize.ToString("#,#") + " bytes");

      return PreviewImageSize;
    }

    private void ReadPreviewImage(Int32 PreviewImageSize) {
      try {
        Byte[] PreviewImageArray = this.Reader.ReadBytes(PreviewImageSize);

        Bitmap PreviewImage = new Bitmap(new MemoryStream(PreviewImageArray));

        this.AddItem(PreviewImageSize, "MEMORY", "PreviewImage", "", new FileViewerObject(PreviewImage));
      } catch(Exception e) {
        throw new Exception("PreviewImage", e);
      }
    }


    private Int32 ReadFileCount() {
      Int32 FileCount = this.Reader.ReadInt32();

      if(FileCount < 0)
        throw new Exception("FileCount", new Exception(FileCount.ToString()));

      this.AddItem(4, "INT32", "FileCount", FileCount.ToString());
      return FileCount;
    }

    private Byte ReadFileType() {
      Byte FileType = this.Reader.ReadByte();

      //Image = 0, Video = 1, Sound = 2, Gif = 3, Animation = 4, Various = 5 

      if(FileType > 5)
        throw new Exception("FileType", new Exception(FileType.ToString()));

      String InfoText = String.Empty;

      switch(FileType) {
        case 0: {
          InfoText = " (Image)";
          break;
        }
        case 1: {
          InfoText = " (Video)";
          break;
        }
        case 2: {
          InfoText = " (Sound)";
          break;
        }
        case 3: {
          InfoText = " (Gif)";
          break;
        }
        case 4: {
          InfoText = " (Animation)";
          break;
        }
        case 5: {
          InfoText = " (Various)";
          break;
        }
      }

      this.AddItem(1, "BYTE", "FileType", FileType + InfoText);
      return FileType;
    }

    private Int32 ReadFileNameLength() {
      Int32 FileNameLength = this.Reader.ReadInt32();

      if(FileNameLength <= 0)
        throw new Exception("FileNameLength", new Exception(FileNameLength.ToString()));

      this.AddItem(4, "INT32", "FileNameLength", FileNameLength + " letters");
      return FileNameLength;
    }

    private String ReadFileName(Int32 FileNameLength) {
      Char[] LetterArray = new Char[FileNameLength];

      for(Int32 i = 0; i < FileNameLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String FileName = new String(LetterArray);

      if(FileName == null || FileName == String.Empty)
        throw new Exception("FileName", new Exception("Empty"));

      this.AddItem(FileNameLength * 2, "WCHAR[ ]", "FileName", FileName);
      return FileName;
    }

    private Int32 ReadFileExtentionLength() {
      Int32 FileExtentionLength = this.Reader.ReadInt32();

      if(FileExtentionLength < 0)
        throw new Exception("FileExtentionLength", new Exception(FileExtentionLength.ToString()));

      this.AddItem(4, "INT32", "FileExtentionLength", FileExtentionLength + " letters");
      return FileExtentionLength;
    }

    private String ReadFileExtention(Int32 FileExtentionLength) {
      if(FileExtentionLength == 0) {
        this.AddItem(0, "WCHAR[ ]", "FileExtention", "Empty");
        return String.Empty;
      }

      Char[] LetterArray = new Char[FileExtentionLength];

      for(Int32 i = 0; i < FileExtentionLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String FileExtention = new String(LetterArray);

      if(FileExtention == null || FileExtention == String.Empty)
        throw new Exception("FileExtention", new Exception("Empty"));

      this.AddItem(FileExtentionLength * 2, "WCHAR[ ]", "FileExtention", FileExtention);
      return FileExtention;
    }


    private Byte ReadImageFormat() {
      Byte ImageFormat = this.Reader.ReadByte();

      //Unknown = 0, Bmp = 1, Emf = 2, Exif = 3,
      //Gif = 4, Icon = 5, Jpeg = 6, Png = 7,
      //Tiff = 8, Wmf = 9, Tdp = 10,

      if(ImageFormat > 10)
        throw new Exception("ImageFormat", new Exception(ImageFormat.ToString()));

      String InfoText = String.Empty;

      switch(ImageFormat) {
        case 0: {
          break;
        }
        case 1: {
          InfoText = " (bmp)";
          break;
        }
        case 2: {
          InfoText = " (emf)";
          break;
        }
        case 3: {
          InfoText = " (exif)";
          break;
        }
        case 4: {
          InfoText = " (gif)";
          break;
        }
        case 5: {
          InfoText = " (icon)";
          break;
        }
        case 6: {
          InfoText = " (jpeg)";
          break;
        }
        case 7: {
          InfoText = " (png)";
          break;
        }
        case 8: {
          InfoText = " (tiff)";
          break;
        }
        case 9: {
          InfoText = " (wmf)";
          break;
        }
        case 10: {
          InfoText = " (tdp)";
          break;
        }
      }

      this.AddItem(1, "BYTE", "ImageFormat", ImageFormat + InfoText);
      return ImageFormat;
    }

    private UInt16 ReadImageFlags() {
      UInt16 ImageFlags = this.Reader.ReadUInt16();

      this.AddItem(2, "UINT16", "ImageFlags", "0x" + ImageFlags.ToString("X4"));
      return ImageFlags;
    }

    private Int32 ReadImagePosX() {
      Int32 ImagePosX = this.Reader.ReadInt32();

      if(ImagePosX < 0 || ImagePosX > 1000000)
        throw new Exception("ImagePosX", new Exception(ImagePosX.ToString()));

      this.AddItem(4, "INT32", "ImagePosX", ImagePosX.ToString());
      return ImagePosX;
    }

    private Int32 ReadImagePosY() {
      Int32 ImagePosY = this.Reader.ReadInt32();

      if(ImagePosY < 0 || ImagePosY > 1000000)
        throw new Exception("ImagePosY", new Exception(ImagePosY.ToString()));

      this.AddItem(4, "INT32", "ImagePosY", ImagePosY.ToString());
      return ImagePosY;
    }

    private Int32 ReadImageWidth() {
      Int32 ImageWidth = this.Reader.ReadInt32();

      if(ImageWidth < 0 || ImageWidth > 1000000)
        throw new Exception("ImageWidth", new Exception(ImageWidth.ToString()));

      this.AddItem(4, "INT32", "ImageWidth", ImageWidth + " pixel");
      return ImageWidth;
    }

    private Int32 ReadImageHeight() {
      Int32 ImageHeight = this.Reader.ReadInt32();

      if(ImageHeight < 0 || ImageHeight > 1000000)
        throw new Exception("ImageHeight", new Exception(ImageHeight.ToString()));

      this.AddItem(4, "INT32", "ImageHeight", ImageHeight + " pixel");
      return ImageHeight;
    }

    private Int32 ReadImagePreview() {
      Int32 ImagePreview = this.Reader.ReadInt32();

      if(ImagePreview != 0)
        throw new Exception("ImagePreview", new Exception(ImagePreview.ToString()));

      this.AddItem(4, "INT32", "ImagePreview", ImagePreview.ToString());
      return ImagePreview;
    }


    private Int32 ReadToolIndex() {
      Int32 ToolIndex = this.Reader.ReadInt32();

      if(ToolIndex < 0 || ToolIndex > 10)
        throw new Exception("ToolIndex", new Exception(ToolIndex.ToString()));

      this.AddItem(4, "INT32", "ToolIndex", ToolIndex.ToString());
      return ToolIndex;
    }

    private Byte ReadPaintAlpha() {
      Byte PaintAlpha = this.Reader.ReadByte();

      this.AddItem(1, "BYTE", "PaintAlpha", "0x" + PaintAlpha.ToString("X2"));
      return PaintAlpha;
    }

    private UInt32 ReadPaintColor(String Name) {
      UInt32 Color = this.Reader.ReadUInt32();

      this.AddItem(4, "UINT32", Name, "0x" + Color.ToString("X8"));
      return Color;
    }

    private Int32 ReadLineSize() {
      Int32 LineSize = this.Reader.ReadInt32();

      if(LineSize < 1 || LineSize > 999)
        throw new Exception("LineSize", new Exception(LineSize.ToString()));

      this.AddItem(4, "INT32", "LineSize", LineSize.ToString());
      return LineSize;
    }

    private Int32 ReadRoundWidth() {
      Int32 RoundWidth = this.Reader.ReadInt32();

      if(RoundWidth < 1 || RoundWidth > 1000000)
        throw new Exception("RoundWidth", new Exception(RoundWidth.ToString()));

      this.AddItem(4, "INT32", "RoundWidth", RoundWidth.ToString());
      return RoundWidth;
    }

    private Int32 ReadRoundHeight() {
      Int32 RoundHeight = this.Reader.ReadInt32();

      if(RoundHeight < 1 || RoundHeight > 1000000)
        throw new Exception("RoundHeight", new Exception(RoundHeight.ToString()));

      this.AddItem(4, "INT32", "RoundHeight", RoundHeight.ToString());
      return RoundHeight;
    }

    private Int32 ReadRotationLeverLength(String Name) {
      Int32 RotationLeverLength = this.Reader.ReadInt32();

      if(RotationLeverLength < 5 || RotationLeverLength > 2000)
        throw new Exception(Name, new Exception(RotationLeverLength.ToString()));

      this.AddItem(4, "INT32", Name, RotationLeverLength.ToString());
      return RotationLeverLength;
    }

    private Double ReadRotationAngle(String Name) {
      Double RotationAngle = this.Reader.ReadDouble();

      if(RotationAngle < 0.0 || RotationAngle > 360.0)
        throw new Exception(Name, new Exception(RotationAngle.ToString()));

      this.AddItem(8, "DOUBLE", Name, RotationAngle.ToString());
      return RotationAngle;
    }


    private Int64 ReadPlayerDuration() {
      Int64 PlayerDuration = this.Reader.ReadInt64();

      if(PlayerDuration < 0)
        throw new Exception("PlayerDuration", new Exception(PlayerDuration.ToString()));

      this.AddItem(8, "INT64", "PlayerDuration", new TimeSpan(PlayerDuration).ToString());
      return PlayerDuration;
    }

    private UInt32 ReadBackColor() {
      UInt32 BackColor = this.Reader.ReadUInt32();

      this.AddItem(4, "UINT32", "BackColor", "0x" + BackColor.ToString("X4"));
      return BackColor;
    }

    private Int32 ReadFileImageSize() {
      Int32 FileImageSize = this.Reader.ReadInt32();

      if(FileImageSize < 0 || FileImageSize > this.Size - this.Position - 4)
        throw new Exception("FileImageSize", new Exception(FileImageSize.ToString()));

      if(FileImageSize == 0)
        this.AddItem(4, "INT32", "FileImageSize", "0 byte");
      else
        this.AddItem(4, "INT32", "FileImageSize", FileImageSize.ToString("#,#") + " bytes");

      return FileImageSize;
    }

    private void ReadFileImage(Int32 FileImageSize) {
      try {
        Byte[] FileImageArray = this.Reader.ReadBytes(FileImageSize);

        Bitmap FileImage = new Bitmap(new MemoryStream(FileImageArray));

        this.AddItem(FileImageSize, "MEMORY", "FileImage", "", new FileViewerObject(FileImage));
      } catch(Exception e) {
        throw new Exception("FileImage", e);
      }
    }

    private Int32 ReadFileMemorySize() {
      Int32 FileMemorySize = this.Reader.ReadInt32();

      if(FileMemorySize < 0 || FileMemorySize > this.Size - this.Position - 4)
        throw new Exception("FileMemorySize", new Exception(FileMemorySize.ToString()));

      this.AddItem(4, "INT32", "FileMemorySize", FileMemorySize.ToString("#,#") + " bytes");
      return FileMemorySize;
    }

    private Byte[] ReadFileMemory(Int32 FileMemorySize, String FileName, String FileExtention) {
      Byte[] FileMemory = this.Reader.ReadBytes(FileMemorySize);

      if(FileExtention != null && FileExtention != String.Empty)
        FileName += FileExtention;

      this.AddItem(FileMemorySize, "BYTE[ ]", "FileMemory", FileMemorySize.ToString("#,#") + " bytes", new FileViewerObject(FileViewerObjectType.PicturePaintFile, new FileViewerTDPicturePaintFileObject(FileName, FileMemory)));
      return FileMemory;
    }


    private Int32 ReadAlphaArrayCount() {
      Int32 AlphaArrayCount = this.Reader.ReadInt32();

      if(AlphaArrayCount < 0 || AlphaArrayCount > 10000)
        throw new Exception("AlphaArrayCount", new Exception(AlphaArrayCount.ToString()));

      this.AddItem(4, "INT32", "AlphaArrayCount", AlphaArrayCount.ToString());
      return AlphaArrayCount;
    }

    private void ReadAlphaArray(Int32 AlphaArrayCount) {
      Byte[] AlphaArray = new Byte[AlphaArrayCount];

      for(Int32 i = 0; i < AlphaArrayCount; i++)
        AlphaArray[i] = this.Reader.ReadByte();

      this.AddItem(AlphaArrayCount, "Memory", "AlphaArray", "", new FileViewerObject(FileViewerObjectType.PicturePaintProject, new FileViewerTDPicturePaintProjectObject(FileViewerTDPicturePaintProjectType.AlphaArray, AlphaArray)));
    }

    private Int32 ReadColorArrayCount() {
      Int32 ColorArrayCount = this.Reader.ReadInt32();

      if(ColorArrayCount < 0 || ColorArrayCount > 10000)
        throw new Exception("ColorArrayCount", new Exception(ColorArrayCount.ToString()));

      this.AddItem(4, "INT32", "ColorArrayCount", ColorArrayCount.ToString());
      return ColorArrayCount;
    }

    private void ReadColorArray(Int32 ColorArrayCount) {
      UInt32[] ColorArray = new UInt32[ColorArrayCount];

      for(Int32 i = 0; i < ColorArrayCount; i++)
        ColorArray[i] = this.Reader.ReadUInt32();

      this.AddItem(ColorArrayCount * 4, "Memory", "ColorArray", "", new FileViewerObject(FileViewerObjectType.PicturePaintProject, new FileViewerTDPicturePaintProjectObject(FileViewerTDPicturePaintProjectType.ColorArray, ColorArray)));
    }


    private Int32 ReadAlphaPaletteCount() {
      Int32 AlphaPaletteCount = this.Reader.ReadInt32();

      if(AlphaPaletteCount < 0 || AlphaPaletteCount > 10000)
        throw new Exception("AlphaPaletteCount", new Exception(AlphaPaletteCount.ToString()));

      this.AddItem(4, "INT32", "AlphaPaletteCount", AlphaPaletteCount.ToString());
      return AlphaPaletteCount;
    }

    private void ReadAlphaPalette(Int32 AlphaPaletteCount) {
      FileViewerTDPicturePaintProjectPalette[] AlphaPalette = new FileViewerTDPicturePaintProjectPalette[AlphaPaletteCount];

      for(Int32 i = 0; i < AlphaPaletteCount; i++) {
        Byte Select = this.Reader.ReadByte();
        Byte Alpha = this.Reader.ReadByte();

        AlphaPalette[i] = new FileViewerTDPicturePaintProjectPalette(Select, Alpha);
      }

      this.AddItem(AlphaPaletteCount * 2, "Memory", "AlphaPalette", "", new FileViewerObject(FileViewerObjectType.PicturePaintProject, new FileViewerTDPicturePaintProjectObject(FileViewerTDPicturePaintProjectType.AlphaPalette, AlphaPalette)));
    }

    private Int32 ReadColorPaletteCount() {
      Int32 ColorPaletteCount = this.Reader.ReadInt32();

      if(ColorPaletteCount < 0 || ColorPaletteCount > 10000)
        throw new Exception("ColorPaletteCount", new Exception(ColorPaletteCount.ToString()));

      this.AddItem(4, "INT32", "ColorPaletteCount", ColorPaletteCount.ToString());
      return ColorPaletteCount;
    }

    private void ReadColorPalette(Int32 ColorPaletteCount) {
      FileViewerTDPicturePaintProjectPalette[] ColorPalette = new FileViewerTDPicturePaintProjectPalette[ColorPaletteCount];

      for(Int32 i = 0; i < ColorPaletteCount; i++) {
        Byte Select = this.Reader.ReadByte();
        UInt32 Color = this.Reader.ReadUInt32();

        ColorPalette[i] = new FileViewerTDPicturePaintProjectPalette(Select, Color);
      }

      this.AddItem(ColorPaletteCount * 5, "Memory", "ColorPalette", "", new FileViewerObject(FileViewerObjectType.PicturePaintProject, new FileViewerTDPicturePaintProjectObject(FileViewerTDPicturePaintProjectType.ColorPalette, ColorPalette)));
    }


    private Int32 ReadAlphaGradientCount() {
      Int32 AlphaGradientCount = this.Reader.ReadInt32();

      if(AlphaGradientCount < 0 || AlphaGradientCount > 10000)
        throw new Exception("AlphaGradientCount", new Exception(AlphaGradientCount.ToString()));

      this.AddItem(4, "INT32", "AlphaGradientCount", AlphaGradientCount.ToString());
      return AlphaGradientCount;
    }

    private Int32 ReadAlphaGradientIndex() {
      Int32 AlphaGradientIndex = this.Reader.ReadInt32();

      if(AlphaGradientIndex < -1 || AlphaGradientIndex >= 10000)
        throw new Exception("AlphaGradientIndex", new Exception(AlphaGradientIndex.ToString()));

      this.AddItem(4, "INT32", "AlphaGradientIndex", AlphaGradientIndex.ToString());
      return AlphaGradientIndex;
    }

    private void ReadAlphaGradient(Int32 AlphaGradientCount) {
      FileViewerTDPicturePaintProjectGradient[] AlphaGradient = new FileViewerTDPicturePaintProjectGradient[AlphaGradientCount];

      Int64 Position = this.Stream.Position;

      for(Int32 i = 0; i < AlphaGradientCount; i++) {
        Byte Type = this.Reader.ReadByte();

        AlphaGradient[i] = new FileViewerTDPicturePaintProjectGradient(Type);

        switch(Type) {
          case 0: { //Linear
            AlphaGradient[i].Begin = this.Reader.ReadByte();
            AlphaGradient[i].End = this.Reader.ReadByte();
            AlphaGradient[i].Length = this.Reader.ReadInt32();
            break;
          }
          case 1: { //Cosinus
            AlphaGradient[i].Begin = this.Reader.ReadByte();
            AlphaGradient[i].End = this.Reader.ReadByte();
            AlphaGradient[i].Factor = this.Reader.ReadInt32();
            AlphaGradient[i].Length = this.Reader.ReadInt32();
            break;
          }
          case 2: { //Sinus
            AlphaGradient[i].Begin = this.Reader.ReadByte();
            AlphaGradient[i].End = this.Reader.ReadByte();
            AlphaGradient[i].Factor = this.Reader.ReadInt32();
            AlphaGradient[i].Length = this.Reader.ReadInt32();
            break;
          }
          case 3: { //Round
            AlphaGradient[i].Begin = this.Reader.ReadByte();
            AlphaGradient[i].Middle = this.Reader.ReadByte();
            AlphaGradient[i].End = this.Reader.ReadByte();
            AlphaGradient[i].Factor = this.Reader.ReadInt32();
            AlphaGradient[i].Length = this.Reader.ReadInt32();
            break;
          }
          case 4: { //RoundExtra
            AlphaGradient[i].Begin = this.Reader.ReadByte();
            AlphaGradient[i].Between = this.Reader.ReadByte();
            AlphaGradient[i].Middle = this.Reader.ReadByte();
            AlphaGradient[i].End = this.Reader.ReadByte();
            AlphaGradient[i].Factor = this.Reader.ReadInt32();
            AlphaGradient[i].FactorBetween = this.Reader.ReadInt32();
            AlphaGradient[i].Length = this.Reader.ReadInt32();
            break;
          }
          /*
          case 5: { //Mixer
            
            Int32 Length = 0;
            Byte[] ByteArray = null;
            Int32 MathLength = 0;
            UInt16[] MathString = null;
            
            break;
          }
          */
          default: {
            throw new Exception("AlphaGradient", new Exception("Wrong Type"));
          }
        }
      }

      this.AddItem(this.Stream.Position - Position, "Memory", "AlphaGradient", "", new FileViewerObject(FileViewerObjectType.PicturePaintProject, new FileViewerTDPicturePaintProjectObject(FileViewerTDPicturePaintProjectType.AlphaGradient, AlphaGradient)));
    }

    private Int32 ReadColorGradientCount() {
      Int32 ColorGradientCount = this.Reader.ReadInt32();

      if(ColorGradientCount < 0 || ColorGradientCount > 10000)
        throw new Exception("ColorGradientCount", new Exception(ColorGradientCount.ToString()));

      this.AddItem(4, "INT32", "ColorGradientCount", ColorGradientCount.ToString());
      return ColorGradientCount;
    }

    private Int32 ReadColorGradientIndex() {
      Int32 ColorGradientIndex = this.Reader.ReadInt32();

      if(ColorGradientIndex < -1 || ColorGradientIndex >= 10000)
        throw new Exception("ColorGradientIndex", new Exception(ColorGradientIndex.ToString()));

      this.AddItem(4, "INT32", "ColorGradientIndex", ColorGradientIndex.ToString());
      return ColorGradientIndex;
    }

    private void ReadColorGradient(Int32 ColorGradientCount) {
      FileViewerTDPicturePaintProjectGradient[] ColorGradient = new FileViewerTDPicturePaintProjectGradient[ColorGradientCount];

      Int64 Position = this.Stream.Position;

      for(Int32 i = 0; i < ColorGradientCount; i++) {
        Byte Type = this.Reader.ReadByte();

        ColorGradient[i] = new FileViewerTDPicturePaintProjectGradient(Type);

        switch(Type) {
          case 0: { //Linear
            ColorGradient[i].Begin = this.Reader.ReadUInt32();
            ColorGradient[i].End = this.Reader.ReadUInt32();
            ColorGradient[i].Length = this.Reader.ReadInt32();

            if(ColorGradient[i].Length < 2 || ColorGradient[i].Length > 256)
              throw new Exception("ColorGradient", new Exception("Wrong Length"));
            break;
          }
          case 1: { //Cosinus
            ColorGradient[i].Begin = this.Reader.ReadUInt32();
            ColorGradient[i].End = this.Reader.ReadUInt32();
            ColorGradient[i].Factor = this.Reader.ReadInt32();
            ColorGradient[i].Length = this.Reader.ReadInt32();

            if(ColorGradient[i].Length < 2 || ColorGradient[i].Length > 256)
              throw new Exception("ColorGradient", new Exception("Wrong Length"));
            break;
          }
          case 2: { //Sinus
            ColorGradient[i].Begin = this.Reader.ReadUInt32();
            ColorGradient[i].End = this.Reader.ReadUInt32();
            ColorGradient[i].Factor = this.Reader.ReadInt32();
            ColorGradient[i].Length = this.Reader.ReadInt32();

            if(ColorGradient[i].Length < 2 || ColorGradient[i].Length > 256)
              throw new Exception("ColorGradient", new Exception("Wrong Length"));
            break;
          }
          case 3: { //Round
            ColorGradient[i].Begin = this.Reader.ReadUInt32();
            ColorGradient[i].Middle = this.Reader.ReadUInt32();
            ColorGradient[i].End = this.Reader.ReadUInt32();
            ColorGradient[i].Factor = this.Reader.ReadInt32();
            ColorGradient[i].Length = this.Reader.ReadInt32();

            if(ColorGradient[i].Length < 3 || ColorGradient[i].Length > 256)
              throw new Exception("ColorGradient", new Exception("Wrong Length"));
            break;
          }
          case 4: { //RoundExtra
            ColorGradient[i].Begin = this.Reader.ReadUInt32();
            ColorGradient[i].Between = this.Reader.ReadUInt32();
            ColorGradient[i].Middle = this.Reader.ReadUInt32();
            ColorGradient[i].End = this.Reader.ReadUInt32();
            ColorGradient[i].Factor = this.Reader.ReadInt32();
            ColorGradient[i].FactorBetween = this.Reader.ReadInt32();
            ColorGradient[i].Length = this.Reader.ReadInt32();

            if(ColorGradient[i].Length < 4 || ColorGradient[i].Length > 256)
              throw new Exception("ColorGradient", new Exception("Wrong Length"));
            break;
          }
          /*
          case 5: { //Mixer
            
            Int32 Length = 0;
            Byte[] ByteArray = null;
            Int32 MathLength = 0;
            UInt16[] MathString = null;
            
            break;
          }
          */
          default: {
            throw new Exception("ColorGradient", new Exception("Wrong Type"));
          }
        }
      }

      this.AddItem(this.Stream.Position - Position, "Memory", "ColorGradient", "", new FileViewerObject(FileViewerObjectType.PicturePaintProject, new FileViewerTDPicturePaintProjectObject(FileViewerTDPicturePaintProjectType.ColorGradient, ColorGradient)));
    }


    private Int32 ReadTextAlphaPaletteCount() {
      Int32 TextAlphaPaletteCount = this.Reader.ReadInt32();

      if(TextAlphaPaletteCount != 0)
        throw new Exception("TextAlphaPaletteCount", new Exception(TextAlphaPaletteCount.ToString()));

      this.AddItem(4, "INT32", "TextAlphaPaletteCount", TextAlphaPaletteCount.ToString());
      return TextAlphaPaletteCount;
    }

    private Int32 ReadTextColorPaletteCount() {
      Int32 TextColorPaletteCount = this.Reader.ReadInt32();

      if(TextColorPaletteCount < 0 || TextColorPaletteCount > 10000)
        throw new Exception("TextColorPaletteCount", new Exception(TextColorPaletteCount.ToString()));

      this.AddItem(4, "INT32", "TextColorPaletteCount", TextColorPaletteCount.ToString());
      return TextColorPaletteCount;
    }

    private void ReadTextColorPalette(Int32 TextColorPaletteCount) {
      FileViewerTDPicturePaintProjectPalette[] TextColorPalette = new FileViewerTDPicturePaintProjectPalette[TextColorPaletteCount];

      for(Int32 i = 0; i < TextColorPaletteCount; i++) {
        Byte Select = this.Reader.ReadByte();
        UInt32 Color = this.Reader.ReadUInt32();

        TextColorPalette[i] = new FileViewerTDPicturePaintProjectPalette(Select, Color);
      }

      this.AddItem(TextColorPaletteCount * 5, "Memory", "TextColorPalette", "", new FileViewerObject(FileViewerObjectType.PicturePaintProject, new FileViewerTDPicturePaintProjectObject(FileViewerTDPicturePaintProjectType.ColorPalette, TextColorPalette)));
    }


    private Int32 ReadTextAlphaGradientCount() {
      Int32 TextAlphaGradientCount = this.Reader.ReadInt32();

      if(TextAlphaGradientCount != 0)
        throw new Exception("TextAlphaGradientCount", new Exception(TextAlphaGradientCount.ToString()));

      this.AddItem(4, "INT32", "TextAlphaGradientCount", TextAlphaGradientCount.ToString());
      return TextAlphaGradientCount;
    }

    private Int32 ReadTextColorGradientCount() {
      Int32 TextColorGradientCount = this.Reader.ReadInt32();

      if(TextColorGradientCount != 0)
        throw new Exception("TextColorGradientCount", new Exception(TextColorGradientCount.ToString()));

      this.AddItem(4, "INT32", "TextColorGradientCount", TextColorGradientCount.ToString());
      return TextColorGradientCount;
    }


    private Int32 ReadFontCount() {
      Int32 FontCount = this.Reader.ReadInt32();

      if(FontCount < 0 || FontCount > 10000)
        throw new Exception("FontCount", new Exception(FontCount.ToString()));

      this.AddItem(4, "INT32", "FontCount", FontCount.ToString());
      return FontCount;
    }

    private Int32 ReadFontIndex() {
      Int32 FontIndex = this.Reader.ReadInt32();

      if(FontIndex < -1 || FontIndex >= 10000)
        throw new Exception("FontCount", new Exception(FontIndex.ToString()));

      this.AddItem(4, "INT32", "FontIndex", FontIndex.ToString());
      return FontIndex;
    }

    private Int32 ReadFontStringLength() {
      Int32 FontStringLength = this.Reader.ReadInt32();

      if(FontStringLength < 1)
        throw new Exception("FontStringLength", new Exception(FontStringLength.ToString()));

      this.AddItem(4, "INT32", "FontStringLength", FontStringLength.ToString());
      return FontStringLength;
    }

    private String ReadFontString(Int32 FontStringLength) {
      String FontString = String.Empty;

      for(Int32 i = 0; i < FontStringLength; i++)
        FontString += (Char) this.Reader.ReadUInt16();

      this.AddItem(FontStringLength * 2, "WCHAR[]", "FontString", FontString);
      return FontString;
    }


    private Int32 ReadGroupIndex() {
      Int32 GroupIndex = this.Reader.ReadInt32();

      if(GroupIndex < -1 || GroupIndex >= 10000)
        throw new Exception("GroupIndex", new Exception(GroupIndex.ToString()));

      this.AddItem(4, "INT32", "GroupIndex", GroupIndex.ToString());
      return GroupIndex;
    }



    private Int32 ReadGroupCount() {
      Int32 GroupCount = this.Reader.ReadInt32();

      if(GroupCount < 0 || GroupCount > 10000)
        throw new Exception("GroupCount", new Exception(GroupCount.ToString()));

      this.AddItem(4, "INT32", "GroupCount", GroupCount.ToString());
      return GroupCount;
    }

    private Int32 ReadGroupNameLength() {
      Int32 GroupNameLength = this.Reader.ReadInt32();

      if(GroupNameLength <= 0)
        throw new Exception("GroupNameLength", new Exception(GroupNameLength.ToString()));

      this.AddItem(4, "INT32", "GroupNameLength", GroupNameLength + " letters");
      return GroupNameLength;
    }

    private String ReadGroupName(Int32 GroupNameLength) {
      Char[] LetterArray = new Char[GroupNameLength];

      for(Int32 i = 0; i < GroupNameLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String GroupName = new String(LetterArray);

      if(GroupName == null || GroupName == String.Empty)
        throw new Exception("GroupName", new Exception("Empty"));

      this.AddItem(GroupNameLength * 2, "WCHAR[ ]", "GroupName", GroupName);
      return GroupName;
    }

    private Int32 ReadGroupItemCount() {
      Int32 GroupItemCount = this.Reader.ReadInt32();

      if(GroupItemCount < 0)
        throw new Exception("GroupItemCount", new Exception(GroupItemCount.ToString()));

      this.AddItem(4, "INT32", "GroupItemCount", GroupItemCount.ToString());
      return GroupItemCount;
    }

    private void ReadGroupItemMemory(Int32 GroupItemCount) {
      Int32[] ItemArray = new Int32[GroupItemCount * 5];

      for(Int32 i = 0; i < ItemArray.Length; i++)
        ItemArray[i] = this.Reader.ReadInt32();

      this.AddItem(GroupItemCount * 5, "MEMORY", "GroupItemMemory", "", new FileViewerObject(FileViewerObjectType.PicturePaintProject, new FileViewerTDPicturePaintProjectObject(FileViewerTDPicturePaintProjectType.GroupItem, ItemArray)));
    }


    private void ReadBegin() {
      UInt32 IDNumber = this.ReadIDNumber();
      Int64 FileSize = this.ReadFileSize();
      Byte Version = this.ReadVersion();

      Int32 PreviewImageSize = this.ReadPreviewImageSize();
      if(PreviewImageSize > 0) this.ReadPreviewImage(PreviewImageSize);

      Int32 FileCount = this.ReadFileCount();

      for(Int32 i = 0; i < FileCount; i++) {
        Byte FileType = this.ReadFileType();

        Int32 FileNameLength = this.ReadFileNameLength();
        String FileName = this.ReadFileName(FileNameLength);

        Int32 FileExtentionLength = this.ReadFileExtentionLength();
        String FileExtention = this.ReadFileExtention(FileExtentionLength);

        switch(FileType) {
          case 0: { //Image
            Byte ImageFormat = this.ReadImageFormat();

            UInt16 ImageFlags = this.ReadImageFlags();

            Int32 ImagePosX = this.ReadImagePosX();
            Int32 ImagePosY = this.ReadImagePosY();
            Int32 ImageWidth = this.ReadImageWidth();
            Int32 ImageHeight = this.ReadImageHeight();

            Int32 FileImageSize = this.ReadFileImageSize();

            if(FileImageSize > 0)
              this.ReadFileImage(FileImageSize);
            else
              if((ImageFlags & 0x8000) != 0x8000) //ProjectPreview (see: PreviewImage)
                throw new Exception("Wrong FileImage");

            Int32 ToolIndex = this.ReadToolIndex();

            Byte PaintAlpha = this.ReadPaintAlpha();
            UInt32 PaintColor = this.ReadPaintColor("PaintColor");
            UInt32 BackColor = this.ReadPaintColor("BackColor");
            UInt32 ImageColor = this.ReadPaintColor("ImageColor");
            UInt32 ImageTransparentColor = this.ReadPaintColor("ImageTransparentColor");
            UInt32 TextColor = this.ReadPaintColor("TextColor");

            Int32 PaintLineSize = this.ReadLineSize();
            Int32 PaintRoundWidth = this.ReadRoundWidth();
            Int32 PaintRoundHeight = this.ReadRoundHeight();

            Int32 PaintRotationLeverLength = this.ReadRotationLeverLength("PaintRotationLength");
            Double PaintRotationAngle = this.ReadRotationAngle("PaintRotationAngle");
            Int32 ImageRotationLeverLength = this.ReadRotationLeverLength("ImageRotationLength");
            Double ImageRotationAngle = this.ReadRotationAngle("ImageRotationAngle");
            Int32 TextRotationLeverLength = this.ReadRotationLeverLength("TextRotationLength");
            Double TextRotationAngle = this.ReadRotationAngle("TextRotationAngle");

            //Alpha and color gradient settings
            Int32 AlphaArrayCount = this.ReadAlphaArrayCount();

            if(AlphaArrayCount > 0) this.ReadAlphaArray(AlphaArrayCount);

            Int32 ColorArrayCount = this.ReadColorArrayCount();

            if(ColorArrayCount > 0) this.ReadColorArray(ColorArrayCount);

            //User define palette
            Int32 AlphaPaletteCount = this.ReadAlphaPaletteCount();

            if(AlphaPaletteCount > 0) this.ReadAlphaPalette(AlphaPaletteCount);

            Int32 ColorPaletteCount = this.ReadColorPaletteCount();

            if(ColorPaletteCount > 0) this.ReadColorPalette(ColorPaletteCount);

            //User defined gradients
            Int32 AlphaGradientCount = this.ReadAlphaGradientCount();

            if(AlphaGradientCount > 0) this.ReadAlphaGradient(AlphaGradientCount);

            Int32 ColorGradientCount = this.ReadColorGradientCount();

            if(ColorGradientCount > 0) this.ReadColorGradient(ColorGradientCount);

            Int32 AlphaGradientIndex = this.ReadAlphaGradientIndex();
            Int32 ColorGradientIndex = this.ReadColorGradientIndex();

            //User defined text palette (alpha not used)
            Int32 TextAlphaPaletteCount = this.ReadTextAlphaPaletteCount();
            Int32 TextColorPaletteCount = this.ReadTextColorPaletteCount();

            if(TextColorPaletteCount > 0) this.ReadTextColorPalette(TextColorPaletteCount);

            //User defined text gradients (not used)
            Int32 TextAlphaGradientCount = this.ReadTextAlphaGradientCount();
            Int32 TextColorGradientCount = this.ReadTextColorGradientCount();

            //Text fonts
            Int32 FontCount = this.ReadFontCount();

            for(Int32 j = 0; j < FontCount; j++) {
              Int32 FontStringLength = this.ReadFontStringLength();
              String FontString = this.ReadFontString(FontStringLength);
            }

            Int32 FontIndex = this.ReadFontIndex();

            Int32 GroupIndex = this.ReadGroupIndex();
            break;
          }
          case 1: { //Video
            Int32 ImageWidth = this.ReadImageWidth();
            Int32 ImageHeight = this.ReadImageHeight();

            Int64 PlayerDuration = this.ReadPlayerDuration();

            Int32 FileImageSize = this.ReadFileImageSize();
            if(FileImageSize > 0) this.ReadFileImage(FileImageSize);

            Int32 FileMemorySize = this.ReadFileMemorySize();
            Byte[] FileMemory = this.ReadFileMemory(FileMemorySize, FileName, FileExtention);
            break;
          }
          case 2: { //Sound
            Int32 ImageWidth = this.ReadImageWidth();
            Int32 ImageHeight = this.ReadImageHeight();

            Int64 PlayerDuration = this.ReadPlayerDuration();

            Int32 FileImageSize = this.ReadFileImageSize();
            if(FileImageSize > 0) this.ReadFileImage(FileImageSize);

            Int32 FileMemorySize = this.ReadFileMemorySize();
            Byte[] FileMemory = this.ReadFileMemory(FileMemorySize, FileName, FileExtention);
            break;
          }
          case 3: { //Gif Animation (.gif)
            Int32 ImageWidth = this.ReadImageWidth();
            Int32 ImageHeight = this.ReadImageHeight();

            Int64 PlayerDuration = this.ReadPlayerDuration();

            Int32 FileImageSize = this.ReadFileImageSize();
            if(FileImageSize > 0) this.ReadFileImage(FileImageSize);

            Int32 FileMemorySize = this.ReadFileMemorySize();
            Byte[] FileMemory = this.ReadFileMemory(FileMemorySize, FileName, FileExtention);
            break;
          }
          case 4: { //Animation (.tda)
            Int32 ImageWidth = this.ReadImageWidth();
            Int32 ImageHeight = this.ReadImageHeight();

            Int64 PlayerDuration = this.ReadPlayerDuration();
            UInt32 BackColor = this.ReadBackColor();

            Int32 FileImageSize = this.ReadFileImageSize();
            if(FileImageSize > 0) this.ReadFileImage(FileImageSize);

            Int32 FileMemorySize = this.ReadFileMemorySize();
            Byte[] FileMemory = this.ReadFileMemory(FileMemorySize, FileName, FileExtention);
            break;
          }
          case 5: { //Various
            Int32 FileMemorySize = this.ReadFileMemorySize();
            Byte[] FileMemory = this.ReadFileMemory(FileMemorySize, FileName, FileExtention);
            break;
          }
          default: {
            throw new Exception("Wrong FileType");
          }
        }
      }

      Int32 GroupCount = this.ReadGroupCount();

      for(Int32 i = 0; i < GroupCount; i++) {
        Int32 GroupNameLength = this.ReadGroupNameLength();
        String GroupName = this.ReadGroupName(GroupNameLength);

        Int32 GroupItemCount = this.ReadGroupItemCount();

        this.ReadGroupItemMemory(GroupItemCount);
      }

      if(this.Position != this.Size) throw new Exception("Wrong FileSize");
    }

    #endregion


    #region Public Methods

    public List<ListViewItem> Run(IFileViewerProgressBar Parent, String FilePath) {
      this.Parent = Parent;

      try {
        this.Stream = new FileStream(FilePath, FileMode.Open, FileAccess.Read);
        this.Reader = new BinaryReader(this.Stream);

        this.Size = this.Stream.Length;

        this.Parent.BeginProgressBar(this.Size);

        this.AddItemBeginRead();

        this.ReadBegin();
      } catch(Exception e) {
        this.AddItemError(e);

        this.Close();
        return this.ItemList;
      }

      this.AddItemEndOfFile();

      this.Close();
      return this.ItemList;
    }

    #endregion
  }


  public enum FileViewerTDHelpWriterType {
    OutlineLetterIndex = 0,
    OutlineInfoType = 1,
    OutlineInfoIndex = 2,
    FieldTextColor = 3,
    FieldTextFont = 4,
    FieldTextLetter = 5,
    FieldTextInfoType = 6,
    FieldTextInfoColor = 7,
    FieldTextInfoFont = 8,
    FieldTextLetterIndex = 9,
  }

  public class FileViewerTDHelpWriterObject {
    #region Public Fields

    public FileViewerTDHelpWriterType Type;

    public Object Value;

    #endregion

    #region Contructor

    public FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType Type, Object Value) {
      this.Type = Type;
      this.Value = Value;
    }

    #endregion
  }

  public class FileViewerTDHelpWriter {
    #region Public Fields

    public IFileViewerProgressBar Parent;
    public List<ListViewItem> ItemList;
    public BinaryReader Reader;
    public FileStream Stream;
    public String FilePath;
    public Int64 Position;
    public Int64 Size;

    #endregion


    #region Contructor

    public FileViewerTDHelpWriter() {
      this.ItemList = new List<ListViewItem>();
    }

    #endregion


    #region Private Methods

    private void AddItem(Int64 Size, String Type, String Name, String Value) {
      this.ItemList.Add(new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value }));

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }

    private void AddItem(Int64 Size, String Type, String Name, String Value, FileViewerObject Tag) {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value });

      Item.BackColor = Color.FromArgb(240, 255, 245);
      Item.Tag = Tag;

      this.ItemList.Add(Item);

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }


    private void AddItemError(String Value) {
      ListViewItem Item = new ListViewItem(new String[] { "".PadRight(10), "".PadRight(10), "".PadRight(10), "Error".PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }

    private void AddItemError(Exception e) {
      String Name = String.Empty;
      String Value = String.Empty;

      if(e.InnerException != null && e.InnerException.Message != null) {
        if(e.Message != null)
          Name = e.Message;

        Value = e.InnerException.Message;
      } else {
        if(e.Message != null)
          Value = e.Message;
      }

      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "ERROR".PadRight(10), Name.PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }


    private void AddItemBeginRead() {
      ListViewItem Item = new ListViewItem(new String[] { "0".PadRight(10), "0".PadRight(10), "".PadRight(10), "BeginRead".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }

    private void AddItemEndOfFile() {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "".PadRight(10), "EndOfFile".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }


    private void Close() {
      if(this.Stream != null) {
        this.Stream.Close();
        this.Stream = null;
      }

      if(this.Reader != null) {
        this.Reader.Close();
        this.Reader = null;
      }
    }

    #endregion

    #region Private Methods: Format

    private UInt32 ReadIDNumber() {
      UInt32 IDNumber = this.Reader.ReadUInt32();

      if(IDNumber != 0x57484454) //TDHW
        throw new Exception("IDNumber", new Exception("0x" + IDNumber.ToString("X8")));

      this.AddItem(4, "UINT32", "IDNumber", "0x" + IDNumber.ToString("X8") + " hex");
      return IDNumber;
    }

    private Int32 ReadFileCount() {
      Int32 FileCount = this.Reader.ReadInt32();

      if(FileCount < 1)
        throw new Exception("FileCount", new Exception(FileCount.ToString()));

      this.AddItem(4, "INT32", "FileCount", FileCount.ToString());
      return FileCount;
    }

    private Int64 ReadFileSize() {
      Int64 FileSize = this.Reader.ReadInt64();

      if(FileSize != this.Size)
        throw new Exception("FileSize", new Exception(FileSize.ToString()));

      this.AddItem(8, "INT64", "FileSize", FileSize.ToString("#,#") + " bytes");
      return FileSize;
    }

    private Int64 ReadFileMaxSize() {
      Int64 FileMaxSize = this.Reader.ReadInt64();

      if(FileMaxSize < this.Size)
        throw new Exception("FileMaxSize", new Exception(FileMaxSize.ToString()));

      this.AddItem(8, "INT64", "FileMaxSize", FileMaxSize.ToString("#,#") + " bytes");
      return FileMaxSize;
    }

    private Byte ReadVersion() {
      Byte Version = this.Reader.ReadByte();

      if(Version == 0 || Version > 2)
        throw new Exception("Version", new Exception(Version.ToString()));

      this.AddItem(1, "BYTE", "Version", Version.ToString());
      return Version;
    }

    private UInt16 ReadFlags() {
      UInt16 Flags = this.Reader.ReadUInt16();

      //UsePreview = 0x0001, UseIcon = 0x0002, UseInfo = 0x0004, UseTwoLetterLanguage = 0x0008

      if(Flags > 0x000F)
        throw new Exception("Flags", new Exception("0x" + Flags.ToString("X4")));

      this.AddItem(2, "UINT16", "Flags", "0x" + Flags.ToString("X4"));
      return Flags;
    }


    private Byte ReadPreviewImageMode() {
      Byte PreviewImageMode = this.Reader.ReadByte();

      //Normal = 0, Transparent = 1, Alpha = 2

      if(PreviewImageMode > 2)
        throw new Exception("PreviewImageMode", new Exception(PreviewImageMode.ToString()));

      this.AddItem(1, "BYTE", "PreviewImageMode", PreviewImageMode.ToString());
      return PreviewImageMode;
    }

    private Int32 ReadPreviewImageWidth() {
      Int32 PreviewImageWidth = this.Reader.ReadInt32();

      if(PreviewImageWidth < 1 || PreviewImageWidth > 16000)
        throw new Exception("PreviewImageWidth", new Exception(PreviewImageWidth.ToString()));

      this.AddItem(4, "INT32", "PreviewImageWidth", PreviewImageWidth.ToString());
      return PreviewImageWidth;
    }

    private Int32 ReadPreviewImageHeight() {
      Int32 PreviewImageHeight = this.Reader.ReadInt32();

      if(PreviewImageHeight < 1 || PreviewImageHeight > 16000)
        throw new Exception("PreviewImageHeight", new Exception(PreviewImageHeight.ToString()));

      this.AddItem(4, "INT32", "PreviewImageHeight", PreviewImageHeight.ToString());
      return PreviewImageHeight;
    }

    private Boolean ReadPreviewImagePng() {
      Byte PreviewImagePng = this.Reader.ReadByte();

      if(PreviewImagePng > 1)
        throw new Exception("PreviewImagePng", new Exception(PreviewImagePng.ToString()));

      this.AddItem(1, "BYTE", "PreviewImagePng", PreviewImagePng.ToString());
      return PreviewImagePng == 0 ? false : true;
    }

    private Boolean ReadPreviewImageJpeg() {
      Byte PreviewImageJpeg = this.Reader.ReadByte();

      if(PreviewImageJpeg > 1)
        throw new Exception("PreviewImageJpeg", new Exception(PreviewImageJpeg.ToString()));

      this.AddItem(1, "BYTE", "PreviewImageJpeg", PreviewImageJpeg.ToString());
      return PreviewImageJpeg == 0 ? false : true;
    }

    private Byte ReadPreviewImageQuality() {
      Byte PreviewImageQuality = this.Reader.ReadByte();

      if(PreviewImageQuality > 100)
        throw new Exception("PreviewImageQuality", new Exception(PreviewImageQuality.ToString()));

      this.AddItem(1, "BYTE", "PreviewImageQuality", PreviewImageQuality.ToString());
      return PreviewImageQuality;
    }

    private Byte ReadPreviewImageFormat() {
      Byte PreviewImageFormat = this.Reader.ReadByte();

      //Unknown = 0, Bmp = 1, Emf = 2, Exif = 3, Gif = 4, Icon = 5,
      //Jpeg = 6, Png = 7, Tiff = 8, Wmf = 9, Tdp = 10

      if(PreviewImageFormat > 10)
        throw new Exception("PreviewImageFormat", new Exception(PreviewImageFormat.ToString()));

      this.AddItem(1, "BYTE", "PreviewImageFormat", PreviewImageFormat.ToString());
      return PreviewImageFormat;
    }

    private Int32 ReadPreviewImageSize() {
      Int32 PreviewImageSize = this.Reader.ReadInt32();

      if(PreviewImageSize < 0 || PreviewImageSize > this.Size - this.Position - 4)
        throw new Exception("PreviewImageSize", new Exception(PreviewImageSize.ToString()));

      if(PreviewImageSize == 0)
        this.AddItem(4, "INT32", "PreviewImageSize", "0 byte");
      else
        this.AddItem(4, "INT32", "PreviewImageSize", PreviewImageSize.ToString("#,#") + " bytes");

      return PreviewImageSize;
    }

    private void ReadPreviewImage(Int32 PreviewImageSize) {
      try {
        Byte[] PreviewImageArray = this.Reader.ReadBytes(PreviewImageSize);

        Bitmap PreviewImage = new Bitmap(new MemoryStream(PreviewImageArray));

        this.AddItem(PreviewImageSize, "MEMORY", "PreviewImage", "", new FileViewerObject(PreviewImage));
      } catch(Exception e) {
        throw new Exception("PreviewImage", e);
      }
    }


    private Byte ReadIconImageMode() {
      Byte IconImageMode = this.Reader.ReadByte();

      //Normal = 0, Transparent = 1, Alpha = 2

      if(IconImageMode > 2)
        throw new Exception("IconImageMode", new Exception(IconImageMode.ToString()));

      this.AddItem(1, "BYTE", "IconImageMode", IconImageMode.ToString());
      return IconImageMode;
    }

    private Int32 ReadIconImageWidth() {
      Int32 IconImageWidth = this.Reader.ReadInt32();

      if(IconImageWidth < 1 || IconImageWidth > 16000)
        throw new Exception("IconImageWidth", new Exception(IconImageWidth.ToString()));

      this.AddItem(4, "INT32", "IconImageWidth", IconImageWidth.ToString());
      return IconImageWidth;
    }

    private Int32 ReadIconImageHeight() {
      Int32 IconImageHeight = this.Reader.ReadInt32();

      if(IconImageHeight < 1 || IconImageHeight > 16000)
        throw new Exception("IconImageHeight", new Exception(IconImageHeight.ToString()));

      this.AddItem(4, "INT32", "IconImageHeight", IconImageHeight.ToString());
      return IconImageHeight;
    }

    private Int32 ReadIconImageSize() {
      Int32 IconImageSize = this.Reader.ReadInt32();

      if(IconImageSize < 0 || IconImageSize > this.Size - this.Position - 4)
        throw new Exception("IconImageSize", new Exception(IconImageSize.ToString()));

      if(IconImageSize == 0)
        this.AddItem(4, "INT32", "IconImageSize", "0 byte");
      else
        this.AddItem(4, "INT32", "IconImageSize", IconImageSize.ToString("#,#") + " bytes");

      return IconImageSize;
    }

    private void ReadIconImage(Int32 IconImageSize) {
      try {
        Byte[] IconImageArray = this.Reader.ReadBytes(IconImageSize);

        Bitmap IconImage = new Bitmap(new MemoryStream(IconImageArray));

        this.AddItem(IconImageSize, "MEMORY", "IconImage", "", new FileViewerObject(IconImage));
      } catch(Exception e) {
        throw new Exception("IconImage", e);
      }
    }


    private Int32 ReadInfoProducerLength() {
      Int32 InfoProducerLength = this.Reader.ReadInt32();

      if(InfoProducerLength < -1 || InfoProducerLength > 1000)
        throw new Exception("InfoProducerLength", new Exception(InfoProducerLength.ToString()));

      this.AddItem(4, "INT32", "InfoProducerLength", InfoProducerLength.ToString());
      return InfoProducerLength;
    }

    private String ReadInfoProducer(Int32 InfoProducerLength) {
      if(InfoProducerLength == -1) {
        this.AddItem(0, "WCHAR[ ]", "InfoProducer", "Null");
        return null;
      }

      if(InfoProducerLength == 0) {
        this.AddItem(0, "WCHAR[ ]", "InfoProducer", "Empty");
        return String.Empty;
      }

      Char[] LetterArray = new Char[InfoProducerLength];

      for(Int32 i = 0; i < InfoProducerLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String InfoProducer = new String(LetterArray);

      if(InfoProducer == null || InfoProducer == String.Empty)
        throw new Exception("InfoProducer", new Exception("Empty"));

      this.AddItem(InfoProducerLength * 2, "WCHAR[ ]", "InfoProducer", InfoProducer);
      return InfoProducer;
    }

    private Int32 ReadInfoInternetLength() {
      Int32 InfoInternetLength = this.Reader.ReadInt32();

      if(InfoInternetLength < -1 || InfoInternetLength > 1000)
        throw new Exception("InfoInternetLength", new Exception(InfoInternetLength.ToString()));

      this.AddItem(4, "INT32", "InfoInternetLength", InfoInternetLength.ToString());
      return InfoInternetLength;
    }

    private String ReadInfoInternet(Int32 InfoInternetLength) {
      if(InfoInternetLength == -1) {
        this.AddItem(0, "WCHAR[ ]", "InfoInternet", "Null");
        return null;
      }

      if(InfoInternetLength == 0) {
        this.AddItem(0, "WCHAR[ ]", "InfoInternet", "Empty");
        return String.Empty;
      }

      Char[] LetterArray = new Char[InfoInternetLength];

      for(Int32 i = 0; i < InfoInternetLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String InfoInternet = new String(LetterArray);

      if(InfoInternet == null || InfoInternet == String.Empty)
        throw new Exception("InfoInternet", new Exception("Empty"));

      this.AddItem(InfoInternetLength * 2, "WCHAR[ ]", "InfoInternet", InfoInternet);
      return InfoInternet;
    }

    private Int32 ReadInfoMailLength() {
      Int32 InfoMailLength = this.Reader.ReadInt32();

      if(InfoMailLength < -1 || InfoMailLength > 1000)
        throw new Exception("InfoMailLength", new Exception(InfoMailLength.ToString()));

      this.AddItem(4, "INT32", "InfoMailLength", InfoMailLength.ToString());
      return InfoMailLength;
    }

    private String ReadInfoMail(Int32 InfoMailLength) {
      if(InfoMailLength == -1) {
        this.AddItem(0, "WCHAR[ ]", "InfoMail", "Null");
        return null;
      }

      if(InfoMailLength == 0) {
        this.AddItem(0, "WCHAR[ ]", "InfoMail", "Empty");
        return String.Empty;
      }

      Char[] LetterArray = new Char[InfoMailLength];

      for(Int32 i = 0; i < InfoMailLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String InfoMail = new String(LetterArray);

      if(InfoMail == null || InfoMail == String.Empty)
        throw new Exception("InfoMail", new Exception("Empty"));

      this.AddItem(InfoMailLength * 2, "WCHAR[ ]", "InfoMail", InfoMail);
      return InfoMail;
    }


    private UInt32 ReadSettingBackColor() {
      UInt32 SettingBackColor = this.Reader.ReadUInt32();

      if(SettingBackColor < 0xFF000000 && SettingBackColor != 0)
        throw new Exception("SettingBackColor", new Exception("0x" + SettingBackColor.ToString("X8")));

      this.AddItem(4, "UINT32", "SettingBackColor", "0x" + SettingBackColor.ToString("X8"));
      return SettingBackColor;
    }

    private UInt32 ReadSettingWindowColor() {
      UInt32 SettingWindowColor = this.Reader.ReadUInt32();

      if(SettingWindowColor < 0xFF000000 && SettingWindowColor != 0)
        throw new Exception("SettingWindowColor", new Exception("0x" + SettingWindowColor.ToString("X8")));

      this.AddItem(4, "UINT32", "SettingWindowColor", "0x" + SettingWindowColor.ToString("X8"));
      return SettingWindowColor;
    }

    private Int32 ReadSettingBorderSize(String Name) {
      Int32 SettingBorderSize = this.Reader.ReadInt32();

      if(SettingBorderSize < 0 || SettingBorderSize > 1000)
        throw new Exception(Name, new Exception(SettingBorderSize.ToString()));

      this.AddItem(4, "INT32", Name, SettingBorderSize.ToString());
      return SettingBorderSize;
    }

    private Double ReadSettingDocumentFactor() {
      Double SettingDocumentFactor = this.Reader.ReadDouble();

      if(SettingDocumentFactor < 1.0 || SettingDocumentFactor > 4.0)
        throw new Exception("SettingDocumentFactor", new Exception(SettingDocumentFactor.ToString()));

      this.AddItem(8, "DOUBLE", "SettingDocumentFactor", SettingDocumentFactor.ToString());
      return SettingDocumentFactor;
    }


    private Int32 ReadLanguageCount() {
      Int32 LanguageCount = this.Reader.ReadInt32();

      if(LanguageCount < 0 || LanguageCount > 200)
        throw new Exception("LanguageCount", new Exception(LanguageCount.ToString()));

      this.AddItem(4, "INT32", "LanguageCount", LanguageCount.ToString());
      return LanguageCount;
    }

    private Byte ReadLanguageLetterLength() {
      Byte LanguageLetterLength = this.Reader.ReadByte();

      if(LanguageLetterLength != 2)
        throw new Exception("LanguageLetterLength", new Exception(LanguageLetterLength.ToString()));

      this.AddItem(1, "BYTE", "LanguageLetterLength", LanguageLetterLength.ToString());
      return LanguageLetterLength;
    }

    private String ReadLanguageLetter(Int32 LanguageLetterLength) {
      Char[] LetterArray = new Char[LanguageLetterLength];

      for(Int32 i = 0; i < LanguageLetterLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadByte();

      String LanguageLetter = new String(LetterArray);

      if(LanguageLetter == null || LanguageLetter == String.Empty)
        throw new Exception("LanguageLetter", new Exception("Empty"));

      this.AddItem(LanguageLetterLength, "CHAR[ ]", "LanguageLetter", LanguageLetter);
      return LanguageLetter;
    }


    private Int32 ReadLetterImageSize() {
      Int32 LetterImageSize = this.Reader.ReadInt32();

      if(LetterImageSize < -1 || LetterImageSize > this.Size - this.Position - 4)
        throw new Exception("LetterImageSize", new Exception(LetterImageSize.ToString()));

      if(LetterImageSize == -1)
        this.AddItem(4, "INT32", "LetterImageSize", "Null");
      else
        if(LetterImageSize == 0)
          this.AddItem(4, "INT32", "LetterImageSize", "0 byte");
        else
          this.AddItem(4, "INT32", "LetterImageSize", LetterImageSize.ToString("#,#") + " bytes");

      return LetterImageSize;
    }

    private void ReadLetterImage(Int32 LetterImageSize) {
      try {
        Byte[] LetterImageArray = this.Reader.ReadBytes(LetterImageSize);

        Bitmap LetterImage = new Bitmap(new MemoryStream(LetterImageArray));

        this.AddItem(LetterImageSize, "MEMORY", "LetterImage", "", new FileViewerObject(LetterImage));
      } catch(Exception e) {
        throw new Exception("LetterImage", e);
      }
    }

    private Boolean ReadLetterUseInfo() {
      Byte LetterInfo = this.Reader.ReadByte();

      if(LetterInfo > 1)
        throw new Exception("LetterInfo", new Exception(LetterInfo.ToString()));

      this.AddItem(1, "BYTE", "LetterInfo", LetterInfo.ToString());
      return LetterInfo == 0 ? false : true;
    }

    private Int32 ReadLetterFontCount() {
      Int32 LetterFontCount = this.Reader.ReadInt32();

      if(LetterFontCount < 1 || LetterFontCount > 10000)
        throw new Exception("LetterFontCount", new Exception(LetterFontCount.ToString()));

      this.AddItem(4, "INT32", "LetterFontCount", LetterFontCount.ToString());
      return LetterFontCount;
    }


    private Int32 ReadFontNameLength() {
      Int32 FontNameLength = this.Reader.ReadInt32();

      if(FontNameLength < 1 || FontNameLength > 200)
        throw new Exception("FontNameLength", new Exception(FontNameLength.ToString()));

      this.AddItem(4, "INT32", "FontNameLength", FontNameLength.ToString());
      return FontNameLength;
    }

    private String ReadFontName(Int32 FontNameLength) {
      Char[] LetterArray = new Char[FontNameLength];

      for(Int32 i = 0; i < FontNameLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String FontName = new String(LetterArray);

      if(FontName == null || FontName == String.Empty)
        throw new Exception("FontName", new Exception("Empty"));

      this.AddItem(FontNameLength * 2, "WCHAR[ ]", "FontName", FontName);
      return FontName;
    }

    private Single ReadFontSize() {
      Single FontSize = this.Reader.ReadSingle();

      if(FontSize < 1 || FontSize > 16000)
        throw new Exception("FontSize", new Exception(FontSize.ToString()));

      this.AddItem(4, "FLOAT", "FontSize", FontSize.ToString());
      return FontSize;
    }

    private Byte ReadFontStyle() {
      Byte FontStyle = this.Reader.ReadByte();

      //Regular = 0x00 (400), Bold = 0x01 (700), Italic = 0x02, Underline = 0x04, Strikeout = 0x08

      if(FontStyle > 0x0F)
        throw new Exception("FontStyle", new Exception("0x" + FontStyle.ToString("X2")));

      this.AddItem(1, "BYTE", "FontStyle", "0x" + FontStyle.ToString("X2"));
      return FontStyle;
    }

    private Byte ReadFontUnit() {
      Byte FontUnit = this.Reader.ReadByte();

      //World = 0x00, Display = 0x01, Pixel = 0x02, Point = 0x03, 
      //Inch = 0x04, Document = 0x05, Millimeter = 0x06

      if(FontUnit > 0x06)
        throw new Exception("FontUnit", new Exception(FontUnit.ToString()));

      this.AddItem(1, "BYTE", "FontUnit", FontUnit.ToString());
      return FontUnit;
    }

    private Int16 ReadFontHeight() {
      Int16 FontHeight = this.Reader.ReadInt16();

      if(FontHeight < 1 || FontHeight > 16000)
        throw new Exception("FontHeight", new Exception(FontHeight.ToString()));

      this.AddItem(2, "INT16", "FontHeight", FontHeight.ToString());
      return FontHeight;
    }

    private Int16 ReadFontAscent() {
      Int16 FontAscent = this.Reader.ReadInt16();

      if(FontAscent < 1 || FontAscent > 16000)
        throw new Exception("FontAscent", new Exception(FontAscent.ToString()));

      this.AddItem(2, "INT16", "FontAscent", FontAscent.ToString());
      return FontAscent;
    }


    private Int32 ReadLetterCount() {
      Int32 LetterCount = this.Reader.ReadInt32();

      if(LetterCount < 1 || LetterCount > 65536)
        throw new Exception("LetterCount", new Exception(LetterCount.ToString()));

      this.AddItem(4, "INT32", "LetterCount", LetterCount.ToString());
      return LetterCount;
    }

    private Char ReadLetter() {
      UInt16 Letter = this.Reader.ReadUInt16();

      if(Char.IsLetter((Char) Letter) || Char.IsNumber((Char) Letter))
        this.AddItem(2, "WCHAR", "Letter", ((Char) Letter).ToString());
      else
        this.AddItem(2, "WCHAR", "Letter", "0x" + Letter.ToString("X4"));

      return (Char) Letter;
    }

    private Int16 ReadLetterWidth(String Name) {
      Int16 LetterWidth = this.Reader.ReadInt16();

      if(LetterWidth < -16000 || LetterWidth > 16000)
        throw new Exception(Name, new Exception(LetterWidth.ToString()));

      this.AddItem(2, "INT16", Name, LetterWidth.ToString());
      return LetterWidth;
    }

    private Int32 ReadLetterImagePosition() {
      Int32 LetterImagePosition = this.Reader.ReadInt32();

      if(LetterImagePosition < -1)
        throw new Exception("LetterImagePosition", new Exception(LetterImagePosition.ToString()));

      this.AddItem(4, "INT32", "LetterImagePosition", LetterImagePosition.ToString());
      return LetterImagePosition;
    }

    private Int16 ReadLetterImageWidth() {
      Int16 LetterImageWidth = this.Reader.ReadInt16();

      if(LetterImageWidth < 1 || LetterImageWidth > 16000)
        throw new Exception("LetterImageWidth", new Exception(LetterImageWidth.ToString()));

      this.AddItem(2, "INT16", "LetterImageWidth", LetterImageWidth.ToString());
      return LetterImageWidth;
    }

    private Int16 ReadLetterImageHeight() {
      Int16 LetterImageHeight = this.Reader.ReadInt16();

      if(LetterImageHeight < 1 || LetterImageHeight > 16000)
        throw new Exception("LetterImageHeight", new Exception(LetterImageHeight.ToString()));

      this.AddItem(2, "INT16", "LetterImageHeight", LetterImageHeight.ToString());
      return LetterImageHeight;
    }


    private Double ReadLetterWidth() {
      Double LetterWidth = this.Reader.ReadDouble();

      if(LetterWidth < 0 || LetterWidth > 3000)
        throw new Exception("LetterWidth", new Exception(LetterWidth.ToString()));

      this.AddItem(8, "DOUBLE", "LetterWidth", LetterWidth.ToString());
      return LetterWidth;
    }

    private Int32 ReadLetterImagePosition(String Name) {
      Int32 LetterImagePosition = this.Reader.ReadInt32();

      if(LetterImagePosition < -1)
        throw new Exception(Name, new Exception(LetterImagePosition.ToString()));

      this.AddItem(4, "INT32", Name, LetterImagePosition.ToString());
      return LetterImagePosition;
    }

    private Int32 ReadLetterImageWidth(String Name) {
      Int32 LetterImageWidth = this.Reader.ReadInt32();

      if(LetterImageWidth < 1 || LetterImageWidth > 3000)
        throw new Exception(Name, new Exception(LetterImageWidth.ToString()));

      this.AddItem(4, "INT32", Name, LetterImageWidth.ToString());
      return LetterImageWidth;
    }

    private Int32 ReadLetterImageHeight(String Name) {
      Int32 LetterImageHeight = this.Reader.ReadInt32();

      if(LetterImageHeight < 1 || LetterImageHeight > 3000)
        throw new Exception(Name, new Exception(LetterImageHeight.ToString()));

      this.AddItem(4, "INT32", Name, LetterImageHeight.ToString());
      return LetterImageHeight;
    }

    private Double ReadLetterImageX(String Name) {
      Double LetterImageX = this.Reader.ReadDouble();

      if(LetterImageX < -3000 || LetterImageX > 3000)
        throw new Exception(Name, new Exception(LetterImageX.ToString()));

      this.AddItem(8, "DOUBLE", Name, LetterImageX.ToString());
      return LetterImageX;
    }

    private Int32 ReadLetterImageY(String Name) {
      Int32 LetterImageY = this.Reader.ReadInt32();

      if(LetterImageY < -3000 || LetterImageY > 3000)
        throw new Exception(Name, new Exception(LetterImageY.ToString()));

      this.AddItem(4, "INT32", Name, LetterImageY.ToString());
      return LetterImageY;
    }


    private Int32 ReadImageCount(String Name) {
      Int32 ImageCount = this.Reader.ReadInt32();

      if(ImageCount < 0)
        throw new Exception(Name, new Exception(ImageCount.ToString()));

      this.AddItem(4, "INT32", Name, ImageCount.ToString());
      return ImageCount;
    }

    private Byte ReadImageMode(String Name) {
      Byte ImageMode = this.Reader.ReadByte();

      //Normal = 0, Transparent = 1, Alpha = 2

      if(ImageMode > 2)
        throw new Exception(Name, new Exception(ImageMode.ToString()));

      this.AddItem(1, "BYTE", Name, ImageMode.ToString());
      return ImageMode;
    }

    private Int32 ReadImageWidth(String Name) {
      Int32 ImageWidth = this.Reader.ReadInt32();

      if(ImageWidth < 1 || ImageWidth > 32000)
        throw new Exception(Name, new Exception(ImageWidth.ToString()));

      this.AddItem(4, "INT32", Name, ImageWidth.ToString());
      return ImageWidth;
    }

    private Int32 ReadImageHeight(String Name) {
      Int32 ImageHeight = this.Reader.ReadInt32();

      if(ImageHeight < 1 || ImageHeight > 32000)
        throw new Exception(Name, new Exception(ImageHeight.ToString()));

      this.AddItem(4, "INT32", Name, ImageHeight.ToString());
      return ImageHeight;
    }

    private Int32 ReadImageSize(String Name) {
      Int32 ImageSize = this.Reader.ReadInt32();

      if(ImageSize < 0 || ImageSize > this.Size - this.Position - 4)
        throw new Exception(Name, new Exception(ImageSize.ToString()));

      if(ImageSize == 0)
        this.AddItem(4, "INT32", Name, "0 byte");
      else
        this.AddItem(4, "INT32", Name, ImageSize.ToString("#,#") + " bytes");

      return ImageSize;
    }

    private void ReadImage(String Name, Int32 ImageSize) {
      try {
        Byte[] ImageArray = this.Reader.ReadBytes(ImageSize);

        Bitmap Image = FileViewerTDPicture.LoadImage(new MemoryStream(ImageArray));

        if(Image == null) Image = new Bitmap(new MemoryStream(ImageArray));

        this.AddItem(ImageSize, "MEMORY", Name, "", new FileViewerObject(Image));
      } catch(Exception e) {
        throw new Exception(Name, e);
      }
    }

    private Int32 ReadImageItemWidth(String Name) {
      Int32 ImageItemWidth = this.Reader.ReadInt32();

      if(ImageItemWidth < 1 || ImageItemWidth > 32000)
        throw new Exception(Name, new Exception(ImageItemWidth.ToString()));

      this.AddItem(4, "INT32", Name, ImageItemWidth.ToString());
      return ImageItemWidth;
    }

    private Int32 ReadImageItemCount(String Name) {
      Int32 ImageItemCount = this.Reader.ReadInt32();

      if(ImageItemCount < 1 || ImageItemCount > 32000)
        throw new Exception(Name, new Exception(ImageItemCount.ToString()));

      this.AddItem(4, "INT32", Name, ImageItemCount.ToString());
      return ImageItemCount;
    }

    private Int32 ReadImagePathLength(String Name) {
      Int32 ImagePathLength = this.Reader.ReadInt32();

      if(ImagePathLength < -1 || ImagePathLength > 32000)
        throw new Exception(Name, new Exception(ImagePathLength.ToString()));

      this.AddItem(4, "INT32", Name, ImagePathLength.ToString());
      return ImagePathLength;
    }

    private String ReadImagePath(String Name, Int32 ImagePathLength) {
      Char[] LetterArray = new Char[ImagePathLength];

      for(Int32 i = 0; i < ImagePathLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String ImagePath = new String(LetterArray);

      if(ImagePath == null || ImagePath == String.Empty)
        throw new Exception(Name, new Exception("Empty"));

      this.AddItem(ImagePathLength * 2, "WCHAR[ ]", Name, ImagePath);
      return ImagePath;
    }


    private Int32 ReadPlayerFileCount() {
      Int32 PlayerFileCount = this.Reader.ReadInt32();

      if(PlayerFileCount < 0)
        throw new Exception("PlayerFileCount", new Exception(PlayerFileCount.ToString()));

      this.AddItem(4, "INT32", "PlayerFileCount", PlayerFileCount.ToString());
      return PlayerFileCount;
    }

    private Byte ReadPlayerFileType() {
      Byte PlayerFileType = this.Reader.ReadByte();

      //Sound = 0, SoundImage = 1, Video = 2, VideoAudio = 3, Gif = 4, TDAnimation = 5,

      if(PlayerFileType > 5)
        throw new Exception("PlayerFileType", new Exception(PlayerFileType.ToString()));

      this.AddItem(1, "BYTE", "PlayerFileType", PlayerFileType.ToString());
      return PlayerFileType;
    }

    private Int64 ReadPlayerFileDuration() {
      Int64 PlayerDuration = this.Reader.ReadInt64();

      if(PlayerDuration < 0)
        throw new Exception("PlayerDuration", new Exception(PlayerDuration.ToString()));

      this.AddItem(8, "INT64", "PlayerDuration", PlayerDuration.ToString());
      return PlayerDuration;
    }

    private Int32 ReadPlayerDisplayWidth() {
      Int32 PlayerDisplayWidth = this.Reader.ReadInt32();

      if(PlayerDisplayWidth < 0 || PlayerDisplayWidth > 16000)
        throw new Exception("PlayerDisplayWidth", new Exception(PlayerDisplayWidth.ToString()));

      this.AddItem(4, "INT32", "PlayerDisplayWidth", PlayerDisplayWidth.ToString());
      return PlayerDisplayWidth;
    }

    private Int32 ReadPlayerDisplayHeight() {
      Int32 PlayerDisplayHeight = this.Reader.ReadInt32();

      if(PlayerDisplayHeight < 0 || PlayerDisplayHeight > 16000)
        throw new Exception("PlayerDisplayHeight", new Exception(PlayerDisplayHeight.ToString()));

      this.AddItem(4, "INT32", "PlayerDisplayHeight", PlayerDisplayHeight.ToString());
      return PlayerDisplayHeight;
    }

    private Int32 ReadPlayerFileSize() {
      Int32 PlayerFileSize = this.Reader.ReadInt32();

      if(PlayerFileSize < 0 || PlayerFileSize > this.Size - this.Position - 4)
        throw new Exception("PlayerFileSize", new Exception(PlayerFileSize.ToString()));

      if(PlayerFileSize == 0)
        this.AddItem(4, "INT32", "PlayerFileSize", "0 byte");
      else
        this.AddItem(4, "INT32", "PlayerFileSize", PlayerFileSize.ToString("#,#") + " bytes");

      return PlayerFileSize;
    }

    private Byte[] ReadPlayerFile(Byte PlayerFileType, Int32 FileSize) {
      Byte[] FileMemory = this.Reader.ReadBytes(FileSize);
      String FileName = String.Empty;

      switch(PlayerFileType) {
        case 0: {
          FileName = "Sound.wav";
          break;
        }
        case 1: {
          FileName = "Sound.wav";
          break;
        }
        case 2: {
          FileName = "Video.avi";
          break;
        }
        case 3: {
          FileName = "Video.avi";
          break;
        }
        case 4: {
          FileName = "Animation.gif";
          break;
        }
        case 5: {
          FileName = "Animation.tda";
          break;
        }
      }

      this.AddItem(FileSize, "BYTE[ ]", "PlayerFile", FileName, new FileViewerObject(FileViewerObjectType.PicturePaintFile, new FileViewerTDPicturePaintFileObject(FileName, FileMemory)));
      return FileMemory;
    }


    private Int32 ReadOutlineCount() {
      Int32 OutlineCount = this.Reader.ReadInt32();

      if(OutlineCount < 0)
        throw new Exception("OutlineCount", new Exception(OutlineCount.ToString()));

      this.AddItem(4, "INT32", "OutlineCount", OutlineCount.ToString());
      return OutlineCount;
    }

    private UInt16 ReadOutlineFlags() {
      UInt16 OutlineFlags = this.Reader.ReadUInt16();

      //UseRound = 0x0001, UseShadow = 0x0002, UseBorder = 0x0004, UseBackground = 0x0008
      //UseBackgroundStyle = 0x0010, UseLanguage = 0x0020, UsePage = 0x0040

      if(OutlineFlags > 0x007F)
        throw new Exception("OutlineFlags", new Exception("0x" + OutlineFlags.ToString("X4")));

      this.AddItem(2, "UINT16", "OutlineFlags", "0x" + OutlineFlags.ToString("X4"));
      return OutlineFlags;
    }

    private Int32 ReadOutlineDistance(String Name) {
      Int32 OutlineDistance = this.Reader.ReadInt32();

      if(OutlineDistance < 0 || OutlineDistance > 10000)
        throw new Exception(Name, new Exception(OutlineDistance.ToString()));

      this.AddItem(4, "INT32", Name, OutlineDistance.ToString());
      return OutlineDistance;
    }

    private Int32 ReadOutlineFrame(String Name) {
      Int32 OutlineFrame = this.Reader.ReadInt32();

      if(OutlineFrame < 0 || OutlineFrame > 1000)
        throw new Exception(Name, new Exception(OutlineFrame.ToString()));

      this.AddItem(4, "INT32", Name, OutlineFrame.ToString());
      return OutlineFrame;
    }

    private Int32 ReadOutlineExpandedLocation(String Name) {
      Int32 OutlineExpandedLocation = this.Reader.ReadInt32();

      if(OutlineExpandedLocation < 0 || OutlineExpandedLocation > 10000)
        throw new Exception(Name, new Exception(OutlineExpandedLocation.ToString()));

      this.AddItem(4, "INT32", Name, OutlineExpandedLocation.ToString());
      return OutlineExpandedLocation;
    }

    private Int32 ReadOutlineExpandedCount() {
      Int32 OutlineExpandedCount = this.Reader.ReadInt32();

      if(OutlineExpandedCount < 2 || OutlineExpandedCount > 999)
        throw new Exception("OutlineExpandedCount", new Exception(OutlineExpandedCount.ToString()));

      this.AddItem(4, "INT32", "OutlineExpandedCount", OutlineExpandedCount.ToString());
      return OutlineExpandedCount;
    }

    private Int32 ReadOutlineExpandedIndex() {
      Int32 OutlineExpandedIndex = this.Reader.ReadInt32();

      if(OutlineExpandedIndex < -1)
        throw new Exception("OutlineExpandedIndex", new Exception(OutlineExpandedIndex.ToString()));

      this.AddItem(4, "INT32", "OutlineExpandedIndex", OutlineExpandedIndex.ToString());
      return OutlineExpandedIndex;
    }

    private Int32 ReadOutlineExpandedIndex(Int32 ExpandedImageCount) {
      Int32 OutlineExpandedIndex = this.Reader.ReadInt32();

      if(OutlineExpandedIndex < -1 || OutlineExpandedIndex >= ExpandedImageCount)
        throw new Exception("OutlineExpandedIndex", new Exception(OutlineExpandedIndex.ToString()));

      this.AddItem(4, "INT32", "OutlineExpandedIndex", OutlineExpandedIndex.ToString());
      return OutlineExpandedIndex;
    }

    private Int32 ReadOutlineCurveLocation(String Name) {
      Int32 OutlineCurveLocation = this.Reader.ReadInt32();

      if(OutlineCurveLocation < 0 || OutlineCurveLocation > 100)
        throw new Exception(Name, new Exception(OutlineCurveLocation.ToString()));

      this.AddItem(4, "INT32", Name, OutlineCurveLocation.ToString());
      return OutlineCurveLocation;
    }

    private UInt32 ReadOutlineBackColor(String Name) {
      UInt32 OutlineBackColor = this.Reader.ReadUInt32();

      if(OutlineBackColor < 0xFF000000)
        throw new Exception(Name, new Exception("0x" + OutlineBackColor.ToString("X8")));

      this.AddItem(4, "UINT32", Name, "0x" + OutlineBackColor.ToString("X8"));
      return OutlineBackColor;
    }

    private Int32 ReadOutlineShadowLocation(String Name) {
      Int32 OutlineShadowLocation = this.Reader.ReadInt32();

      if(OutlineShadowLocation < 0 || OutlineShadowLocation > 1000)
        throw new Exception(Name, new Exception(OutlineShadowLocation.ToString()));

      this.AddItem(4, "INT32", Name, OutlineShadowLocation.ToString());
      return OutlineShadowLocation;
    }


    private Int32 ReadOutlineTextWidth() {
      Int32 OutlineTextWidth = this.Reader.ReadInt32();

      if(OutlineTextWidth < 1 || OutlineTextWidth > 16000)
        throw new Exception("OutlineTextWidth", new Exception(OutlineTextWidth.ToString()));

      this.AddItem(4, "INT32", "OutlineTextWidth", OutlineTextWidth.ToString());
      return OutlineTextWidth;
    }

    private Int32 ReadOutlineTextHeight() {
      Int32 OutlineTextHeight = this.Reader.ReadInt32();

      if(OutlineTextHeight < 1 || OutlineTextHeight > 16000)
        throw new Exception("OutlineTextHeight", new Exception(OutlineTextHeight.ToString()));

      this.AddItem(4, "INT32", "OutlineTextHeight", OutlineTextHeight.ToString());
      return OutlineTextHeight;
    }

    private Int32 ReadOutlineTextLength() {
      Int32 OutlineTextLength = this.Reader.ReadInt32();

      if(OutlineTextLength < 0 || OutlineTextLength > 1000)
        throw new Exception("OutlineTextLength", new Exception(OutlineTextLength.ToString()));

      this.AddItem(4, "INT32", "OutlineTextLength", OutlineTextLength.ToString());
      return OutlineTextLength;
    }

    private Int32 ReadOutlineFontIndex(Int32 LetterFontCount) {
      Int32 OutlineFontIndex = this.Reader.ReadInt32();

      if(OutlineFontIndex < 0 || OutlineFontIndex >= LetterFontCount)
        throw new Exception("OutlineFontIndex", new Exception(OutlineFontIndex.ToString()));

      this.AddItem(4, "INT32", "OutlineFontIndex", OutlineFontIndex.ToString());
      return OutlineFontIndex;
    }

    private Int32 ReadOutlineLetterIndexCount() {
      Int32 OutlineLetterIndexCount = this.Reader.ReadInt32();

      if(OutlineLetterIndexCount < 0)
        throw new Exception("OutlineLetterIndexCount", new Exception(OutlineLetterIndexCount.ToString()));

      this.AddItem(4, "INT32", "OutlineLetterIndexCount", OutlineLetterIndexCount.ToString());
      return OutlineLetterIndexCount;
    }

    private void ReadOutlineLetterIndexMemory(Int32 OutlineLetterIndexCount) {
      Int32[] OutlineLetterIndexMemory = new Int32[OutlineLetterIndexCount];

      for(Int32 i = 0; i < OutlineLetterIndexCount; i++)
        OutlineLetterIndexMemory[i] = this.Reader.ReadInt32();

      this.AddItem(OutlineLetterIndexCount * 4, "INT32[ ]", "OutlineLetterIndexMemory", "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType.OutlineLetterIndex, OutlineLetterIndexMemory)));
    }

    private Int32 ReadOutlineInfoTypeSize() {
      Int32 OutlineInfoTypeSize = this.Reader.ReadInt32();

      if(OutlineInfoTypeSize < -1 || OutlineInfoTypeSize > this.Size - this.Position - 4)
        throw new Exception("OutlineInfoTypeSize", new Exception(OutlineInfoTypeSize.ToString()));

      if(OutlineInfoTypeSize == -1)
        this.AddItem(4, "INT32", "OutlineInfoTypeSize", "-1 (empty)");
      else
        if(OutlineInfoTypeSize == 0)
          this.AddItem(4, "INT32", "OutlineInfoTypeSize", "0 byte");
        else
          this.AddItem(4, "INT32", "OutlineInfoTypeSize", OutlineInfoTypeSize.ToString("#,#") + " bytes");

      return OutlineInfoTypeSize;
    }

    private void ReadOutlineInfoTypeMemoryV1(Int32 OutlineInfoTypeSize) {
      Byte[] OutlineInfoTypeMemory = this.Reader.ReadBytes(OutlineInfoTypeSize);

      //Info(0): 0x00 Letter, 0x01 Seperator, 0x02 Space, 0x03 Control, 0x04 NewLine (Flag)
      //Info(1): Font index
      //Info(2): Color index

      //Not Used
      //this.AddItem(OutlineInfoTypeSize, "BYTE[ ]", "OutlineInfoTypeMemory", "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType.OutlineInfoType, OutlineInfoTypeMemory)));
    }

    private void ReadOutlineInfoTypeMemoryV2(Int32 OutlineInfoTypeSize, Int32 OutlineTextLength) {
      Byte[] OutlineInfoTypeMemory = this.Reader.ReadBytes(OutlineInfoTypeSize);

      //Info(0): 0x00 Letter, 0x01 Seperator, 0x02 Space, 0x03 Control
      //Info(1): Font index
      //Info(2): Color index

      this.AddItem(OutlineInfoTypeSize, "BYTE[ ]", "OutlineInfoTypeMemory", "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType.OutlineInfoType, new FileViewerTDCompressTextObject(null, FileViewerTDCompressTextType.Info, OutlineInfoTypeMemory, 4, OutlineTextLength))));
    }

    private Int32 ReadOutlineInfoIndexCount() {
      Int32 OutlineInfoIndexCount = this.Reader.ReadInt32();

      if(OutlineInfoIndexCount < 0)
        throw new Exception("OutlineInfoIndexCount", new Exception(OutlineInfoIndexCount.ToString()));

      this.AddItem(4, "INT32", "OutlineInfoIndexCount", OutlineInfoIndexCount.ToString());
      return OutlineInfoIndexCount;
    }

    private Int32 ReadOutlineInfoIndexSize() {
      Int32 OutlineInfoIndexSize = this.Reader.ReadInt32();

      if(OutlineInfoIndexSize < 0 || OutlineInfoIndexSize > this.Size - this.Position - 4)
        throw new Exception("OutlineInfoIndexSize", new Exception(OutlineInfoIndexSize.ToString()));

      if(OutlineInfoIndexSize == 0)
        this.AddItem(4, "INT32", "OutlineInfoIndexSize", "0 byte");
      else
        this.AddItem(4, "INT32", "OutlineInfoIndexSize", OutlineInfoIndexSize.ToString("#,#") + " bytes");

      return OutlineInfoIndexSize;
    }

    private void ReadOutlineInfoIndexMemory(Int32 OutlineInfoIndexSize, Int32 OutlineTextLength, Int32 OutlineInfoIndexCount) {
      Byte[] OutlineInfoIndexMemory = this.Reader.ReadBytes(OutlineInfoIndexSize);

      this.AddItem(OutlineInfoIndexSize, "BYTE[ ]", "OutlineInfoIndexMemory", "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType.OutlineInfoIndex, new FileViewerTDCompressTextObject(null, FileViewerTDCompressTextType.Index, OutlineInfoIndexMemory, OutlineInfoIndexCount, OutlineTextLength))));
    }

    private Int32 ReadOutlineTextStringLength() {
      Int32 OutlineTextStringLength = this.Reader.ReadInt32();

      if(OutlineTextStringLength < -1 || OutlineTextStringLength > 100)
        throw new Exception("OutlineTextStringLength", new Exception(OutlineTextStringLength.ToString()));

      this.AddItem(4, "INT32", "OutlineTextStringLength", OutlineTextStringLength.ToString());
      return OutlineTextStringLength;
    }

    private String ReadOutlineTextString(Int32 OutlineTextStringLength) {
      Char[] LetterArray = new Char[OutlineTextStringLength];

      for(Int32 i = 0; i < OutlineTextStringLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String OutlineTextString = new String(LetterArray);

      if(OutlineTextString == null || OutlineTextString == String.Empty)
        throw new Exception("OutlineTextString", new Exception("Empty"));

      this.AddItem(OutlineTextStringLength * 2, "WCHAR[ ]", "OutlineTextString", OutlineTextString);
      return OutlineTextString;
    }


    private Int32 ReadOutlineChildCount() {
      Int32 OutlineChildCount = this.Reader.ReadInt32();

      if(OutlineChildCount < 0)
        throw new Exception("OutlineChildCount", new Exception(OutlineChildCount.ToString()));

      this.AddItem(4, "INT32", "OutlineChildCount", OutlineChildCount.ToString());
      return OutlineChildCount;
    }


    private Int32 ReadPageLineCount() {
      Int32 PageLineCount = this.Reader.ReadInt32();

      if(PageLineCount < 0 || PageLineCount > 1000)
        throw new Exception("PageLineCount", new Exception(PageLineCount.ToString()));

      this.AddItem(4, "INT32", "PageLineCount", PageLineCount.ToString());
      return PageLineCount;
    }

    private Int32 ReadPageLineHeight() {
      Int32 PageLineHeight = this.Reader.ReadInt32();

      if(PageLineHeight < 1)
        throw new Exception("PageLineHeight", new Exception(PageLineHeight.ToString()));

      this.AddItem(4, "INT32", "PageLineHeight", PageLineHeight.ToString());
      return PageLineHeight;
    }

    private Int32 ReadPageFieldCount() {
      Int32 PageFieldCount = this.Reader.ReadInt32();

      if(PageFieldCount < 0)
        throw new Exception("PageFieldCount", new Exception(PageFieldCount.ToString()));

      this.AddItem(4, "INT32", "PageFieldCount", PageFieldCount.ToString());
      return PageFieldCount;
    }


    private Byte ReadPageFieldType() {
      Byte PageFieldType = this.Reader.ReadByte();

      //Empty = 0, Text = 1, Image = 2, Player = 3

      if(PageFieldType > 3)
        throw new Exception("PageFieldType", new Exception(PageFieldType.ToString()));

      this.AddItem(1, "BYTE", "PageFieldType", PageFieldType.ToString());
      return PageFieldType;
    }

    private Int32 ReadPageFieldWidth(String Name) {
      Int32 PageFieldWidth = this.Reader.ReadInt32();

      if(PageFieldWidth < 1 || PageFieldWidth > 800000)
        throw new Exception(Name, new Exception(PageFieldWidth.ToString()));

      this.AddItem(4, "INT32", Name, PageFieldWidth.ToString());
      return PageFieldWidth;
    }

    private Int32 ReadPageFieldHeight(String Name) {
      Int32 PageFieldHeight = this.Reader.ReadInt32();

      if(PageFieldHeight < 1 || PageFieldHeight > 800000)
        throw new Exception(Name, new Exception(PageFieldHeight.ToString()));

      this.AddItem(4, "INT32", Name, PageFieldHeight.ToString());
      return PageFieldHeight;
    }


    private UInt16 ReadFieldTextFlags() {
      UInt16 FieldTextFlags = this.Reader.ReadUInt16();

      //Alignment_Left = 0x0001, Alignment_Right = 0x0002, Alignment_Center = 0x0004, Alignment_Justified = 0x0008
      //UseLanguage = 0x0010, UseSizeable = 0x0020, UseSeparator = 0x0040

      if(FieldTextFlags > 0x007F)
        throw new Exception("FieldTextFlags", new Exception("0x" + FieldTextFlags.ToString("X4")));

      this.AddItem(2, "UINT16", "FieldTextFlags", "0x" + FieldTextFlags.ToString("X4"));
      return FieldTextFlags;
    }

    private Int32 ReadFieldTextWordMax() {
      Int32 FieldTextWordMax = this.Reader.ReadInt32();

      if(FieldTextWordMax < 1)
        throw new Exception("FieldTextWordMax", new Exception(FieldTextWordMax.ToString()));

      this.AddItem(4, "INT32", "FieldTextWordMax", FieldTextWordMax.ToString());
      return FieldTextWordMax;
    }

    private Int32 ReadFieldTextColorCount() {
      Int32 FieldTextColorCount = this.Reader.ReadInt32();

      if(FieldTextColorCount < 1 || FieldTextColorCount > 256)
        throw new Exception("FieldTextColorCount", new Exception(FieldTextColorCount.ToString()));

      this.AddItem(4, "INT32", "FieldTextColorCount", FieldTextColorCount.ToString());
      return FieldTextColorCount;
    }

    private void ReadFieldTextColorMemory(Int32 FieldTextColorCount) {
      UInt32[] FieldTextColorArray = new UInt32[FieldTextColorCount];

      for(Int32 i = 0; i < FieldTextColorCount; i++)
        FieldTextColorArray[i] = this.Reader.ReadUInt32();

      this.AddItem(FieldTextColorCount * 4, "UINT32[ ]", "FieldTextColorMemory", "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType.FieldTextColor, FieldTextColorArray)));
    }

    private Int32 ReadFieldTextFontCount() {
      Int32 FieldTextFontCount = this.Reader.ReadInt32();

      if(FieldTextFontCount < 1 || FieldTextFontCount > 256)
        throw new Exception("FieldTextFontCount", new Exception(FieldTextFontCount.ToString()));

      this.AddItem(4, "INT32", "FieldTextFontCount", FieldTextFontCount.ToString());
      return FieldTextFontCount;
    }

    private void ReadFieldTextFontMemory(Int32 FieldTextFontCount) {
      Int32[] FieldTextFontArray = new Int32[FieldTextFontCount];

      for(Int32 i = 0; i < FieldTextFontCount; i++)
        FieldTextFontArray[i] = this.Reader.ReadInt32();

      this.AddItem(FieldTextFontCount * 4, "INT32[ ]", "FieldTextFontMemory", "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType.FieldTextFont, FieldTextFontArray)));
    }

    private Int32 ReadFieldTextLetterCount() {
      Int32 FieldTextLetterCount = this.Reader.ReadInt32();

      if(FieldTextLetterCount < 1 || FieldTextLetterCount > 65535)
        throw new Exception("FieldTextLetterCount", new Exception(FieldTextLetterCount.ToString()));

      this.AddItem(4, "INT32", "FieldTextLetterCount", FieldTextLetterCount.ToString());
      return FieldTextLetterCount;
    }

    private void ReadFieldTextLetterMemory(Int32 FieldTextLetterCount) {
      Int32[] FieldTextLetterArray = new Int32[FieldTextLetterCount];

      for(Int32 i = 0; i < FieldTextLetterCount; i++)
        FieldTextLetterArray[i] = this.Reader.ReadInt32();

      this.AddItem(FieldTextLetterCount * 4, "INT32[ ]", "FieldTextLetterMemory", "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType.FieldTextLetter, FieldTextLetterArray)));
    }

    private Int32 ReadFieldTextLength() {
      Int32 FieldTextLength = this.Reader.ReadInt32();

      if(FieldTextLength < 1)
        throw new Exception("FieldTextLength", new Exception(FieldTextLength.ToString()));

      this.AddItem(4, "INT32", "FieldTextLength", FieldTextLength.ToString());
      return FieldTextLength;
    }

    private Int32 ReadFieldTextInfoSize(String Name) {
      Int32 FieldTextInfoSize = this.Reader.ReadInt32();

      if(FieldTextInfoSize < 1 || FieldTextInfoSize > this.Size - this.Position - 4)
        throw new Exception(Name, new Exception(FieldTextInfoSize.ToString()));

      this.AddItem(4, "INT32", Name, FieldTextInfoSize.ToString("#,#") + " bytes");
      return FieldTextInfoSize;
    }

    private void ReadFieldTextInfoMemoryV1(String Name, Int32 FieldTextInfoSize, Int32 FieldTextLength, Int32 FieldTextInfoCount, FileViewerTDHelpWriterType Type) {
      Byte[] FieldTextInfoMemory = this.Reader.ReadBytes(FieldTextInfoSize);

      //Info Type: 0x00 Letter, 0x01 Seperator, 0x02 Space, 0x03 Control, 0x04 NewLine (Flag)
      //Info Font: Font index
      //Info Color: Color index

      this.AddItem(FieldTextInfoSize, "BYTE[ ]", Name, "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(Type, new FileViewerTDCompressTextObject(null, FileViewerTDCompressTextType.Index, FieldTextInfoMemory, FieldTextInfoCount, FieldTextLength))));
    }

    private void ReadFieldTextInfoMemoryV2(String Name, Int32 FieldTextInfoSize, Int32 FieldTextLength, Int32 FieldTextInfoCount, FileViewerTDHelpWriterType Type) {
      Byte[] FieldTextInfoMemory = this.Reader.ReadBytes(FieldTextInfoSize);

      //Info Type: 0x00 Letter, 0x01 Seperator, 0x02 Space, 0x03 Control, 0x04 NewLine (Flag)
      //Info Font: Font index
      //Info Color: Color index

      this.AddItem(FieldTextInfoSize, "BYTE[ ]", Name, "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(Type, new FileViewerTDCompressTextObject(null, FileViewerTDCompressTextType.Index, FieldTextInfoMemory, FieldTextInfoCount, FieldTextLength))));
    }

    private Int32 ReadFieldTextIndexCount() {
      Int32 FieldTextIndexCount = this.Reader.ReadInt32();

      if(FieldTextIndexCount < 0)
        throw new Exception("FieldTextIndexCount", new Exception(FieldTextIndexCount.ToString()));

      this.AddItem(4, "INT32", "FieldTextIndexCount", FieldTextIndexCount.ToString());
      return FieldTextIndexCount;
    }

    private void ReadFieldTextIndexMemory(Int32 FieldTextIndexSize, Int32 FieldTextLength, Int32 FieldTextIndexCount) {
      Byte[] FieldTextIndexMemory = this.Reader.ReadBytes(FieldTextIndexSize);

      this.AddItem(FieldTextIndexSize, "BYTE[ ]", "FieldTextIndexMemory", "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType.FieldTextLetterIndex, new FileViewerTDCompressTextObject(null, FileViewerTDCompressTextType.Index, FieldTextIndexMemory, FieldTextIndexCount, FieldTextLength))));
    }

    private Int32 ReadFieldTextSeparatorIndex() {
      Int32 FieldTextSeparatorIndex = this.Reader.ReadInt32();

      if(FieldTextSeparatorIndex < 0)
        throw new Exception("FieldTextSeparatorIndex", new Exception(FieldTextSeparatorIndex.ToString()));

      this.AddItem(4, "INT32", "FieldTextSeparatorIndex", FieldTextSeparatorIndex.ToString());
      return FieldTextSeparatorIndex;
    }


    private UInt16 ReadFieldImageFlags() {
      UInt16 FieldImageFlags = this.Reader.ReadUInt16();

      //UseLanguage = 0x0001, UseFullscreen = 0x0002, UseSizeable = 0x0004, UseExpanded = 0x0008
      //UseEdit = 0x0010, UseDownload = 0x0020, UseJpeg = 0x0040, UsePng = 0x0080

      if(FieldImageFlags > 0x00FF)
        throw new Exception("FieldImageFlags", new Exception("0x" + FieldImageFlags.ToString("X4")));

      this.AddItem(2, "UINT16", "FieldImageFlags", "0x" + FieldImageFlags.ToString("X4"));
      return FieldImageFlags;
    }

    private Int32 ReadFieldImageUrlLength() {
      Int32 FieldImageUrlLength = this.Reader.ReadInt32();

      if(FieldImageUrlLength < -1 || FieldImageUrlLength > 16000)
        throw new Exception("FieldImageUrlLength", new Exception(FieldImageUrlLength.ToString()));

      this.AddItem(4, "INT32", "FieldImageUrlLength", FieldImageUrlLength.ToString());
      return FieldImageUrlLength;
    }

    private String ReadFieldImageUrl(Int32 FieldImageUrlLength) {
      Char[] LetterArray = new Char[FieldImageUrlLength];

      for(Int32 i = 0; i < FieldImageUrlLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String FieldImageUrl = new String(LetterArray);

      if(FieldImageUrl == null || FieldImageUrl == String.Empty)
        throw new Exception("FieldImageUrl", new Exception("Empty"));

      this.AddItem(FieldImageUrlLength * 2, "WCHAR[ ]", "FieldImageUrl", FieldImageUrl);
      return FieldImageUrl;
    }

    private Int32 ReadFieldImageDownloadIndex() {
      Int32 FieldImageDownloadIndex = this.Reader.ReadInt32();

      if(FieldImageDownloadIndex < 0)
        throw new Exception("FieldImageDownloadIndex", new Exception(FieldImageDownloadIndex.ToString()));

      this.AddItem(4, "INT32", "FieldImageDownloadIndex", FieldImageDownloadIndex.ToString());
      return FieldImageDownloadIndex;
    }

    private Int32 ReadFieldImageDisplayIndexV1() {
      Int32 FieldImageDisplayIndex = this.Reader.ReadInt32();

      if(FieldImageDisplayIndex < -1)
        throw new Exception("FieldImageDisplayIndex", new Exception(FieldImageDisplayIndex.ToString()));

      this.AddItem(4, "INT32", "FieldImageDisplayIndex", FieldImageDisplayIndex.ToString());
      return FieldImageDisplayIndex;
    }

    private Int32 ReadFieldImageDisplayIndexV2() {
      Int32 FieldImageDisplayIndex = this.Reader.ReadInt32();

      if(FieldImageDisplayIndex < 0)
        throw new Exception("FieldImageDisplayIndex", new Exception(FieldImageDisplayIndex.ToString()));

      this.AddItem(4, "INT32", "FieldImageDisplayIndex", FieldImageDisplayIndex.ToString());
      return FieldImageDisplayIndex;
    }

    private Int32 ReadFieldImageOriginalIndexV1() {
      Int32 FieldImageOriginalIndex = this.Reader.ReadInt32();

      if(FieldImageOriginalIndex < 0)
        throw new Exception("FieldImageOriginalIndex", new Exception(FieldImageOriginalIndex.ToString()));

      this.AddItem(4, "INT32", "FieldImageOriginalIndex", FieldImageOriginalIndex.ToString());
      return FieldImageOriginalIndex;
    }

    private Int32 ReadFieldImageOriginalIndexV2() {
      Int32 FieldImageOriginalIndex = this.Reader.ReadInt32();

      if(FieldImageOriginalIndex < -1)
        throw new Exception("FieldImageOriginalIndex", new Exception(FieldImageOriginalIndex.ToString()));

      this.AddItem(4, "INT32", "FieldImageOriginalIndex", FieldImageOriginalIndex.ToString());
      return FieldImageOriginalIndex;
    }

    private Byte ReadFieldImageQuality() {
      Byte FieldImageQuality = this.Reader.ReadByte();

      if(FieldImageQuality < 20 || FieldImageQuality > 100)
        throw new Exception("FieldImageQuality", new Exception(FieldImageQuality.ToString()));

      this.AddItem(1, "BYTE", "FieldImageQuality", FieldImageQuality.ToString());
      return FieldImageQuality;
    }


    private Boolean ReadFieldPlayerUseImage() {
      Byte FieldPlayerUseImage = this.Reader.ReadByte();

      if(FieldPlayerUseImage > 1)
        throw new Exception("FieldPlayerUseImage", new Exception(FieldPlayerUseImage.ToString()));

      this.AddItem(1, "BYTE", "FieldPlayerUseImage", FieldPlayerUseImage.ToString());
      return FieldPlayerUseImage == 0 ? false : true;
    }

    private UInt16 ReadFieldPlayerFlagsV1() {
      UInt16 FieldPlayerFlags = this.Reader.ReadUInt16();

      //UseLanguage = 0x0001, UseFullscreen = 0x0002, UseSizeable = 0x0004, UseExpanded = 0x0008
      //UseDownload = 0x0010, UseRepeat = 0x0020, UsePosition = 0x0040, UseVolume = 0x0080

      if(FieldPlayerFlags > 0x00FF)
        throw new Exception("FieldPlayerFlags", new Exception("0x" + FieldPlayerFlags.ToString("X4")));

      this.AddItem(2, "UINT16", "FieldPlayerFlags", "0x" + FieldPlayerFlags.ToString("X4"));
      return FieldPlayerFlags;
    }

    private UInt16 ReadFieldPlayerFlagsV2() {
      UInt16 FieldPlayerFlags = this.Reader.ReadUInt16();

      //UseLanguage = 0x0001, UseFullscreen = 0x0002, UseSizeable = 0x0004, UseExpanded = 0x0008
      //UseDownload = 0x0010, UseRepeat = 0x0020, UsePosition = 0x0040, UseVolume = 0x0080
      //UseStreaming = 0x0100

      if(FieldPlayerFlags > 0x01FF)
        throw new Exception("FieldPlayerFlags", new Exception("0x" + FieldPlayerFlags.ToString("X4")));

      this.AddItem(2, "UINT16", "FieldPlayerFlags", "0x" + FieldPlayerFlags.ToString("X4"));
      return FieldPlayerFlags;
    }

    private Int32 ReadFieldPlayerUrlLength() {
      Int32 FieldPlayerUrlLength = this.Reader.ReadInt32();

      if(FieldPlayerUrlLength < -1 || FieldPlayerUrlLength > 16000)
        throw new Exception("FieldPlayerUrlLength", new Exception(FieldPlayerUrlLength.ToString()));

      this.AddItem(4, "INT32", "FieldPlayerUrlLength", FieldPlayerUrlLength.ToString());
      return FieldPlayerUrlLength;
    }

    private String ReadFieldPlayerUrl(Int32 FieldPlayerUrlLength) {
      Char[] LetterArray = new Char[FieldPlayerUrlLength];

      for(Int32 i = 0; i < FieldPlayerUrlLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String FieldPlayerUrl = new String(LetterArray);

      if(FieldPlayerUrl == null || FieldPlayerUrl == String.Empty)
        throw new Exception("FieldImageUrl", new Exception("Empty"));

      this.AddItem(FieldPlayerUrlLength * 2, "WCHAR[ ]", "FieldPlayerUrl", FieldPlayerUrl);
      return FieldPlayerUrl;
    }

    private Byte ReadFieldPlayerType() {
      Byte FieldPlayerType = this.Reader.ReadByte();

      //Sound = 0, SoundImage = 1, Video = 2, VideoAudio = 3, Gif = 4, TDAnimation = 5,

      if(FieldPlayerType > 5)
        throw new Exception("FieldPlayerType", new Exception(FieldPlayerType.ToString()));

      this.AddItem(1, "BYTE", "FieldPlayerType", FieldPlayerType.ToString());
      return FieldPlayerType;
    }

    private Int64 ReadFieldPlayerDuration() {
      Int64 FieldPlayerDuration = this.Reader.ReadInt64();

      if(FieldPlayerDuration < 0)
        throw new Exception("FieldPlayerDuration", new Exception(FieldPlayerDuration.ToString()));

      this.AddItem(8, "INT64", "FieldPlayerDuration", FieldPlayerDuration.ToString());
      return FieldPlayerDuration;
    }

    private Int32 ReadFieldPlayerDisplayWidth() {
      Int32 FieldPlayerDisplayWidth = this.Reader.ReadInt32();

      if(FieldPlayerDisplayWidth < 0 || FieldPlayerDisplayWidth > 16000)
        throw new Exception("FieldPlayerDisplayWidth", new Exception(FieldPlayerDisplayWidth.ToString()));

      this.AddItem(4, "INT32", "FieldPlayerDisplayWidth", FieldPlayerDisplayWidth.ToString());
      return FieldPlayerDisplayWidth;
    }

    private Int32 ReadFieldPlayerDisplayHeight() {
      Int32 FieldPlayerDisplayHeight = this.Reader.ReadInt32();

      if(FieldPlayerDisplayHeight < 0 || FieldPlayerDisplayHeight > 16000)
        throw new Exception("FieldPlayerDisplayHeight", new Exception(FieldPlayerDisplayHeight.ToString()));

      this.AddItem(4, "INT32", "FieldPlayerDisplayHeight", FieldPlayerDisplayHeight.ToString());
      return FieldPlayerDisplayHeight;
    }

    private Int32 ReadFieldPlayerIndex() {
      Int32 FieldPlayerIndex = this.Reader.ReadInt32();

      if(FieldPlayerIndex < 0)
        throw new Exception("FieldPlayerIndex", new Exception(FieldPlayerIndex.ToString()));

      this.AddItem(4, "INT32", "FieldPlayerIndex", FieldPlayerIndex.ToString());
      return FieldPlayerIndex;
    }


    private UInt16 ReadFieldVariousFlags() {
      UInt16 FieldVariousFlags = this.Reader.ReadUInt16();

      //UseTipText = 0x0001, UseTipTextLanguage = 0x0002, UseLinkPath = 0x0004, UseLinkPathLanguage = 0x0008
      //UseFieldID = 0x0010, UseFieldIDHorizontal = 0x0020, UseFieldIDVertical = 0x0040, UseHover = 0x0080

      if(FieldVariousFlags > 0x00FF)
        throw new Exception("FieldVariousFlags", new Exception("0x" + FieldVariousFlags.ToString("X4")));

      this.AddItem(2, "UINT16", "FieldVariousFlags", "0x" + FieldVariousFlags.ToString("X4"));
      return FieldVariousFlags;
    }

    private SByte ReadFieldVariousHover(String Name) {
      SByte FieldVariousHover = this.Reader.ReadSByte();

      if(FieldVariousHover < 0)
        throw new Exception(Name, new Exception(FieldVariousHover.ToString()));

      this.AddItem(1, "INT8", Name, FieldVariousHover.ToString());
      return FieldVariousHover;
    }

    private Int32 ReadFieldVariousLength(String Name) {
      Int32 FieldVariousLength = this.Reader.ReadInt32();

      if(FieldVariousLength < 1 || FieldVariousLength > 16000)
        throw new Exception(Name, new Exception(FieldVariousLength.ToString()));

      this.AddItem(4, "INT32", Name, FieldVariousLength.ToString());
      return FieldVariousLength;
    }

    private String ReadFieldVariousText(String Name, Int32 FieldVariousLength) {
      Char[] LetterArray = new Char[FieldVariousLength];

      for(Int32 i = 0; i < FieldVariousLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String FieldVariousText = new String(LetterArray);

      if(FieldVariousText == null || FieldVariousText == String.Empty)
        throw new Exception(Name, new Exception("Empty"));

      this.AddItem(FieldVariousLength * 2, "WCHAR[ ]", Name, FieldVariousText);
      return FieldVariousText;
    }


    private UInt16 ReadFieldFlags() {
      UInt16 FieldFlags = this.Reader.ReadUInt16();

      //AlignmentHorizontal_Left = 0x0000, AlignmentHorizontal_Center = 00001, AlignmentHorizontal_Right = 0x0002
      //AlignmentVertical_Top = 0x0000, AlignmentVertical_Center = 0x0010, AlignmentVertical_Bottom = 0x0020
      //UseCurve = 0x0004, UseShadow = 0x0008, UseBackground = 0x0040, UseBorder = 0x0080
      //UseFrame = 0x0100, UseID = 0x0200, UseVarious = 0x0400, UseImage = 0x0800
      //UseImageAnimate = 0x1000, UseImageForeground = 0x2000, UseImageShadow = 0x4000

      if(FieldFlags > 0x7FFF)
        throw new Exception("FieldFlags", new Exception("0x" + FieldFlags.ToString("X4")));

      this.AddItem(2, "UINT16", "FieldFlags", "0x" + FieldFlags.ToString("X4"));
      return FieldFlags;
    }

    private Int32 ReadFieldDistance(String Name) {
      Int32 FieldDistance = this.Reader.ReadInt32();

      if(FieldDistance < 0 || FieldDistance > 10000)
        throw new Exception(Name, new Exception(FieldDistance.ToString()));

      this.AddItem(4, "INT32", Name, FieldDistance.ToString());
      return FieldDistance;
    }

    private Int32 ReadFieldWidth() {
      Int32 FieldWidth = this.Reader.ReadInt32();

      if(FieldWidth < 3 || FieldWidth > 1000000)
        throw new Exception("FieldWidth", new Exception(FieldWidth.ToString()));

      this.AddItem(4, "INT32", "FieldWidth", FieldWidth.ToString());
      return FieldWidth;
    }

    private Int32 ReadFieldCurve(String Name) {
      Int32 FieldFrame = this.Reader.ReadInt32();

      if(FieldFrame < 0 || FieldFrame > 1000)
        throw new Exception(Name, new Exception(FieldFrame.ToString()));

      this.AddItem(4, "INT32", Name, FieldFrame.ToString());
      return FieldFrame;
    }

    private Int32 ReadFieldFrame(String Name) {
      Int32 FieldFrame = this.Reader.ReadInt32();

      if(FieldFrame < 0 || FieldFrame > 10000)
        throw new Exception(Name, new Exception(FieldFrame.ToString()));

      this.AddItem(4, "INT32", Name, FieldFrame.ToString());
      return FieldFrame;
    }

    private Int32 ReadFieldShadow(String Name) {
      Int32 FieldShadow = this.Reader.ReadInt32();

      if(FieldShadow < 0 || FieldShadow > 1000)
        throw new Exception(Name, new Exception(FieldShadow.ToString()));

      this.AddItem(4, "INT32", Name, FieldShadow.ToString());
      return FieldShadow;
    }

    private UInt32 ReadFieldColor(String Name) {
      UInt32 FieldColor = this.Reader.ReadUInt32();

      if(FieldColor < 0xFF000000)
        throw new Exception(Name, new Exception("0x" + FieldColor.ToString("X8")));

      this.AddItem(4, "UINT32", Name, "0x" + FieldColor.ToString("X8"));
      return FieldColor;
    }

    private Int32 ReadFieldImage(String Name) {
      Int32 FieldImage = this.Reader.ReadInt32();

      if(FieldImage < 0 || FieldImage > 100000)
        throw new Exception(Name, new Exception(FieldImage.ToString()));

      this.AddItem(4, "INT32", Name, FieldImage.ToString());
      return FieldImage;
    }

    private Int32 ReadFieldImageCount() {
      Int32 FieldImageCount = this.Reader.ReadInt32();

      if(FieldImageCount < 1 || FieldImageCount > 999)
        throw new Exception("FieldImageCount", new Exception(FieldImageCount.ToString()));

      this.AddItem(4, "INT32", "FieldImageCount", FieldImageCount.ToString());
      return FieldImageCount;
    }

    private Int32 ReadFieldImageIndex() {
      Int32 FieldImageIndex = this.Reader.ReadInt32();

      if(FieldImageIndex < 0)
        throw new Exception("FieldImageIndex", new Exception(FieldImageIndex.ToString()));

      this.AddItem(4, "INT32", "FieldImageIndex", FieldImageIndex.ToString());
      return FieldImageIndex;
    }

    private Int32 ReadFieldImageTime(String Name) {
      Int32 FieldImageTime = this.Reader.ReadInt32();

      if(FieldImageTime < 0 || FieldImageTime > 3600000)
        throw new Exception(Name, new Exception(FieldImageTime.ToString()));

      this.AddItem(4, "INT32", Name, FieldImageTime.ToString());
      return FieldImageTime;
    }

    private UInt32 ReadFieldImageShadowColor() {
      UInt32 FieldImageShadowColor = this.Reader.ReadUInt32();

      this.AddItem(4, "UINT32", "FieldImageShadowColor", "0x" + FieldImageShadowColor.ToString("X8"));
      return FieldImageShadowColor;
    }

    private Int32 ReadFieldIDLength() {
      Int32 FieldIDLength = this.Reader.ReadInt32();

      if(FieldIDLength < 1 || FieldIDLength > 100)
        throw new Exception("FieldIDLength", new Exception(FieldIDLength.ToString()));

      this.AddItem(4, "INT32", "FieldIDLength", FieldIDLength.ToString());
      return FieldIDLength;
    }

    private String ReadFieldID(Int32 FieldIDLength) {
      Char[] LetterArray = new Char[FieldIDLength];

      for(Int32 i = 0; i < FieldIDLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String FieldID = new String(LetterArray);

      if(FieldID == null || FieldID == String.Empty)
        throw new Exception("FieldID", new Exception("Empty"));

      this.AddItem(FieldIDLength * 2, "WCHAR[ ]", "FieldID", FieldID);
      return FieldID;
    }


    private Boolean ReadPageFieldEmptyV1() {
      Int32 FieldEmptyWidth = this.ReadPageFieldWidth("FieldEmptyWidth");
      Int32 FieldEmptyHeight = this.ReadPageFieldHeight("FieldEmptyHeight");
      return false;
    }

    private Boolean ReadPageFieldTextV1(Int32 LanguageCount) {
      Int32 FieldTextCount = 1;

      for(Int32 i = 0; i < FieldTextCount; i++) {
        UInt16 FieldTextFlags = this.ReadFieldTextFlags();

        //UseLanguage
        if(i == 0 && (FieldTextFlags & 0x0010) == 0x0010 && LanguageCount > 1)
          FieldTextCount = LanguageCount;

        Int32 FieldTextWidth = this.ReadPageFieldWidth("FieldTextWidth");
        Int32 FieldTextHeight = this.ReadPageFieldHeight("FieldTextHeight");

        Int32 FieldTextWordMax = this.ReadFieldTextWordMax();

        Int32 FieldTextColorCount = this.ReadFieldTextColorCount();
        this.ReadFieldTextColorMemory(FieldTextColorCount);

        Int32 FieldTextFontCount = this.ReadFieldTextFontCount();
        this.ReadFieldTextFontMemory(FieldTextFontCount);

        for(Int32 j = 0; j < FieldTextFontCount; j++) {
          Int32 FieldTextLetterCount = this.ReadFieldTextLetterCount();
          this.ReadFieldTextLetterMemory(FieldTextLetterCount);
        }

        Int32 FieldTextLength = this.ReadFieldTextLength();

        Int32 FieldTextInfoTypeSize = this.ReadFieldTextInfoSize("FieldTextInfoTypeSize");
        this.ReadFieldTextInfoMemoryV1("FieldTextInfoTypeMemory", FieldTextInfoTypeSize, FieldTextLength, 8, FileViewerTDHelpWriterType.FieldTextInfoType);

        if(FieldTextFontCount > 1) {
          Int32 FieldTextInfoFontSize = this.ReadFieldTextInfoSize("FieldTextInfoFontSize");
          this.ReadFieldTextInfoMemoryV1("FieldTextInfoFontMemory", FieldTextInfoFontSize, FieldTextLength, FieldTextFontCount, FileViewerTDHelpWriterType.FieldTextInfoFont);
        }

        if(FieldTextColorCount > 1) {
          Int32 FieldTextInfoColorSize = this.ReadFieldTextInfoSize("FieldTextInfoColorSize");
          this.ReadFieldTextInfoMemoryV1("FieldTextInfoColorMemory", FieldTextInfoColorSize, FieldTextLength, FieldTextColorCount, FileViewerTDHelpWriterType.FieldTextInfoColor);
        }

        Int32 FieldTextIndexCount = this.ReadFieldTextIndexCount();

        if(FieldTextIndexCount > 1) {
          Int32 FieldTextIndexSize = this.ReadFieldTextInfoSize("FieldTextIndexSize");
          this.ReadFieldTextIndexMemory(FieldTextIndexSize, FieldTextLength, FieldTextIndexCount);
        }

        //UseSeparator
        if((FieldTextFlags & 0x0040) == 0x0040) {
          for(Int32 j = 0; j < FieldTextFontCount; j++) {
            Int32 FieldTextSeparatorIndex = this.ReadFieldTextSeparatorIndex();
          }
        }
      }
      return FieldTextCount == 1 ? false : true;
    }

    private Boolean ReadPageFieldImageV1(Int32 LanguageCount) {
      Int32 FieldImageCount = 1;

      for(Int32 i = 0; i < FieldImageCount; i++) {
        UInt16 FieldImageFlags = this.ReadFieldImageFlags();

        //UseLanguage
        if(i == 0 && (FieldImageFlags & 0x0001) == 0x0001 && LanguageCount > 1)
          FieldImageCount = LanguageCount;

        Int32 FieldImageWidth = this.ReadPageFieldWidth("FieldImageWidth");
        Int32 FieldImageHeight = this.ReadPageFieldHeight("FieldImageHeight");

        Int32 FieldImageUrlLength = this.ReadFieldImageUrlLength();
        if(FieldImageUrlLength > 0) this.ReadFieldImageUrl(FieldImageUrlLength);

        Int32 FieldImageDisplayIndex = this.ReadFieldImageDisplayIndexV1();
        Int32 FieldImageOriginalIndex = this.ReadFieldImageOriginalIndexV1();

        Byte FieldImageQuality = this.ReadFieldImageQuality();
      }
      return FieldImageCount == 1 ? false : true;
    }

    private Boolean ReadPageFieldPlayerV1(Int32 LanguageCount) {
      Boolean UseItemLanguage = false;
      Boolean FieldPlayerUseImage = this.ReadFieldPlayerUseImage();

      if(FieldPlayerUseImage)
        UseItemLanguage = this.ReadPageFieldImageV1(LanguageCount);

      Int32 FieldPlayerCount = 1;

      for(Int32 i = 0; i < FieldPlayerCount; i++) {
        UInt16 FieldPlayerFlags = this.ReadFieldPlayerFlagsV1();

        //UseLanguage
        if(i == 0 && (FieldPlayerFlags & 0x0001) == 0x0001 && LanguageCount > 1)
          FieldPlayerCount = LanguageCount;

        Int32 FieldPlayerWidth = this.ReadPageFieldWidth("FieldPlayerWidth");
        Int32 FieldPlayerHeight = this.ReadPageFieldHeight("FieldPlayerHeight");

        UInt32 FieldPlayerColor = this.ReadFieldColor("FieldPlayerColor");

        Int32 FieldPlayerUrlLength = this.ReadFieldPlayerUrlLength();
        if(FieldPlayerUrlLength > 0) this.ReadFieldPlayerUrl(FieldPlayerUrlLength);

        //UseDownload
        if((FieldPlayerFlags & 0x0010) == 0x0010) {
          Byte FieldPlayerType = this.ReadFieldPlayerType();

          Int32 FieldPlayerDisplayWidth = this.ReadFieldPlayerDisplayWidth();
          Int32 FieldPlayerDisplayHeight = this.ReadFieldPlayerDisplayHeight();

          Int64 FieldPlayerDuration = this.ReadFieldPlayerDuration();
        } else {
          Int32 FieldPlayerIndex = this.ReadFieldPlayerIndex();
        }
      }
      return FieldPlayerCount == 1 ? UseItemLanguage : true;
    }


    private void ReadPageFieldSettingV1(Boolean UseItemLanguage, Int32 LanguageCount) {
      Int32 FieldSettingCount = 1;

      if(UseItemLanguage && LanguageCount > 1) FieldSettingCount = LanguageCount;

      UInt16 FieldFlags = 0;

      for(Int32 i = 0; i < FieldSettingCount; i++) {
        FieldFlags = this.ReadFieldFlags();

        Int32 FieldDistanceLeft = this.ReadFieldDistance("FieldDistanceLeft");
        Int32 FieldDistanceTop = this.ReadFieldDistance("FieldDistanceTop");
        Int32 FieldDistanceRight = this.ReadFieldDistance("FieldDistanceRight");
        Int32 FieldDistanceBottom = this.ReadFieldDistance("FieldDistanceBottom");

        Int32 FieldWidth = this.ReadFieldWidth();

        Int32 FieldCurveX = this.ReadFieldCurve("FieldCurveX");
        Int32 FieldCurveY = this.ReadFieldCurve("FieldCurveY");

        Int32 FieldFrameLeft = this.ReadFieldFrame("FieldFrameLeft");
        Int32 FieldFrameTop = this.ReadFieldFrame("FieldFrameTop");
        Int32 FieldFrameRight = this.ReadFieldFrame("FieldFrameRight");
        Int32 FieldFrameBottom = this.ReadFieldFrame("FieldFrameBottom");

        Int32 FieldShadowX = this.ReadFieldShadow("FieldShadowX");
        Int32 FieldShadowY = this.ReadFieldShadow("FieldShadowY");
        UInt32 FieldShadowColor = this.ReadFieldColor("FieldShadowColor");

        UInt32 FieldBorderColor = this.ReadFieldColor("FieldBorderColor");
        UInt32 FieldBackColor = this.ReadFieldColor("FieldBackColor");

        //UseImage
        if((FieldFlags & 0x0800) == 0x0800) {
          Int32 FieldImageX = this.ReadFieldImage("FieldImageX");
          Int32 FieldImageY = this.ReadFieldImage("FieldImageY");

          Int32 FieldImageCount = this.ReadFieldImageCount();
          Int32 FieldImageIndex = this.ReadFieldImageIndex();

          //UseImageAnimate
          if((FieldFlags & 0x1000) == 0x1000) {
            Int32 FieldImageStartTime = this.ReadFieldImageTime("FieldImageStartTime");
            Int32 FieldImageDisplayTime = this.ReadFieldImageTime("FieldImageDisplayTime");
          }
        }

        Int32 FieldImageShadowX = this.ReadFieldShadow("FieldImageShadowX");
        Int32 FieldImageShadowY = this.ReadFieldShadow("FieldImageShadowY");
        UInt32 FieldImageShadowColor = this.ReadFieldImageShadowColor();
      }

      //UseID
      if((FieldFlags & 0x0200) == 0x0200) {
        Int32 FieldIDLength = this.ReadFieldIDLength();

        if(FieldIDLength > 0) {
          String FieldID = this.ReadFieldID(FieldIDLength);
        }
      }

      //UseVarious
      if((FieldFlags & 0x0400) == 0x0400)
        this.ReadPageFieldVariousV1(LanguageCount);
    }

    private void ReadPageFieldVariousV1(Int32 LanguageCount) {
      UInt16 FieldVariousFlags = this.ReadFieldVariousFlags();

      //UseHover
      if((FieldVariousFlags & 0x0080) == 0x0080) {
        SByte FieldVariousHoverX = this.ReadFieldVariousHover("FieldVariousHoverX");
        SByte FieldVariousHoverY = this.ReadFieldVariousHover("FieldVariousHoverY");
      }

      //UseFieldID
      if((FieldVariousFlags & 0x0010) == 0x0010) {
        Int32 FieldVariousIDLength = this.ReadFieldVariousLength("FieldVariousIDLength");
        String FieldVariousID = this.ReadFieldVariousText("FieldVariousID", FieldVariousIDLength);
      }

      //UseTipText
      if((FieldVariousFlags & 0x0001) == 0x0001) {
        Int32 FieldVariousCount = 1;

        //UseTipTextLanguage
        if((FieldVariousFlags & 0x0002) == 0x0002 && LanguageCount > 1)
          FieldVariousCount = LanguageCount;

        for(Int32 i = 0; i < FieldVariousCount; i++) {
          Int32 FieldVariousTipLength = this.ReadFieldVariousLength("FieldVariousTipLength");
          String FieldVariousTip = this.ReadFieldVariousText("FieldVariousTip", FieldVariousTipLength);
        }
      }

      //UseLinkPath
      if((FieldVariousFlags & 0x0004) == 0x0004) {
        Int32 FieldVariousCount = 1;

        //UseLinkPathLanguage
        if((FieldVariousFlags & 0x0008) == 0x0008 && LanguageCount > 1)
          FieldVariousCount = LanguageCount;

        for(Int32 i = 0; i < FieldVariousCount; i++) {
          Int32 FieldVariousLinkLength = this.ReadFieldVariousLength("FieldVariousLinkLength");
          String FieldVariousLink = this.ReadFieldVariousText("FieldVariousLink", FieldVariousLinkLength);
        }
      }
    }


    private Boolean ReadPageFieldEmptyV2() {
      Int32 FieldEmptyWidth = this.ReadPageFieldWidth("FieldEmptyWidth");
      Int32 FieldEmptyHeight = this.ReadPageFieldHeight("FieldEmptyHeight");
      return false;
    }

    private Boolean ReadPageFieldTextV2(Int32 LanguageCount) {
      Int32 FieldTextCount = 1;

      for(Int32 i = 0; i < FieldTextCount; i++) {
        UInt16 FieldTextFlags = this.ReadFieldTextFlags();

        //UseLanguage
        if(i == 0 && (FieldTextFlags & 0x0010) == 0x0010 && LanguageCount > 1)
          FieldTextCount = LanguageCount;

        Int32 FieldTextWidth = this.ReadPageFieldWidth("FieldTextWidth");
        Int32 FieldTextHeight = this.ReadPageFieldHeight("FieldTextHeight");

        Int32 FieldTextColorCount = this.ReadFieldTextColorCount();
        this.ReadFieldTextColorMemory(FieldTextColorCount);

        Int32 FieldTextFontCount = this.ReadFieldTextFontCount();
        this.ReadFieldTextFontMemory(FieldTextFontCount);

        for(Int32 j = 0; j < FieldTextFontCount; j++) {
          Int32 FieldTextLetterCount = this.ReadFieldTextLetterCount();
          this.ReadFieldTextLetterMemory(FieldTextLetterCount);
        }

        Int32 FieldTextLength = this.ReadFieldTextLength();

        Int32 FieldTextInfoTypeSize = this.ReadFieldTextInfoSize("FieldTextInfoTypeSize");
        this.ReadFieldTextInfoMemoryV2("FieldTextInfoTypeMemory", FieldTextInfoTypeSize, FieldTextLength, 4, FileViewerTDHelpWriterType.FieldTextInfoType);

        if(FieldTextFontCount > 1) {
          Int32 FieldTextInfoFontSize = this.ReadFieldTextInfoSize("FieldTextInfoFontSize");
          this.ReadFieldTextInfoMemoryV2("FieldTextInfoFontMemory", FieldTextInfoFontSize, FieldTextLength, FieldTextFontCount, FileViewerTDHelpWriterType.FieldTextInfoFont);
        }

        if(FieldTextColorCount > 1) {
          Int32 FieldTextInfoColorSize = this.ReadFieldTextInfoSize("FieldTextInfoColorSize");
          this.ReadFieldTextInfoMemoryV2("FieldTextInfoColorMemory", FieldTextInfoColorSize, FieldTextLength, FieldTextColorCount, FileViewerTDHelpWriterType.FieldTextInfoColor);
        }

        Int32 FieldTextIndexCount = this.ReadFieldTextIndexCount();

        if(FieldTextIndexCount > 1) {
          Int32 FieldTextIndexSize = this.ReadFieldTextInfoSize("FieldTextIndexSize");
          this.ReadFieldTextIndexMemory(FieldTextIndexSize, FieldTextLength, FieldTextIndexCount);
        }

        //UseSeparator
        if((FieldTextFlags & 0x0040) == 0x0040) {
          for(Int32 j = 0; j < FieldTextFontCount; j++) {
            Int32 FieldTextSeparatorIndex = this.ReadFieldTextSeparatorIndex();
          }
        }
      }
      return FieldTextCount == 1 ? false : true;
    }

    private Boolean ReadPageFieldImageV2(Int32 LanguageCount) {
      Int32 FieldImageCount = 1;

      for(Int32 i = 0; i < FieldImageCount; i++) {
        UInt16 FieldImageFlags = this.ReadFieldImageFlags();

        //UseLanguage
        if(i == 0 && (FieldImageFlags & 0x0001) == 0x0001 && LanguageCount > 1)
          FieldImageCount = LanguageCount;

        Int32 FieldImageWidth = this.ReadPageFieldWidth("FieldImageWidth");
        Int32 FieldImageHeight = this.ReadPageFieldHeight("FieldImageHeight");

        //UseDownload
        if((FieldImageFlags & 0x0020) == 0x0020) {
          Int32 FieldImageDownloadIndex = this.ReadFieldImageDownloadIndex();
        } else {
          Int32 FieldImageDisplayIndex = this.ReadFieldImageDisplayIndexV2();
          Int32 FieldImageOriginalIndex = this.ReadFieldImageOriginalIndexV2();
        }

        Byte FieldImageQuality = this.ReadFieldImageQuality();
      }
      return FieldImageCount == 1 ? false : true;
    }

    private Boolean ReadPageFieldPlayerV2(Int32 LanguageCount) {
      Boolean UseItemLanguage = false;
      Boolean FieldPlayerUseImage = this.ReadFieldPlayerUseImage();

      if(FieldPlayerUseImage)
        UseItemLanguage = this.ReadPageFieldImageV2(LanguageCount);

      Int32 FieldPlayerCount = 1;

      for(Int32 i = 0; i < FieldPlayerCount; i++) {
        UInt16 FieldPlayerFlags = this.ReadFieldPlayerFlagsV2();

        //UseLanguage
        if(i == 0 && (FieldPlayerFlags & 0x0001) == 0x0001 && LanguageCount > 1)
          FieldPlayerCount = LanguageCount;

        Int32 FieldPlayerWidth = this.ReadPageFieldWidth("FieldPlayerWidth");
        Int32 FieldPlayerHeight = this.ReadPageFieldHeight("FieldPlayerHeight");

        UInt32 FieldPlayerColor = this.ReadFieldColor("FieldPlayerColor");

        Int32 FieldPlayerUrlLength = this.ReadFieldPlayerUrlLength();
        if(FieldPlayerUrlLength > 0) this.ReadFieldPlayerUrl(FieldPlayerUrlLength);

        //UseDownload == 0x0010, UseStreaming = 0x0100
        if((FieldPlayerFlags & 0x0010) == 0x0010 || (FieldPlayerFlags & 0x0100) == 0x0100) {
          Byte FieldPlayerType = this.ReadFieldPlayerType();

          Int32 FieldPlayerDisplayWidth = this.ReadFieldPlayerDisplayWidth();
          Int32 FieldPlayerDisplayHeight = this.ReadFieldPlayerDisplayHeight();

          Int64 FieldPlayerDuration = this.ReadFieldPlayerDuration();
        } else {
          Int32 FieldPlayerIndex = this.ReadFieldPlayerIndex();
        }
      }
      return FieldPlayerCount == 1 ? UseItemLanguage : true;
    }


    private void ReadPageFieldSettingV2(Boolean UseItemLanguage, Int32 LanguageCount) {
      Int32 FieldSettingCount = 1;

      if(UseItemLanguage && LanguageCount > 1) FieldSettingCount = LanguageCount;

      UInt16 FieldFlags = 0;

      for(Int32 i = 0; i < FieldSettingCount; i++) {
        FieldFlags = this.ReadFieldFlags();

        Int32 FieldDistanceLeft = this.ReadFieldDistance("FieldDistanceLeft");
        Int32 FieldDistanceTop = this.ReadFieldDistance("FieldDistanceTop");
        Int32 FieldDistanceRight = this.ReadFieldDistance("FieldDistanceRight");
        Int32 FieldDistanceBottom = this.ReadFieldDistance("FieldDistanceBottom");

        Int32 FieldWidth = this.ReadFieldWidth();

        Int32 FieldCurveX = this.ReadFieldCurve("FieldCurveX");
        Int32 FieldCurveY = this.ReadFieldCurve("FieldCurveY");

        Int32 FieldFrameLeft = this.ReadFieldFrame("FieldFrameLeft");
        Int32 FieldFrameTop = this.ReadFieldFrame("FieldFrameTop");
        Int32 FieldFrameRight = this.ReadFieldFrame("FieldFrameRight");
        Int32 FieldFrameBottom = this.ReadFieldFrame("FieldFrameBottom");

        Int32 FieldShadowX = this.ReadFieldShadow("FieldShadowX");
        Int32 FieldShadowY = this.ReadFieldShadow("FieldShadowY");
        UInt32 FieldShadowColor = this.ReadFieldColor("FieldShadowColor");

        UInt32 FieldBorderColor = this.ReadFieldColor("FieldBorderColor");
        UInt32 FieldBackColor = this.ReadFieldColor("FieldBackColor");

        //UseImage
        if((FieldFlags & 0x0800) == 0x0800) {
          Int32 FieldImageX = this.ReadFieldImage("FieldImageX");
          Int32 FieldImageY = this.ReadFieldImage("FieldImageY");
          Int32 FieldImageIndex = this.ReadFieldImageIndex();

          //UseImageAnimate
          if((FieldFlags & 0x1000) == 0x1000) {
            Int32 FieldImageStartTime = this.ReadFieldImageTime("FieldImageStartTime");
            Int32 FieldImageDisplayTime = this.ReadFieldImageTime("FieldImageDisplayTime");
          }
        }

        Int32 FieldImageShadowX = this.ReadFieldShadow("FieldImageShadowX");
        Int32 FieldImageShadowY = this.ReadFieldShadow("FieldImageShadowY");
        UInt32 FieldImageShadowColor = this.ReadFieldImageShadowColor();
      }

      //UseID
      if((FieldFlags & 0x0200) == 0x0200) {
        Int32 FieldIDLength = this.ReadFieldIDLength();

        if(FieldIDLength > 0) {
          String FieldID = this.ReadFieldID(FieldIDLength);
        }
      }

      //UseVarious
      if((FieldFlags & 0x0400) == 0x0400)
        this.ReadPageFieldVariousV2(LanguageCount);
    }

    private void ReadPageFieldVariousV2(Int32 LanguageCount) {
      UInt16 FieldVariousFlags = this.ReadFieldVariousFlags();

      //UseHover
      if((FieldVariousFlags & 0x0080) == 0x0080) {
        SByte FieldVariousHoverX = this.ReadFieldVariousHover("FieldVariousHoverX");
        SByte FieldVariousHoverY = this.ReadFieldVariousHover("FieldVariousHoverY");
      }

      //UseFieldID
      if((FieldVariousFlags & 0x0010) == 0x0010) {
        Int32 FieldVariousIDLength = this.ReadFieldVariousLength("FieldVariousIDLength");
        String FieldVariousID = this.ReadFieldVariousText("FieldVariousID", FieldVariousIDLength);
      }

      //UseTipText
      if((FieldVariousFlags & 0x0001) == 0x0001) {
        Int32 FieldVariousCount = 1;

        //UseTipTextLanguage
        if((FieldVariousFlags & 0x0002) == 0x0002 && LanguageCount > 1)
          FieldVariousCount = LanguageCount;

        for(Int32 i = 0; i < FieldVariousCount; i++) {
          Int32 FieldVariousTipLength = this.ReadFieldVariousLength("FieldVariousTipLength");
          String FieldVariousTip = this.ReadFieldVariousText("FieldVariousTip", FieldVariousTipLength);
        }
      }

      //UseLinkPath
      if((FieldVariousFlags & 0x0004) == 0x0004) {
        Int32 FieldVariousCount = 1;

        //UseLinkPathLanguage
        if((FieldVariousFlags & 0x0008) == 0x0008 && LanguageCount > 1)
          FieldVariousCount = LanguageCount;

        for(Int32 i = 0; i < FieldVariousCount; i++) {
          Int32 FieldVariousLinkLength = this.ReadFieldVariousLength("FieldVariousLinkLength");
          String FieldVariousLink = this.ReadFieldVariousText("FieldVariousLink", FieldVariousLinkLength);
        }
      }
    }


    private void ReadDataV1() {
      UInt16 Flags = this.ReadFlags();

      //UsePreview
      if((Flags & 0x0001) == 0x0001) {
        Byte PreviewImageMode = this.ReadPreviewImageMode();

        Int32 PreviewImageWidth = this.ReadPreviewImageWidth();
        Int32 PreviewImageHeight = this.ReadPreviewImageHeight();

        Boolean PreviewImagePng = this.ReadPreviewImagePng();
        Boolean PreviewImageJpeg = this.ReadPreviewImageJpeg();

        Byte PreviewImageQuality = this.ReadPreviewImageQuality();
        Byte PreviewImageFormat = this.ReadPreviewImageFormat();

        Int32 PreviewImageSize = this.ReadPreviewImageSize();

        this.ReadPreviewImage(PreviewImageSize);
      }

      //UseIcon
      if((Flags & 0x0002) == 0x0002) {
        Byte IconImageMode = this.ReadIconImageMode();

        Int32 IconImageWidth = this.ReadIconImageWidth();
        Int32 IconImageHeight = this.ReadIconImageHeight();

        Int32 IconImageSize = this.ReadIconImageSize();

        this.ReadIconImage(IconImageSize);
      }

      //UseInfo
      if((Flags & 0x0004) == 0x0004) {
        Int32 InfoProducerLength = this.ReadInfoProducerLength();
        String InfoProducer = this.ReadInfoProducer(InfoProducerLength);

        Int32 InfoInternetLength = this.ReadInfoInternetLength();
        String InfoInternet = this.ReadInfoInternet(InfoInternetLength);

        Int32 InfoMailLength = this.ReadInfoMailLength();
        String InfoMail = this.ReadInfoMail(InfoMailLength);
      }

      //MainSetting
      UInt32 SettingBackColor = this.ReadSettingBackColor();
      UInt32 SettingWindowColor = this.ReadSettingWindowColor();

      Int32 SettingOutlineLeft = this.ReadSettingBorderSize("SettingOutlineLeft");
      Int32 SettingOutlineTop = this.ReadSettingBorderSize("SettingOutlineTop");
      Int32 SettingOutlineRight = this.ReadSettingBorderSize("SettingOutlineRight");
      Int32 SettingOutlineBottom = this.ReadSettingBorderSize("SettingOutlineBottom");

      Int32 SettingPageLeft = this.ReadSettingBorderSize("SettingPageLeft");
      Int32 SettingPageTop = this.ReadSettingBorderSize("SettingPageTop");
      Int32 SettingPageRight = this.ReadSettingBorderSize("SettingPageRight");
      Int32 SettingPageBottom = this.ReadSettingBorderSize("SettingPageBottom");

      //Language
      Int32 LanguageCount = this.ReadLanguageCount();

      for(Int32 i = 0; i < LanguageCount; i++) {
        Byte LanguageLetterLength = this.ReadLanguageLetterLength();
        String LanguageLetter = this.ReadLanguageLetter(LanguageLetterLength);
      }

      //LetterImage
      Int32 LetterImageSize = this.ReadLetterImageSize();
      Int32 FontCount = 0;

      if(LetterImageSize > 0) {
        this.ReadLetterImage(LetterImageSize);

        Boolean LetterUseInfo = this.ReadLetterUseInfo();

        //LetterFont
        FontCount = this.ReadLetterFontCount();

        for(Int32 i = 0; i < FontCount; i++) {
          Int32 FontNameLength = this.ReadFontNameLength();
          String FontName = this.ReadFontName(FontNameLength);
          Single FontSize = this.ReadFontSize();
          Byte FontStyle = this.ReadFontStyle();
          Byte FontUnit = this.ReadFontUnit();

          Int16 FontHeight = this.ReadFontHeight();
          Int16 FontAscent = this.ReadFontAscent();

          //Letter
          Int32 LetterCount = this.ReadLetterCount();

          for(Int32 j = 0; j < LetterCount; j++) {
            Char Letter = this.ReadLetter();

            Int16 LetterWidthA = this.ReadLetterWidth("LetterWidthA");
            Int16 LetterWidthB = this.ReadLetterWidth("LetterWidthB");
            Int16 LetterWidthC = this.ReadLetterWidth("LetterWidthC");

            Int32 LetterImagePosition = this.ReadLetterImagePosition();

            if(LetterImagePosition >= 0) {
              Int16 LetterImageWidth = this.ReadLetterImageWidth();
              Int16 LetterImageHeight = this.ReadLetterImageHeight();
            }
          }
        }
      }

      //Image for page field
      Int32 FieldImageCount = this.ReadImageCount("FieldImageCount");

      for(Int32 i = 0; i < FieldImageCount; i++) {
        Byte FieldImageMode = this.ReadImageMode("FieldImageMode");

        Int32 FieldImageWidth = this.ReadImageWidth("FieldImageWidth");
        Int32 FieldImageHeight = this.ReadImageHeight("FieldImageHeight");

        Int32 FieldImageSize = this.ReadImageSize("FieldImageSize");

        this.ReadImage("FieldImage", FieldImageSize);
      }

      //Original image for page field content
      Int32 OriginalImageCount = this.ReadImageCount("OriginalImageCount");

      for(Int32 i = 0; i < OriginalImageCount; i++) {
        Byte OriginalImageMode = this.ReadImageMode("OriginalImageMode");

        Int32 OriginalImageWidth = this.ReadImageWidth("OriginalImageWidth");
        Int32 OriginalImageHeight = this.ReadImageHeight("OriginalImageHeight");

        Int32 OriginalImageSize = this.ReadImageSize("OriginalImageSize");

        this.ReadImage("OriginalImage", OriginalImageSize);
      }

      //Display image for page field content
      Int32 DisplayImageCount = this.ReadImageCount("DisplayImageCount");

      for(Int32 i = 0; i < DisplayImageCount; i++) {
        Byte DisplayImageMode = this.ReadImageMode("DisplayImageMode");

        Int32 DisplayImageWidth = this.ReadImageWidth("DisplayImageWidth");
        Int32 DisplayImageHeight = this.ReadImageHeight("DisplayImageHeight");

        Int32 DisplayImageSize = this.ReadImageSize("DisplayImageSize");

        this.ReadImage("DisplayImage", DisplayImageSize);
      }

      //Player file for page field content
      Int32 PlayerFileCount = this.ReadPlayerFileCount();

      for(Int32 i = 0; i < PlayerFileCount; i++) {
        Byte PlayerFileType = this.ReadPlayerFileType();
        Int64 PlayerDuration = this.ReadPlayerFileDuration();

        Int32 PlayerDisplayWidth = this.ReadPlayerDisplayWidth();
        Int32 PlayerDisplayHeight = this.ReadPlayerDisplayHeight();

        Int32 PlayerFileSize = this.ReadPlayerFileSize();

        this.ReadPlayerFile(PlayerFileType, PlayerFileSize);
      }

      //Expanded image for outline with children
      Int32 ExpandedImageCount = this.ReadImageCount("ExpandedImageCount");

      for(Int32 i = 0; i < ExpandedImageCount; i++) {
        Byte ExpandedImageMode = this.ReadImageMode("ExpandedImageMode");

        Int32 ExpandedImageWidth = this.ReadImageWidth("ExpandedImageWidth");
        Int32 ExpandedImageHeight = this.ReadImageHeight("ExpandedImageHeight");

        Int32 ExpandedImageSize = this.ReadImageSize("ExpandedImageSize");

        this.ReadImage("ExpandedImage", ExpandedImageSize);
      }

      //Outline item
      Int32 OutlineCount = this.ReadOutlineCount();

      for(Int32 i = 0; i < OutlineCount; i++) {
        UInt16 OutlineFlags = this.ReadOutlineFlags();

        Int32 OutlineDistanceLeft = this.ReadOutlineDistance("OutlineDistanceLeft");
        Int32 OutlineDistanceTop = this.ReadOutlineDistance("OutlineDistanceTop");
        Int32 OutlineDistanceRight = this.ReadOutlineDistance("OutlineDistanceRight");
        Int32 OutlineDistanceBottom = this.ReadOutlineDistance("OutlineDistanceBottom");

        Int32 OutlineFrameLeft = this.ReadOutlineFrame("OutlineFrameLeft");
        Int32 OutlineFrameTop = this.ReadOutlineFrame("OutlineFrameTop");
        Int32 OutlineFrameRight = this.ReadOutlineFrame("OutlineFrameRight");
        Int32 OutlineFrameBottom = this.ReadOutlineFrame("OutlineFrameBottom");

        Int32 OutlineExpandedX = this.ReadOutlineExpandedLocation("OutlineExpandedX");
        Int32 OutlineExpandedY = this.ReadOutlineExpandedLocation("OutlineExpandedY");
        Int32 OutlineExpandedCount = this.ReadOutlineExpandedCount();
        Int32 OutlineExpandedIndex = this.ReadOutlineExpandedIndex(ExpandedImageCount);

        Int32 OutlineCurveX = this.ReadOutlineCurveLocation("OutlineCurveX");
        Int32 OutlineCurveY = this.ReadOutlineCurveLocation("OutlineCurveY");

        UInt32 OutlineColorMiddle = this.ReadOutlineBackColor("OutlineColorMiddle");
        UInt32 OutlineColorBegin = this.ReadOutlineBackColor("OutlineColorBegin");
        UInt32 OutlineColorEnd = this.ReadOutlineBackColor("OutlineColorEnd");

        Int32 OutlineShadowX = this.ReadOutlineShadowLocation("OutlineShadowX");
        Int32 OutlineShadowY = this.ReadOutlineShadowLocation("OutlineShadowY");
        UInt32 OutlineShadowColor = this.ReadOutlineBackColor("OutlineShadowColor");

        UInt32 OutlineBorderColor = this.ReadOutlineBackColor("OutlineBorderColor");

        Int32 OutlineTextCount = 1;

        //UseLanguage
        if((OutlineFlags & 0x0020) == 0x0020 && LanguageCount > 1)
          OutlineTextCount = LanguageCount;

        for(Int32 j = 0; j < OutlineTextCount; j++) {
          UInt32 OutlineTextColor = this.ReadOutlineBackColor("OutlineTextColor");

          Int32 OutlineTextWidth = this.ReadOutlineTextWidth();
          Int32 OutlineTextHeight = this.ReadOutlineTextHeight();
          Int32 OutlineTextLength = this.ReadOutlineTextLength();

          Int32 OutlineFontIndex = this.ReadOutlineFontIndex(FontCount);

          Int32 OutlineLetterIndexCount = this.ReadOutlineLetterIndexCount();
          if(OutlineLetterIndexCount > 0) this.ReadOutlineLetterIndexMemory(OutlineLetterIndexCount);

          Int32 OutlineInfoTypeSize = this.ReadOutlineInfoTypeSize();

          //Not used
          //if(OutlineInfoTypeSize > 0) this.ReadOutlineInfoTypeMemoryV1(OutlineInfoTypeSize);

          Int32 OutlineInfoIndexCount = this.ReadOutlineInfoIndexCount();

          if(OutlineInfoIndexCount > 1) {
            Int32 OutlineInfoIndexSize = this.ReadOutlineInfoIndexSize();
            if(OutlineInfoIndexSize > 0) this.ReadOutlineInfoIndexMemory(OutlineInfoIndexSize, OutlineTextLength, OutlineInfoIndexCount);
          }

          Int32 OutlineTextStringLength = this.ReadOutlineTextStringLength();
          if(OutlineTextStringLength > 0) this.ReadOutlineTextString(OutlineTextStringLength);
        }

        //UsePage
        if((OutlineFlags & 0x0040) == 0x0040) {
          //PageLine
          Int32 PageLineCount = this.ReadPageLineCount();

          for(Int32 j = 0; j < PageLineCount; j++) {
            Int32 PageLineHeightCount = 1;

            if(LanguageCount > 1)
              PageLineHeightCount = LanguageCount;

            for(Int32 k = 0; k < PageLineHeightCount; k++) {
              Int32 PageLineHeight = this.ReadPageLineHeight();
            }

            //PageField
            Int32 PageFieldCount = this.ReadPageFieldCount();

            for(Int32 k = 0; k < PageFieldCount; k++) {
              Boolean UseItemLanguage = false;

              Byte PageFieldType = this.ReadPageFieldType();

              switch(PageFieldType) {
                case 0: { //Empty
                  UseItemLanguage = this.ReadPageFieldEmptyV1();
                  break;
                }
                case 1: { //Text
                  UseItemLanguage = this.ReadPageFieldTextV1(LanguageCount);
                  break;
                }
                case 2: { //Image
                  UseItemLanguage = this.ReadPageFieldImageV1(LanguageCount);
                  break;
                }
                case 3: { //Player
                  UseItemLanguage = this.ReadPageFieldPlayerV1(LanguageCount);
                  break;
                }
              }

              this.ReadPageFieldSettingV1(UseItemLanguage, LanguageCount);
            }
          }
        }

        //Children
        Int32 OutlineChildCount = this.ReadOutlineChildCount();

        OutlineCount += OutlineChildCount;
      }

      if(this.Position != this.Size) throw new Exception("Wrong FileSize");
    }

    private void ReadDataV2() {
      UInt16 Flags = this.ReadFlags();

      //UsePreview
      if((Flags & 0x0001) == 0x0001) {
        Byte PreviewImageMode = this.ReadPreviewImageMode();

        Int32 PreviewImageWidth = this.ReadPreviewImageWidth();
        Int32 PreviewImageHeight = this.ReadPreviewImageHeight();

        Boolean PreviewImagePng = this.ReadPreviewImagePng();
        Boolean PreviewImageJpeg = this.ReadPreviewImageJpeg();

        Byte PreviewImageQuality = this.ReadPreviewImageQuality();
        Byte PreviewImageFormat = this.ReadPreviewImageFormat();

        Int32 PreviewImageSize = this.ReadPreviewImageSize();

        this.ReadPreviewImage(PreviewImageSize);
      }

      //UseIcon
      if((Flags & 0x0002) == 0x0002) {
        Byte IconImageMode = this.ReadIconImageMode();
        Int32 IconImageSize = this.ReadIconImageSize();

        this.ReadIconImage(IconImageSize);
      }

      //UseInfo
      if((Flags & 0x0004) == 0x0004) {
        Int32 InfoProducerLength = this.ReadInfoProducerLength();
        String InfoProducer = this.ReadInfoProducer(InfoProducerLength);

        Int32 InfoInternetLength = this.ReadInfoInternetLength();
        String InfoInternet = this.ReadInfoInternet(InfoInternetLength);

        Int32 InfoMailLength = this.ReadInfoMailLength();
        String InfoMail = this.ReadInfoMail(InfoMailLength);
      }

      //MainSetting
      UInt32 SettingBackColor = this.ReadSettingBackColor();
      UInt32 SettingWindowColor = this.ReadSettingWindowColor();

      Int32 SettingOutlineLeft = this.ReadSettingBorderSize("SettingOutlineLeft");
      Int32 SettingOutlineTop = this.ReadSettingBorderSize("SettingOutlineTop");
      Int32 SettingOutlineRight = this.ReadSettingBorderSize("SettingOutlineRight");
      Int32 SettingOutlineBottom = this.ReadSettingBorderSize("SettingOutlineBottom");

      Int32 SettingPageLeft = this.ReadSettingBorderSize("SettingPageLeft");
      Int32 SettingPageTop = this.ReadSettingBorderSize("SettingPageTop");
      Int32 SettingPageRight = this.ReadSettingBorderSize("SettingPageRight");
      Int32 SettingPageBottom = this.ReadSettingBorderSize("SettingPageBottom");

      Double DocumentFactor = this.ReadSettingDocumentFactor();

      //Language
      Int32 LanguageCount = this.ReadLanguageCount();

      for(Int32 i = 0; i < LanguageCount; i++) {
        Byte LanguageLetterLength = this.ReadLanguageLetterLength();
        String LanguageLetter = this.ReadLanguageLetter(LanguageLetterLength);
      }

      //LetterImage
      Int32 LetterImageSize = this.ReadLetterImageSize();
      Int32 FontCount = 0;

      if(LetterImageSize > 0) {
        this.ReadLetterImage(LetterImageSize);

        //LetterFont
        FontCount = this.ReadLetterFontCount();

        for(Int32 i = 0; i < FontCount; i++) {
          Int32 FontNameLength = this.ReadFontNameLength();
          String FontName = this.ReadFontName(FontNameLength);
          Single FontSize = this.ReadFontSize();
          Byte FontStyle = this.ReadFontStyle();
          Byte FontUnit = this.ReadFontUnit();

          //Letter
          Int32 LetterCount = this.ReadLetterCount();

          for(Int32 j = 0; j < LetterCount; j++) {
            Char Letter = this.ReadLetter();
            Double LetterWidth = this.ReadLetterWidth();

            Int32 LetterImagePosition = this.ReadLetterImagePosition("LetterImageAPosition");

            if(LetterImagePosition >= 0) {
              Int32 LetterImageWidth = this.ReadLetterImageWidth("LetterImageAWidth");
              Int32 LetterImageHeight = this.ReadLetterImageHeight("LetterImageAHeight");
              Double LetterImageX = this.ReadLetterImageX("LetterImageAX");
              Int32 LetterImageY = this.ReadLetterImageY("LetterImageAY");

              LetterImagePosition = this.ReadLetterImagePosition("LetterImageBPosition");
              LetterImageWidth = this.ReadLetterImageWidth("LetterImageBWidth");
              LetterImageHeight = this.ReadLetterImageHeight("LetterImageBHeight");
              LetterImageX = this.ReadLetterImageX("LetterImageBX");
              LetterImageY = this.ReadLetterImageY("LetterImageBY");

              LetterImagePosition = this.ReadLetterImagePosition("LetterImageCPosition");
              LetterImageWidth = this.ReadLetterImageWidth("LetterImageCWidth");
              LetterImageHeight = this.ReadLetterImageHeight("LetterImageCHeight");
              LetterImageX = this.ReadLetterImageX("LetterImageCX");
              LetterImageY = this.ReadLetterImageY("LetterImageCY");

              LetterImagePosition = this.ReadLetterImagePosition("LetterImageDPosition");
              LetterImageWidth = this.ReadLetterImageWidth("LetterImageDWidth");
              LetterImageHeight = this.ReadLetterImageHeight("LetterImageDHeight");
              LetterImageX = this.ReadLetterImageX("LetterImageDX");
              LetterImageY = this.ReadLetterImageY("LetterImageDY");
            }
          }
        }
      }

      //Image for page field
      Int32 FieldImageCount = this.ReadImageCount("FieldImageCount");

      for(Int32 i = 0; i < FieldImageCount; i++) {
        Byte FieldImageMode = this.ReadImageMode("FieldImageMode");

        Int32 FieldImageWidth = this.ReadImageWidth("FieldImageWidth");
        Int32 FieldImageHeight = this.ReadImageHeight("FieldImageHeight");

        Int32 FieldImageSize = this.ReadImageSize("FieldImageSize");

        this.ReadImage("FieldImage", FieldImageSize);

        Int32 FieldImageItemWidth = this.ReadImageItemWidth("FieldImageItemWidth");
        Int32 FieldImageItemCount = this.ReadImageItemCount("FieldImageItemCount");

        Int32 FieldImagePathLength = this.ReadImagePathLength("FieldImagePathLength");

        if(FieldImagePathLength > 0) this.ReadImagePath("FieldImagePath", FieldImagePathLength);
      }

      //Original image for page field content
      Int32 OriginalImageCount = this.ReadImageCount("OriginalImageCount");

      for(Int32 i = 0; i < OriginalImageCount; i++) {
        Byte OriginalImageMode = this.ReadImageMode("OriginalImageMode");

        Int32 OriginalImageWidth = this.ReadImageWidth("OriginalImageWidth");
        Int32 OriginalImageHeight = this.ReadImageHeight("OriginalImageHeight");

        Int32 OriginalImageSize = this.ReadImageSize("OriginalImageSize");

        this.ReadImage("OriginalImage", OriginalImageSize);

        Int32 OriginalImagePathLength = this.ReadImagePathLength("OriginalImagePathLength");

        if(OriginalImagePathLength > 0) this.ReadImagePath("OriginalImagePath", OriginalImagePathLength);
      }

      //Display image for page field content
      Int32 DisplayImageCount = this.ReadImageCount("DisplayImageCount");

      for(Int32 i = 0; i < DisplayImageCount; i++) {
        Byte DisplayImageMode = this.ReadImageMode("DisplayImageMode");

        Int32 DisplayImageWidth = this.ReadImageWidth("DisplayImageWidth");
        Int32 DisplayImageHeight = this.ReadImageHeight("DisplayImageHeight");

        Int32 DisplayImageSize = this.ReadImageSize("DisplayImageSize");

        this.ReadImage("DisplayImage", DisplayImageSize);

        Int32 DisplayImagePathLength = this.ReadImagePathLength("DisplayImagePathLength");

        if(DisplayImagePathLength > 0) this.ReadImagePath("DisplayImagePath", DisplayImagePathLength);
      }

      //Download image for page field content
      Int32 DownloadImageCount = this.ReadImageCount("DownloadImageCount");

      for(Int32 i = 0; i < DownloadImageCount; i++) {
        Byte DownloadImageMode = this.ReadImageMode("DownloadImageMode");

        Int32 DownloadImageWidth = this.ReadImageWidth("DownloadImageWidth");
        Int32 DownloadImageHeight = this.ReadImageHeight("DownloadImageHeight");

        Int32 DownloadImageSize = this.ReadImageSize("DownloadImageSize");

        this.ReadImage("DownloadImage", DownloadImageSize);

        Int32 DownloadImagePathLength = this.ReadImagePathLength("DownloadImagePathLength");

        if(DownloadImagePathLength <= 0) throw new Exception("Wrong DownloadPath");

        this.ReadImagePath("DownloadImagePath", DownloadImagePathLength);
      }

      //Player file for page field content
      Int32 PlayerFileCount = this.ReadPlayerFileCount();

      for(Int32 i = 0; i < PlayerFileCount; i++) {
        Byte PlayerFileType = this.ReadPlayerFileType();
        Int64 PlayerDuration = this.ReadPlayerFileDuration();

        Int32 PlayerDisplayWidth = this.ReadPlayerDisplayWidth();
        Int32 PlayerDisplayHeight = this.ReadPlayerDisplayHeight();

        Int32 PlayerFileSize = this.ReadPlayerFileSize();

        this.ReadPlayerFile(PlayerFileType, PlayerFileSize);
      }

      //Expanded image for outline with children
      Int32 ExpandedImageCount = this.ReadImageCount("ExpandedImageCount");

      for(Int32 i = 0; i < ExpandedImageCount; i++) {
        Byte ExpandedImageMode = this.ReadImageMode("ExpandedImageMode");

        Int32 ExpandedImageWidth = this.ReadImageWidth("ExpandedImageWidth");
        Int32 ExpandedImageHeight = this.ReadImageHeight("ExpandedImageHeight");

        Int32 ExpandedImageSize = this.ReadImageSize("ExpandedImageSize");

        this.ReadImage("ExpandedImage", ExpandedImageSize);

        Int32 ExpandedImageItemWidth = this.ReadImageItemWidth("ExpandedImageItemWidth");
        Int32 ExpandedImageItemCount = this.ReadImageItemCount("ExpandedImageItemCount");

        Int32 ExpandedImagePathLength = this.ReadImagePathLength("ExpandedImagePathLength");

        if(ExpandedImagePathLength > 0) this.ReadImagePath("ExpandedImagePath", ExpandedImagePathLength);
      }

      //Outline item
      Int32 OutlineCount = this.ReadOutlineCount();

      for(Int32 i = 0; i < OutlineCount; i++) {
        UInt16 OutlineFlags = this.ReadOutlineFlags();

        Int32 OutlineDistanceLeft = this.ReadOutlineDistance("OutlineDistanceLeft");
        Int32 OutlineDistanceTop = this.ReadOutlineDistance("OutlineDistanceTop");
        Int32 OutlineDistanceRight = this.ReadOutlineDistance("OutlineDistanceRight");
        Int32 OutlineDistanceBottom = this.ReadOutlineDistance("OutlineDistanceBottom");

        Int32 OutlineFrameLeft = this.ReadOutlineFrame("OutlineFrameLeft");
        Int32 OutlineFrameTop = this.ReadOutlineFrame("OutlineFrameTop");
        Int32 OutlineFrameRight = this.ReadOutlineFrame("OutlineFrameRight");
        Int32 OutlineFrameBottom = this.ReadOutlineFrame("OutlineFrameBottom");

        Int32 OutlineExpandedX = this.ReadOutlineExpandedLocation("OutlineExpandedX");
        Int32 OutlineExpandedY = this.ReadOutlineExpandedLocation("OutlineExpandedY");
        Int32 OutlineExpandedIndex = this.ReadOutlineExpandedIndex();

        Int32 OutlineCurveX = this.ReadOutlineCurveLocation("OutlineCurveX");
        Int32 OutlineCurveY = this.ReadOutlineCurveLocation("OutlineCurveY");

        UInt32 OutlineColorMiddle = this.ReadOutlineBackColor("OutlineColorMiddle");
        UInt32 OutlineColorBegin = this.ReadOutlineBackColor("OutlineColorBegin");
        UInt32 OutlineColorEnd = this.ReadOutlineBackColor("OutlineColorEnd");

        Int32 OutlineShadowX = this.ReadOutlineShadowLocation("OutlineShadowX");
        Int32 OutlineShadowY = this.ReadOutlineShadowLocation("OutlineShadowY");
        UInt32 OutlineShadowColor = this.ReadOutlineBackColor("OutlineShadowColor");

        UInt32 OutlineBorderColor = this.ReadOutlineBackColor("OutlineBorderColor");

        Int32 OutlineTextCount = 1;

        //UseLanguage
        if((OutlineFlags & 0x0020) == 0x0020 && LanguageCount > 1)
          OutlineTextCount = LanguageCount;

        for(Int32 j = 0; j < OutlineTextCount; j++) {
          UInt32 OutlineTextColor = this.ReadOutlineBackColor("OutlineTextColor");
          Int32 OutlineTextLength = this.ReadOutlineTextLength();
          Int32 OutlineFontIndex = this.ReadOutlineFontIndex(FontCount);

          Int32 OutlineLetterIndexCount = this.ReadOutlineLetterIndexCount();
          if(OutlineLetterIndexCount > 0) this.ReadOutlineLetterIndexMemory(OutlineLetterIndexCount);

          Int32 OutlineInfoTypeSize = this.ReadOutlineInfoTypeSize();
          this.ReadOutlineInfoTypeMemoryV2(OutlineInfoTypeSize, OutlineTextLength);

          if(OutlineLetterIndexCount > 1) {
            Int32 OutlineInfoIndexSize = this.ReadOutlineInfoIndexSize();
            if(OutlineInfoIndexSize > 0) this.ReadOutlineInfoIndexMemory(OutlineInfoIndexSize, OutlineTextLength, OutlineLetterIndexCount);
          }

          Int32 OutlineTextStringLength = this.ReadOutlineTextStringLength();
          if(OutlineTextStringLength > 0) this.ReadOutlineTextString(OutlineTextStringLength);
        }

        //UsePage
        if((OutlineFlags & 0x0040) == 0x0040) {
          //PageLine
          Int32 PageLineCount = this.ReadPageLineCount();

          for(Int32 j = 0; j < PageLineCount; j++) {
            Int32 PageLineHeightCount = 1;

            if(LanguageCount > 1)
              PageLineHeightCount = LanguageCount;

            for(Int32 k = 0; k < PageLineHeightCount; k++) {
              Int32 PageLineHeight = this.ReadPageLineHeight();
            }

            //PageField
            Int32 PageFieldCount = this.ReadPageFieldCount();

            for(Int32 k = 0; k < PageFieldCount; k++) {
              Boolean UseItemLanguage = false;

              Byte PageFieldType = this.ReadPageFieldType();

              switch(PageFieldType) {
                case 0: { //Empty
                  UseItemLanguage = this.ReadPageFieldEmptyV2();
                  break;
                }
                case 1: { //Text
                  UseItemLanguage = this.ReadPageFieldTextV2(LanguageCount);
                  break;
                }
                case 2: { //Image
                  UseItemLanguage = this.ReadPageFieldImageV2(LanguageCount);
                  break;
                }
                case 3: { //Player
                  UseItemLanguage = this.ReadPageFieldPlayerV2(LanguageCount);
                  break;
                }
              }

              this.ReadPageFieldSettingV2(UseItemLanguage, LanguageCount);
            }
          }
        }

        //Children
        Int32 OutlineChildCount = this.ReadOutlineChildCount();

        OutlineCount += OutlineChildCount;
      }

      if(this.Position != this.Size) throw new Exception("Wrong FileSize");
    }


    private void ReadBegin() {
      UInt32 IDNumber = this.ReadIDNumber();

      Int32 FileCount = this.ReadFileCount();
      Int64 FileSize = this.ReadFileSize();
      Int64 FileMaxSize = this.ReadFileMaxSize();

      Byte Version = this.ReadVersion();

      switch(Version) {
        case 1: {
          this.ReadDataV1();
          return;
        }
        case 2: {
          this.ReadDataV2();
          return;
        }
      }
    }

    #endregion


    #region Public Methods

    public List<ListViewItem> Run(IFileViewerProgressBar Parent, String FilePath) {
      this.Parent = Parent;

      try {
        this.Stream = new FileStream(FilePath, FileMode.Open, FileAccess.Read);
        this.Reader = new BinaryReader(this.Stream);

        this.Size = this.Stream.Length;

        this.Parent.BeginProgressBar(this.Size);

        this.AddItemBeginRead();

        this.ReadBegin();
      } catch(Exception e) {
        this.AddItemError(e);

        this.Close();
        return this.ItemList;
      }

      this.AddItemEndOfFile();

      this.Close();
      return this.ItemList;
    }

    public Byte[] LetterMemory(Bitmap Image, Int32 Position, Int32 Width, Int32 Height) {
      //System.Drawing.Bitmap, System.Drawing.Imaging.BitmapData
      //System.Runtime.InteropServices.Marshal

      Int32 Size = Width * Height * 4;
      Byte[] Memory = new Byte[Size];

      BitmapData Data = Image.LockBits(new Rectangle(0, 0, Image.Width, Image.Height), ImageLockMode.ReadOnly, PixelFormat.Format32bppArgb);

      Marshal.Copy(new IntPtr(Data.Scan0.ToInt64() + Position), Memory, 0, Size);

      Image.UnlockBits(Data);
      return Memory;
    }

    public Color LetterColor(Color BackImage, Color LetterImage, Color TextColor) {
      //System.Drawing.Color

      if(LetterImage.R == 0 && LetterImage.G == 0 && LetterImage.B == 0)
        return TextColor;

      if(LetterImage.R == 255 && LetterImage.G == 255 && LetterImage.B == 255)
        return BackImage;

      Int32 Red = (LetterImage.R * BackImage.R + (255 - LetterImage.R) * TextColor.R) / 255;
      Int32 Green = (LetterImage.G * BackImage.G + (255 - LetterImage.G) * TextColor.G) / 255;
      Int32 Blue = (LetterImage.B * BackImage.B + (255 - LetterImage.B) * TextColor.B) / 255;
      return Color.FromArgb(Red, Green, Blue);
    }


    public void CreateSineColor(UInt32[] ColorArray, Int32 Index, Int32 Count, Double Factor, Color Begin, Color End) {
      if(ColorArray == null || ColorArray.Length == 0) return;

      if(ColorArray.Length == 1) {
        ColorArray[0] = (UInt32) Begin.ToArgb();
        return;
      }

      double A = End.A - Begin.A;
      double R = End.R - Begin.R;
      double G = End.G - Begin.G;
      double B = End.B - Begin.B;

      double dCount = (Count - 1) * (Count - 1);

      for(Int32 i = 0; i < Count; i++) {
        double Value = Math.Pow(1.0 - i * i / dCount, Factor);

        ColorArray[Index + Count - 1 - i] = (UInt32) Color.FromArgb(Convert.ToInt32(Begin.A + Value * A), Convert.ToInt32(Begin.R + Value * R), Convert.ToInt32(Begin.G + Value * G), Convert.ToInt32(Begin.B + Value * B)).ToArgb();
      }
    }

    public void CreateCosineColor(UInt32[] ColorArray, Int32 Index, Int32 Count, Double Factor, Color Begin, Color End) {
      if(ColorArray == null || ColorArray.Length == 0) return;

      if(ColorArray.Length == 1) {
        ColorArray[0] = (UInt32) Begin.ToArgb();
        return;
      }

      double A = End.A - Begin.A;
      double R = End.R - Begin.R;
      double G = End.G - Begin.G;
      double B = End.B - Begin.B;

      double dCount = (Count - 1) * (Count - 1);

      for(Int32 i = 0; i < Count; i++) {
        double Value = Math.Pow(1.0 - i * i / dCount, Factor);

        ColorArray[Index + i] = (UInt32) Color.FromArgb(Convert.ToInt32(End.A - Value * A), Convert.ToInt32(End.R - Value * R), Convert.ToInt32(End.G - Value * G), Convert.ToInt32(End.B - Value * B)).ToArgb();
      }
    }

    public UInt32[] CreateRoundColor(Int32 Count, Double Factor, Color Begin, Color Middle, Color End) {
      if(Count < 1) return new UInt32[0];

      UInt32[] ColorArray = new UInt32[Count];

      if(Count < 4) {
        if(Count == 1) {
          ColorArray[0] = (UInt32) Middle.ToArgb();
          return ColorArray;
        }
        if(Count == 2) {
          ColorArray[0] = (UInt32) Middle.ToArgb();
          ColorArray[1] = (UInt32) End.ToArgb();
          return ColorArray;
        }


        ColorArray[0] = (UInt32) Begin.ToArgb();
        ColorArray[1] = (UInt32) Middle.ToArgb();
        ColorArray[2] = (UInt32) End.ToArgb();
        return ColorArray;
      }

      Int32 SizeA = Count / 2;
      Int32 SizeB = Count - SizeA;

      this.CreateSineColor(ColorArray, 0, SizeA, Factor, Begin, Middle);
      this.CreateCosineColor(ColorArray, SizeA, SizeB, Factor, Middle, End);
      return ColorArray;
    }

    #endregion
  }


  public class FileViewerTDHelpReader {
    #region Public Fields

    public IFileViewerProgressBar Parent;
    public List<ListViewItem> ItemList;
    public BinaryReader Reader;
    public FileStream Stream;
    public String FilePath;
    public Int64 Position;
    public Int64 Size;

    #endregion


    #region Contructor

    public FileViewerTDHelpReader() {
      this.ItemList = new List<ListViewItem>();
    }

    #endregion


    #region Private Methods

    private void AddItem(Int64 Size, String Type, String Name, String Value) {
      this.ItemList.Add(new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value }));

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }

    private void AddItem(Int64 Size, String Type, String Name, String Value, FileViewerObject Tag) {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value });

      Item.BackColor = Color.FromArgb(240, 255, 245);
      Item.Tag = Tag;

      this.ItemList.Add(Item);

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }


    private void AddItemError(String Value) {
      ListViewItem Item = new ListViewItem(new String[] { "".PadRight(10), "".PadRight(10), "".PadRight(10), "Error".PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }

    private void AddItemError(Exception e) {
      String Name = String.Empty;
      String Value = String.Empty;

      if(e.InnerException != null && e.InnerException.Message != null) {
        if(e.Message != null)
          Name = e.Message;

        Value = e.InnerException.Message;
      } else {
        if(e.Message != null)
          Value = e.Message;
      }

      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "ERROR".PadRight(10), Name.PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }


    private void AddItemBeginRead() {
      ListViewItem Item = new ListViewItem(new String[] { "0".PadRight(10), "0".PadRight(10), "".PadRight(10), "BeginRead".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }

    private void AddItemEndOfFile() {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "".PadRight(10), "EndOfFile".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }


    private void Close() {
      if(this.Stream != null) {
        this.Stream.Close();
        this.Stream = null;
      }

      if(this.Reader != null) {
        this.Reader.Close();
        this.Reader = null;
      }
    }

    #endregion

    #region Private Methods: Format

    private UInt32 ReadIDNumber() {
      UInt32 IDNumber = this.Reader.ReadUInt32();

      if(IDNumber != 0x52484454) //TDHR
        throw new Exception("IDNumber", new Exception("0x" + IDNumber.ToString("X8")));

      this.AddItem(4, "UINT32", "IDNumber", "0x" + IDNumber.ToString("X8") + " hex");
      return IDNumber;
    }

    private Int32 ReadFileCount() {
      Int32 FileCount = this.Reader.ReadInt32();

      if(FileCount < 1)
        throw new Exception("FileCount", new Exception(FileCount.ToString()));

      this.AddItem(4, "INT32", "FileCount", FileCount.ToString());
      return FileCount;
    }

    private Int64 ReadFileSize() {
      Int64 FileSize = this.Reader.ReadInt64();

      if(FileSize != this.Size)
        throw new Exception("FileSize", new Exception(FileSize.ToString()));

      this.AddItem(8, "INT64", "FileSize", FileSize.ToString("#,#") + " bytes");
      return FileSize;
    }

    private Int64 ReadFileMaxSize() {
      Int64 FileMaxSize = this.Reader.ReadInt64();

      if(FileMaxSize < this.Size)
        throw new Exception("FileMaxSize", new Exception(FileMaxSize.ToString()));

      this.AddItem(8, "INT64", "FileMaxSize", FileMaxSize.ToString("#,#") + " bytes");
      return FileMaxSize;
    }

    private Byte ReadVersion() {
      Byte Version = this.Reader.ReadByte();

      if(Version == 0 || Version > 2)
        throw new Exception("Version", new Exception(Version.ToString()));

      this.AddItem(1, "BYTE", "Version", Version.ToString());
      return Version;
    }

    private UInt16 ReadFlags() {
      UInt16 Flags = this.Reader.ReadUInt16();

      //UsePreview = 0x0001, UseIcon = 0x0002, UseInfo = 0x0004, UseTwoLetterLanguage = 0x0008

      if(Flags > 0x000F)
        throw new Exception("Flags", new Exception("0x" + Flags.ToString("X4")));

      this.AddItem(2, "UINT16", "Flags", "0x" + Flags.ToString("X4"));
      return Flags;
    }


    private Byte ReadPreviewImageMode() {
      Byte PreviewImageMode = this.Reader.ReadByte();

      //Normal = 0, Transparent = 1, Alpha = 2

      if(PreviewImageMode > 2)
        throw new Exception("PreviewImageMode", new Exception(PreviewImageMode.ToString()));

      this.AddItem(1, "BYTE", "PreviewImageMode", PreviewImageMode.ToString());
      return PreviewImageMode;
    }

    private Int32 ReadPreviewImageWidth() {
      Int32 PreviewImageWidth = this.Reader.ReadInt32();

      if(PreviewImageWidth < 1 || PreviewImageWidth > 16000)
        throw new Exception("PreviewImageWidth", new Exception(PreviewImageWidth.ToString()));

      this.AddItem(4, "INT32", "PreviewImageWidth", PreviewImageWidth.ToString());
      return PreviewImageWidth;
    }

    private Int32 ReadPreviewImageHeight() {
      Int32 PreviewImageHeight = this.Reader.ReadInt32();

      if(PreviewImageHeight < 1 || PreviewImageHeight > 16000)
        throw new Exception("PreviewImageHeight", new Exception(PreviewImageHeight.ToString()));

      this.AddItem(4, "INT32", "PreviewImageHeight", PreviewImageHeight.ToString());
      return PreviewImageHeight;
    }

    private Int32 ReadPreviewImageSize() {
      Int32 PreviewImageSize = this.Reader.ReadInt32();

      if(PreviewImageSize < 0 || PreviewImageSize > this.Size - this.Position - 4)
        throw new Exception("PreviewImageSize", new Exception(PreviewImageSize.ToString()));

      if(PreviewImageSize == 0)
        this.AddItem(4, "INT32", "PreviewImageSize", "0 byte");
      else
        this.AddItem(4, "INT32", "PreviewImageSize", PreviewImageSize.ToString("#,#") + " bytes");

      return PreviewImageSize;
    }

    private void ReadPreviewImage(Int32 PreviewImageSize) {
      try {
        Byte[] PreviewImageArray = this.Reader.ReadBytes(PreviewImageSize);

        Bitmap PreviewImage = new Bitmap(new MemoryStream(PreviewImageArray));

        this.AddItem(PreviewImageSize, "MEMORY", "PreviewImage", "", new FileViewerObject(PreviewImage));
      } catch(Exception e) {
        throw new Exception("PreviewImage", e);
      }
    }


    private Byte ReadIconImageMode() {
      Byte IconImageMode = this.Reader.ReadByte();

      //Normal = 0, Transparent = 1, Alpha = 2

      if(IconImageMode > 2)
        throw new Exception("IconImageMode", new Exception(IconImageMode.ToString()));

      this.AddItem(1, "BYTE", "IconImageMode", IconImageMode.ToString());
      return IconImageMode;
    }

    private Int32 ReadIconImageWidth() {
      Int32 IconImageWidth = this.Reader.ReadInt32();

      if(IconImageWidth < 1 || IconImageWidth > 16000)
        throw new Exception("IconImageWidth", new Exception(IconImageWidth.ToString()));

      this.AddItem(4, "INT32", "IconImageWidth", IconImageWidth.ToString());
      return IconImageWidth;
    }

    private Int32 ReadIconImageHeight() {
      Int32 IconImageHeight = this.Reader.ReadInt32();

      if(IconImageHeight < 1 || IconImageHeight > 16000)
        throw new Exception("IconImageHeight", new Exception(IconImageHeight.ToString()));

      this.AddItem(4, "INT32", "IconImageHeight", IconImageHeight.ToString());
      return IconImageHeight;
    }

    private Int32 ReadIconImageSize() {
      Int32 IconImageSize = this.Reader.ReadInt32();

      if(IconImageSize < 0 || IconImageSize > this.Size - this.Position - 4)
        throw new Exception("IconImageSize", new Exception(IconImageSize.ToString()));

      if(IconImageSize == 0)
        this.AddItem(4, "INT32", "IconImageSize", "0 byte");
      else
        this.AddItem(4, "INT32", "IconImageSize", IconImageSize.ToString("#,#") + " bytes");

      return IconImageSize;
    }

    private void ReadIconImage(Int32 IconImageSize) {
      try {
        Byte[] IconImageArray = this.Reader.ReadBytes(IconImageSize);

        Bitmap IconImage = new Bitmap(new MemoryStream(IconImageArray));

        this.AddItem(IconImageSize, "MEMORY", "IconImage", "", new FileViewerObject(IconImage));
      } catch(Exception e) {
        throw new Exception("IconImage", e);
      }
    }


    private Int32 ReadInfoProducerLength() {
      Int32 InfoProducerLength = this.Reader.ReadInt32();

      if(InfoProducerLength < -1 || InfoProducerLength > 1000)
        throw new Exception("InfoProducerLength", new Exception(InfoProducerLength.ToString()));

      this.AddItem(4, "INT32", "InfoProducerLength", InfoProducerLength.ToString());
      return InfoProducerLength;
    }

    private String ReadInfoProducer(Int32 InfoProducerLength) {
      if(InfoProducerLength == -1) {
        this.AddItem(0, "WCHAR[ ]", "InfoProducer", "Null");
        return null;
      }

      if(InfoProducerLength == 0) {
        this.AddItem(0, "WCHAR[ ]", "InfoProducer", "Empty");
        return String.Empty;
      }

      Char[] LetterArray = new Char[InfoProducerLength];

      for(Int32 i = 0; i < InfoProducerLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String InfoProducer = new String(LetterArray);

      if(InfoProducer == null || InfoProducer == String.Empty)
        throw new Exception("InfoProducer", new Exception("Empty"));

      this.AddItem(InfoProducerLength * 2, "WCHAR[ ]", "InfoProducer", InfoProducer);
      return InfoProducer;
    }

    private Int32 ReadInfoInternetLength() {
      Int32 InfoInternetLength = this.Reader.ReadInt32();

      if(InfoInternetLength < -1 || InfoInternetLength > 1000)
        throw new Exception("InfoInternetLength", new Exception(InfoInternetLength.ToString()));

      this.AddItem(4, "INT32", "InfoInternetLength", InfoInternetLength.ToString());
      return InfoInternetLength;
    }

    private String ReadInfoInternet(Int32 InfoInternetLength) {
      if(InfoInternetLength == -1) {
        this.AddItem(0, "WCHAR[ ]", "InfoInternet", "Null");
        return null;
      }

      if(InfoInternetLength == 0) {
        this.AddItem(0, "WCHAR[ ]", "InfoInternet", "Empty");
        return String.Empty;
      }

      Char[] LetterArray = new Char[InfoInternetLength];

      for(Int32 i = 0; i < InfoInternetLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String InfoInternet = new String(LetterArray);

      if(InfoInternet == null || InfoInternet == String.Empty)
        throw new Exception("InfoInternet", new Exception("Empty"));

      this.AddItem(InfoInternetLength * 2, "WCHAR[ ]", "InfoInternet", InfoInternet);
      return InfoInternet;
    }

    private Int32 ReadInfoMailLength() {
      Int32 InfoMailLength = this.Reader.ReadInt32();

      if(InfoMailLength < -1 || InfoMailLength > 1000)
        throw new Exception("InfoMailLength", new Exception(InfoMailLength.ToString()));

      this.AddItem(4, "INT32", "InfoMailLength", InfoMailLength.ToString());
      return InfoMailLength;
    }

    private String ReadInfoMail(Int32 InfoMailLength) {
      if(InfoMailLength == -1) {
        this.AddItem(0, "WCHAR[ ]", "InfoMail", "Null");
        return null;
      }

      if(InfoMailLength == 0) {
        this.AddItem(0, "WCHAR[ ]", "InfoMail", "Empty");
        return String.Empty;
      }

      Char[] LetterArray = new Char[InfoMailLength];

      for(Int32 i = 0; i < InfoMailLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String InfoMail = new String(LetterArray);

      if(InfoMail == null || InfoMail == String.Empty)
        throw new Exception("InfoMail", new Exception("Empty"));

      this.AddItem(InfoMailLength * 2, "WCHAR[ ]", "InfoMail", InfoMail);
      return InfoMail;
    }


    private UInt32 ReadSettingBackColor() {
      UInt32 SettingBackColor = this.Reader.ReadUInt32();

      if(SettingBackColor < 0xFF000000 && SettingBackColor != 0)
        throw new Exception("SettingBackColor", new Exception("0x" + SettingBackColor.ToString("X8")));

      this.AddItem(4, "UINT32", "SettingBackColor", "0x" + SettingBackColor.ToString("X8"));
      return SettingBackColor;
    }

    private UInt32 ReadSettingWindowColor() {
      UInt32 SettingWindowColor = this.Reader.ReadUInt32();

      if(SettingWindowColor < 0xFF000000 && SettingWindowColor != 0)
        throw new Exception("SettingWindowColor", new Exception("0x" + SettingWindowColor.ToString("X8")));

      this.AddItem(4, "UINT32", "SettingWindowColor", "0x" + SettingWindowColor.ToString("X8"));
      return SettingWindowColor;
    }

    private Int32 ReadSettingBorderSize(String Name) {
      Int32 SettingBorderSize = this.Reader.ReadInt32();

      if(SettingBorderSize < 0 || SettingBorderSize > 1000)
        throw new Exception(Name, new Exception(SettingBorderSize.ToString()));

      this.AddItem(4, "INT32", Name, SettingBorderSize.ToString());
      return SettingBorderSize;
    }

    private Double ReadSettingDocumentFactor() {
      Double SettingDocumentFactor = this.Reader.ReadDouble();

      if(SettingDocumentFactor < 1.0 || SettingDocumentFactor > 4.0)
        throw new Exception("SettingDocumentFactor", new Exception(SettingDocumentFactor.ToString()));

      this.AddItem(8, "DOUBLE", "SettingDocumentFactor", SettingDocumentFactor.ToString());
      return SettingDocumentFactor;
    }


    private Int32 ReadLanguageCount() {
      Int32 LanguageCount = this.Reader.ReadInt32();

      if(LanguageCount < 0 || LanguageCount > 200)
        throw new Exception("LanguageCount", new Exception(LanguageCount.ToString()));

      this.AddItem(4, "INT32", "LanguageCount", LanguageCount.ToString());
      return LanguageCount;
    }

    private Byte ReadLanguageLetterLength() {
      Byte LanguageLetterLength = this.Reader.ReadByte();

      if(LanguageLetterLength != 2)
        throw new Exception("LanguageLetterLength", new Exception(LanguageLetterLength.ToString()));

      this.AddItem(1, "BYTE", "LanguageLetterLength", LanguageLetterLength.ToString());
      return LanguageLetterLength;
    }

    private String ReadLanguageLetter(Int32 LanguageLetterLength) {
      Char[] LetterArray = new Char[LanguageLetterLength];

      for(Int32 i = 0; i < LanguageLetterLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadByte();

      String LanguageLetter = new String(LetterArray);

      if(LanguageLetter == null || LanguageLetter == String.Empty)
        throw new Exception("LanguageLetter", new Exception("Empty"));

      this.AddItem(LanguageLetterLength, "CHAR[ ]", "LanguageLetter", LanguageLetter);
      return LanguageLetter;
    }


    private Int32 ReadLetterImageSize() {
      Int32 LetterImageSize = this.Reader.ReadInt32();

      if(LetterImageSize < -1 || LetterImageSize > this.Size - this.Position - 4)
        throw new Exception("LetterImageSize", new Exception(LetterImageSize.ToString()));

      if(LetterImageSize == -1)
        this.AddItem(4, "INT32", "LetterImageSize", "Null");
      else
        if(LetterImageSize == 0)
          this.AddItem(4, "INT32", "LetterImageSize", "0 byte");
        else
          this.AddItem(4, "INT32", "LetterImageSize", LetterImageSize.ToString("#,#") + " bytes");

      return LetterImageSize;
    }

    private void ReadLetterImage(Int32 LetterImageSize) {
      try {
        Byte[] LetterImageArray = this.Reader.ReadBytes(LetterImageSize);

        Bitmap LetterImage = new Bitmap(new MemoryStream(LetterImageArray));

        this.AddItem(LetterImageSize, "MEMORY", "LetterImage", "", new FileViewerObject(LetterImage));
      } catch(Exception e) {
        throw new Exception("LetterImage", e);
      }
    }

    private Boolean ReadLetterUseInfo() {
      Byte LetterInfo = this.Reader.ReadByte();

      if(LetterInfo != 0)
        throw new Exception("LetterInfo", new Exception(LetterInfo.ToString()));

      this.AddItem(1, "BYTE", "LetterInfo", LetterInfo.ToString());
      return LetterInfo == 0 ? false : true;
    }

    private Int32 ReadLetterFontCount() {
      Int32 LetterFontCount = this.Reader.ReadInt32();

      if(LetterFontCount < 1 || LetterFontCount > 10000)
        throw new Exception("LetterFontCount", new Exception(LetterFontCount.ToString()));

      this.AddItem(4, "INT32", "LetterFontCount", LetterFontCount.ToString());
      return LetterFontCount;
    }

    private Int16 ReadFontHeight() {
      Int16 FontHeight = this.Reader.ReadInt16();

      if(FontHeight < 1 || FontHeight > 16000)
        throw new Exception("FontHeight", new Exception(FontHeight.ToString()));

      this.AddItem(2, "INT16", "FontHeight", FontHeight.ToString());
      return FontHeight;
    }

    private Int16 ReadFontAscent() {
      Int16 FontAscent = this.Reader.ReadInt16();

      if(FontAscent < 1 || FontAscent > 16000)
        throw new Exception("FontAscent", new Exception(FontAscent.ToString()));

      this.AddItem(2, "INT16", "FontAscent", FontAscent.ToString());
      return FontAscent;
    }


    private Byte ReadFontStyle() {
      Byte FontStyle = this.Reader.ReadByte();

      //Regular = 0x00, Bold = 0x01, Italic = 0x02, Underline = 0x04, Strikeout = 0x08

      if(FontStyle > 0x0F)
        throw new Exception("FontStyle", new Exception("0x" + FontStyle.ToString("X2")));

      this.AddItem(1, "BYTE", "FontStyle", "0x" + FontStyle.ToString("X2"));
      return FontStyle;
    }

    private Double ReadFontSize() {
      Double FontSize = this.Reader.ReadDouble();

      if(FontSize > 3000)
        throw new Exception("FontSize", new Exception(FontSize.ToString()));

      this.AddItem(8, "DOUBLE", "FontSize", FontSize.ToString());
      return FontSize;
    }

    private Int32 ReadFontEm(String Name, Int32 Min, Int32 Max) {
      Int32 FontEm = this.Reader.ReadInt32();

      if(FontEm < Min || FontEm > Max)
        throw new Exception(Name, new Exception(FontEm.ToString()));

      this.AddItem(4, "INT32", Name, FontEm.ToString());
      return FontEm;
    }


    private Int32 ReadLetterCount() {
      Int32 LetterCount = this.Reader.ReadInt32();

      if(LetterCount < 1 || LetterCount > 65536)
        throw new Exception("LetterCount", new Exception(LetterCount.ToString()));

      this.AddItem(4, "INT32", "LetterCount", LetterCount.ToString());
      return LetterCount;
    }

    private Int16 ReadLetterWidth(String Name) {
      Int16 LetterWidth = this.Reader.ReadInt16();

      if(LetterWidth < -16000 || LetterWidth > 16000)
        throw new Exception(Name, new Exception(LetterWidth.ToString()));

      this.AddItem(2, "INT16", Name, LetterWidth.ToString());
      return LetterWidth;
    }

    private Int32 ReadLetterImagePosition() {
      Int32 LetterImagePosition = this.Reader.ReadInt32();

      if(LetterImagePosition < -1)
        throw new Exception("LetterImagePosition", new Exception(LetterImagePosition.ToString()));

      this.AddItem(4, "INT32", "LetterImagePosition", LetterImagePosition.ToString());
      return LetterImagePosition;
    }

    private Int16 ReadLetterImageWidth() {
      Int16 LetterImageWidth = this.Reader.ReadInt16();

      if(LetterImageWidth < 1 || LetterImageWidth > 16000)
        throw new Exception("LetterImageWidth", new Exception(LetterImageWidth.ToString()));

      this.AddItem(2, "INT16", "LetterImageWidth", LetterImageWidth.ToString());
      return LetterImageWidth;
    }

    private Int16 ReadLetterImageHeight() {
      Int16 LetterImageHeight = this.Reader.ReadInt16();

      if(LetterImageHeight < 1 || LetterImageHeight > 16000)
        throw new Exception("LetterImageHeight", new Exception(LetterImageHeight.ToString()));

      this.AddItem(2, "INT16", "LetterImageHeight", LetterImageHeight.ToString());
      return LetterImageHeight;
    }


    private Double ReadLetterWidth() {
      Double LetterWidth = this.Reader.ReadDouble();

      if(LetterWidth < 0 || LetterWidth > 3000)
        throw new Exception("LetterWidth", new Exception(LetterWidth.ToString()));

      this.AddItem(8, "DOUBLE", "LetterWidth", LetterWidth.ToString());
      return LetterWidth;
    }

    private Int32 ReadLetterImagePosition(String Name) {
      Int32 LetterImagePosition = this.Reader.ReadInt32();

      if(LetterImagePosition < -1)
        throw new Exception(Name, new Exception(LetterImagePosition.ToString()));

      this.AddItem(4, "INT32", Name, LetterImagePosition.ToString());
      return LetterImagePosition;
    }

    private Int32 ReadLetterImageWidth(String Name) {
      Int32 LetterImageWidth = this.Reader.ReadInt32();

      if(LetterImageWidth < 1 || LetterImageWidth > 3000)
        throw new Exception(Name, new Exception(LetterImageWidth.ToString()));

      this.AddItem(4, "INT32", Name, LetterImageWidth.ToString());
      return LetterImageWidth;
    }

    private Int32 ReadLetterImageHeight(String Name) {
      Int32 LetterImageHeight = this.Reader.ReadInt32();

      if(LetterImageHeight < 1 || LetterImageHeight > 3000)
        throw new Exception(Name, new Exception(LetterImageHeight.ToString()));

      this.AddItem(4, "INT32", Name, LetterImageHeight.ToString());
      return LetterImageHeight;
    }

    private Double ReadLetterImageX(String Name) {
      Double LetterImageX = this.Reader.ReadDouble();

      if(LetterImageX < -3000 || LetterImageX > 3000)
        throw new Exception(Name, new Exception(LetterImageX.ToString()));

      this.AddItem(8, "DOUBLE", Name, LetterImageX.ToString());
      return LetterImageX;
    }

    private Int32 ReadLetterImageY(String Name) {
      Int32 LetterImageY = this.Reader.ReadInt32();

      if(LetterImageY < -3000 || LetterImageY > 3000)
        throw new Exception(Name, new Exception(LetterImageY.ToString()));

      this.AddItem(4, "INT32", Name, LetterImageY.ToString());
      return LetterImageY;
    }


    private Int32 ReadImageCount(String Name) {
      Int32 ImageCount = this.Reader.ReadInt32();

      if(ImageCount < 0)
        throw new Exception(Name, new Exception(ImageCount.ToString()));

      this.AddItem(4, "INT32", Name, ImageCount.ToString());
      return ImageCount;
    }

    private Byte ReadImageMode(String Name) {
      Byte ImageMode = this.Reader.ReadByte();

      //Normal = 0, Transparent = 1, Alpha = 2

      if(ImageMode > 2)
        throw new Exception(Name, new Exception(ImageMode.ToString()));

      this.AddItem(1, "BYTE", Name, ImageMode.ToString());
      return ImageMode;
    }

    private Int32 ReadImageWidth(String Name) {
      Int32 ImageWidth = this.Reader.ReadInt32();

      if(ImageWidth < 1 || ImageWidth > 16000)
        throw new Exception(Name, new Exception(ImageWidth.ToString()));

      this.AddItem(4, "INT32", Name, ImageWidth.ToString());
      return ImageWidth;
    }

    private Int32 ReadImageHeight(String Name) {
      Int32 ImageHeight = this.Reader.ReadInt32();

      if(ImageHeight < 1 || ImageHeight > 16000)
        throw new Exception(Name, new Exception(ImageHeight.ToString()));

      this.AddItem(4, "INT32", Name, ImageHeight.ToString());
      return ImageHeight;
    }

    private Int32 ReadImageSize(String Name) {
      Int32 ImageSize = this.Reader.ReadInt32();

      if(ImageSize < 0 || ImageSize > this.Size - this.Position - 4)
        throw new Exception(Name, new Exception(ImageSize.ToString()));

      if(ImageSize == 0)
        this.AddItem(4, "INT32", Name, "0 byte");
      else
        this.AddItem(4, "INT32", Name, ImageSize.ToString("#,#") + " bytes");

      return ImageSize;
    }

    private void ReadImage(String Name, Int32 ImageSize) {
      try {
        Byte[] ImageArray = this.Reader.ReadBytes(ImageSize);

        Bitmap Image = FileViewerTDPicture.LoadImage(new MemoryStream(ImageArray));

        if(Image == null) Image = new Bitmap(new MemoryStream(ImageArray));

        this.AddItem(ImageSize, "MEMORY", Name, "", new FileViewerObject(Image));
      } catch(Exception e) {
        throw new Exception(Name, e);
      }
    }

    private Int32 ReadImageItemWidth(String Name) {
      Int32 ImageItemWidth = this.Reader.ReadInt32();

      if(ImageItemWidth < 1 || ImageItemWidth > 32000)
        throw new Exception(Name, new Exception(ImageItemWidth.ToString()));

      this.AddItem(4, "INT32", Name, ImageItemWidth.ToString());
      return ImageItemWidth;
    }

    private Int32 ReadImageItemCount(String Name) {
      Int32 ImageItemCount = this.Reader.ReadInt32();

      if(ImageItemCount < 1 || ImageItemCount > 32000)
        throw new Exception(Name, new Exception(ImageItemCount.ToString()));

      this.AddItem(4, "INT32", Name, ImageItemCount.ToString());
      return ImageItemCount;
    }

    private Int32 ReadImagePathLength(String Name) {
      Int32 ImagePathLength = this.Reader.ReadInt32();

      if(ImagePathLength < -1 || ImagePathLength > 32000)
        throw new Exception(Name, new Exception(ImagePathLength.ToString()));

      this.AddItem(4, "INT32", Name, ImagePathLength.ToString());
      return ImagePathLength;
    }

    private String ReadImagePath(String Name, Int32 ImagePathLength) {
      Char[] LetterArray = new Char[ImagePathLength];

      for(Int32 i = 0; i < ImagePathLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String ImagePath = new String(LetterArray);

      if(ImagePath == null || ImagePath == String.Empty)
        throw new Exception(Name, new Exception("Empty"));

      this.AddItem(ImagePathLength * 2, "WCHAR[ ]", Name, ImagePath);
      return ImagePath;
    }


    private Int32 ReadPlayerFileCount() {
      Int32 PlayerFileCount = this.Reader.ReadInt32();

      if(PlayerFileCount < 0)
        throw new Exception("PlayerFileCount", new Exception(PlayerFileCount.ToString()));

      this.AddItem(4, "INT32", "PlayerFileCount", PlayerFileCount.ToString());
      return PlayerFileCount;
    }

    private Byte ReadPlayerFileType() {
      Byte PlayerFileType = this.Reader.ReadByte();

      //Sound = 0, SoundImage = 1, Video = 2, VideoAudio = 3, Gif = 4, TDAnimation = 5,

      if(PlayerFileType > 5)
        throw new Exception("PlayerFileType", new Exception(PlayerFileType.ToString()));

      this.AddItem(1, "BYTE", "PlayerFileType", PlayerFileType.ToString());
      return PlayerFileType;
    }

    private Int64 ReadPlayerFileDuration() {
      Int64 PlayerDuration = this.Reader.ReadInt64();

      if(PlayerDuration < 0)
        throw new Exception("PlayerDuration", new Exception(PlayerDuration.ToString()));

      this.AddItem(8, "INT64", "PlayerDuration", PlayerDuration.ToString());
      return PlayerDuration;
    }

    private Int32 ReadPlayerDisplayWidth() {
      Int32 PlayerDisplayWidth = this.Reader.ReadInt32();

      if(PlayerDisplayWidth < 0 || PlayerDisplayWidth > 16000)
        throw new Exception("PlayerDisplayWidth", new Exception(PlayerDisplayWidth.ToString()));

      this.AddItem(4, "INT32", "PlayerDisplayWidth", PlayerDisplayWidth.ToString());
      return PlayerDisplayWidth;
    }

    private Int32 ReadPlayerDisplayHeight() {
      Int32 PlayerDisplayHeight = this.Reader.ReadInt32();

      if(PlayerDisplayHeight < 0 || PlayerDisplayHeight > 16000)
        throw new Exception("PlayerDisplayHeight", new Exception(PlayerDisplayHeight.ToString()));

      this.AddItem(4, "INT32", "PlayerDisplayHeight", PlayerDisplayHeight.ToString());
      return PlayerDisplayHeight;
    }

    private Int32 ReadPlayerFileSize() {
      Int32 PlayerFileSize = this.Reader.ReadInt32();

      if(PlayerFileSize < 0 || PlayerFileSize > this.Size - this.Position - 4)
        throw new Exception("PlayerFileSize", new Exception(PlayerFileSize.ToString()));

      if(PlayerFileSize == 0)
        this.AddItem(4, "INT32", "PlayerFileSize", "0 byte");
      else
        this.AddItem(4, "INT32", "PlayerFileSize", PlayerFileSize.ToString("#,#") + " bytes");

      return PlayerFileSize;
    }

    private Byte[] ReadPlayerFile(Byte PlayerFileType, Int32 FileSize) {
      Byte[] FileMemory = this.Reader.ReadBytes(FileSize);
      String FileName = String.Empty;

      switch(PlayerFileType) {
        case 0: {
          FileName = "Sound.wav";
          break;
        }
        case 1: {
          FileName = "Sound.wav";
          break;
        }
        case 2: {
          FileName = "Video.avi";
          break;
        }
        case 3: {
          FileName = "Video.avi";
          break;
        }
        case 4: {
          FileName = "Animation.gif";
          break;
        }
        case 5: {
          FileName = "Animation.tda";
          break;
        }
      }

      this.AddItem(FileSize, "BYTE[ ]", "PlayerFile", FileName, new FileViewerObject(FileViewerObjectType.PicturePaintFile, new FileViewerTDPicturePaintFileObject(FileName, FileMemory)));
      return FileMemory;
    }


    private Int32 ReadOutlineCount() {
      Int32 OutlineCount = this.Reader.ReadInt32();

      if(OutlineCount < 0)
        throw new Exception("OutlineCount", new Exception(OutlineCount.ToString()));

      this.AddItem(4, "INT32", "OutlineCount", OutlineCount.ToString());
      return OutlineCount;
    }

    private UInt16 ReadOutlineFlags() {
      UInt16 OutlineFlags = this.Reader.ReadUInt16();

      //UseRound = 0x0001, UseShadow = 0x0002, UseBorder = 0x0004, UseBackground = 0x0008
      //UseBackgroundStyle = 0x0010, UseLanguage = 0x0020, UsePage = 0x0040

      if(OutlineFlags > 0x007F)
        throw new Exception("OutlineFlags", new Exception("0x" + OutlineFlags.ToString("X4")));

      this.AddItem(2, "UINT16", "OutlineFlags", "0x" + OutlineFlags.ToString("X4"));
      return OutlineFlags;
    }

    private Int32 ReadOutlineDistance(String Name) {
      Int32 OutlineDistance = this.Reader.ReadInt32();

      if(OutlineDistance < 0 || OutlineDistance > 10000)
        throw new Exception(Name, new Exception(OutlineDistance.ToString()));

      this.AddItem(4, "INT32", Name, OutlineDistance.ToString());
      return OutlineDistance;
    }

    private Int32 ReadOutlineFrame(String Name) {
      Int32 OutlineFrame = this.Reader.ReadInt32();

      if(OutlineFrame < 0 || OutlineFrame > 1000)
        throw new Exception(Name, new Exception(OutlineFrame.ToString()));

      this.AddItem(4, "INT32", Name, OutlineFrame.ToString());
      return OutlineFrame;
    }

    private Int32 ReadOutlineExpandedLocation(String Name) {
      Int32 OutlineExpandedLocation = this.Reader.ReadInt32();

      if(OutlineExpandedLocation < 0 || OutlineExpandedLocation > 10000)
        throw new Exception(Name, new Exception(OutlineExpandedLocation.ToString()));

      this.AddItem(4, "INT32", Name, OutlineExpandedLocation.ToString());
      return OutlineExpandedLocation;
    }

    private Int32 ReadOutlineExpandedCount() {
      Int32 OutlineExpandedCount = this.Reader.ReadInt32();

      if(OutlineExpandedCount < 2 || OutlineExpandedCount > 999)
        throw new Exception("OutlineExpandedCount", new Exception(OutlineExpandedCount.ToString()));

      this.AddItem(4, "INT32", "OutlineExpandedCount", OutlineExpandedCount.ToString());
      return OutlineExpandedCount;
    }

    private Int32 ReadOutlineExpandedIndex(Int32 ExpandedImageCount) {
      Int32 OutlineExpandedIndex = this.Reader.ReadInt32();

      if(OutlineExpandedIndex < -1 || OutlineExpandedIndex >= ExpandedImageCount)
        throw new Exception("OutlineExpandedIndex", new Exception(OutlineExpandedIndex.ToString()));

      this.AddItem(4, "INT32", "OutlineExpandedIndex", OutlineExpandedIndex.ToString());
      return OutlineExpandedIndex;
    }

    private Int32 ReadOutlineCurveLocation(String Name) {
      Int32 OutlineCurveLocation = this.Reader.ReadInt32();

      if(OutlineCurveLocation < 0 || OutlineCurveLocation > 100)
        throw new Exception(Name, new Exception(OutlineCurveLocation.ToString()));

      this.AddItem(4, "INT32", Name, OutlineCurveLocation.ToString());
      return OutlineCurveLocation;
    }

    private UInt32 ReadOutlineBackColor(String Name) {
      UInt32 OutlineBackColor = this.Reader.ReadUInt32();

      if(OutlineBackColor < 0xFF000000)
        throw new Exception(Name, new Exception("0x" + OutlineBackColor.ToString("X8")));

      this.AddItem(4, "UINT32", Name, "0x" + OutlineBackColor.ToString("X8"));
      return OutlineBackColor;
    }

    private Int32 ReadOutlineShadowLocation(String Name) {
      Int32 OutlineShadowLocation = this.Reader.ReadInt32();

      if(OutlineShadowLocation < 0 || OutlineShadowLocation > 1000)
        throw new Exception(Name, new Exception(OutlineShadowLocation.ToString()));

      this.AddItem(4, "INT32", Name, OutlineShadowLocation.ToString());
      return OutlineShadowLocation;
    }


    private Int32 ReadOutlineTextWidth() {
      Int32 OutlineTextWidth = this.Reader.ReadInt32();

      if(OutlineTextWidth < 1 || OutlineTextWidth > 16000)
        throw new Exception("OutlineTextWidth", new Exception(OutlineTextWidth.ToString()));

      this.AddItem(4, "INT32", "OutlineTextWidth", OutlineTextWidth.ToString());
      return OutlineTextWidth;
    }

    private Int32 ReadOutlineTextHeight() {
      Int32 OutlineTextHeight = this.Reader.ReadInt32();

      if(OutlineTextHeight < 1 || OutlineTextHeight > 16000)
        throw new Exception("OutlineTextHeight", new Exception(OutlineTextHeight.ToString()));

      this.AddItem(4, "INT32", "OutlineTextHeight", OutlineTextHeight.ToString());
      return OutlineTextHeight;
    }

    private Int32 ReadOutlineTextLength() {
      Int32 OutlineTextLength = this.Reader.ReadInt32();

      if(OutlineTextLength < 0 || OutlineTextLength > 1000)
        throw new Exception("OutlineTextLength", new Exception(OutlineTextLength.ToString()));

      this.AddItem(4, "INT32", "OutlineTextLength", OutlineTextLength.ToString());
      return OutlineTextLength;
    }

    private Int32 ReadOutlineFontIndex(Int32 LetterFontCount) {
      Int32 OutlineFontIndex = this.Reader.ReadInt32();

      if(OutlineFontIndex < 0 || OutlineFontIndex >= LetterFontCount)
        throw new Exception("OutlineFontIndex", new Exception(OutlineFontIndex.ToString()));

      this.AddItem(4, "INT32", "OutlineFontIndex", OutlineFontIndex.ToString());
      return OutlineFontIndex;
    }

    private Int32 ReadOutlineLetterIndexCount() {
      Int32 OutlineLetterIndexCount = this.Reader.ReadInt32();

      if(OutlineLetterIndexCount < 1)
        throw new Exception("OutlineLetterIndexCount", new Exception(OutlineLetterIndexCount.ToString()));

      this.AddItem(4, "INT32", "OutlineLetterIndexCount", OutlineLetterIndexCount.ToString());
      return OutlineLetterIndexCount;
    }

    private void ReadOutlineLetterIndexMemory(Int32 OutlineLetterIndexCount) {
      Int32[] OutlineLetterIndexMemory = new Int32[OutlineLetterIndexCount];

      for(Int32 i = 0; i < OutlineLetterIndexCount; i++)
        OutlineLetterIndexMemory[i] = this.Reader.ReadInt32();

      this.AddItem(OutlineLetterIndexCount * 4, "INT32[ ]", "OutlineLetterIndexMemory", "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType.OutlineLetterIndex, OutlineLetterIndexMemory)));
    }

    private Int32 ReadOutlineInfoTypeSize() {
      Int32 OutlineInfoTypeSize = this.Reader.ReadInt32();

      if(OutlineInfoTypeSize < -1 || OutlineInfoTypeSize > this.Size - this.Position - 4)
        throw new Exception("OutlineInfoTypeSize", new Exception(OutlineInfoTypeSize.ToString()));

      if(OutlineInfoTypeSize == -1)
        this.AddItem(4, "INT32", "OutlineInfoTypeSize", "-1 (empty)");
      else
        if(OutlineInfoTypeSize == 0)
          this.AddItem(4, "INT32", "OutlineInfoTypeSize", "0 byte");
        else
          this.AddItem(4, "INT32", "OutlineInfoTypeSize", OutlineInfoTypeSize.ToString("#,#") + " bytes");

      return OutlineInfoTypeSize;
    }

    private void ReadOutlineInfoTypeMemoryV1(Int32 OutlineInfoTypeSize) {
      Byte[] OutlineInfoTypeMemory = this.Reader.ReadBytes(OutlineInfoTypeSize);

      //Info(0): 0x00 Letter, 0x01 Seperator, 0x02 Space, 0x03 Control, 0x04 NewLine (Flag)
      //Info(1): Font index
      //Info(2): Color index

      //Not Used
      //this.AddItem(OutlineInfoTypeSize, "BYTE[ ]", "OutlineInfoTypeMemory", "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType.OutlineInfoType, OutlineInfoTypeMemory)));
    }

    private void ReadOutlineInfoTypeMemoryV2(Int32 OutlineInfoTypeSize, Int32 OutlineTextLength) {
      Byte[] OutlineInfoTypeMemory = this.Reader.ReadBytes(OutlineInfoTypeSize);

      //Info(0): 0x00 Letter, 0x01 Seperator, 0x02 Space, 0x03 Control
      //Info(1): Font index
      //Info(2): Color index

      this.AddItem(OutlineInfoTypeSize, "BYTE[ ]", "OutlineInfoTypeMemory", "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType.OutlineInfoType, new FileViewerTDCompressTextObject(null, FileViewerTDCompressTextType.Info, OutlineInfoTypeMemory, 4, OutlineTextLength))));
    }

    private Int32 ReadOutlineInfoIndexCount() {
      Int32 OutlineInfoIndexCount = this.Reader.ReadInt32();

      if(OutlineInfoIndexCount < 0)
        throw new Exception("OutlineInfoIndexCount", new Exception(OutlineInfoIndexCount.ToString()));

      this.AddItem(4, "INT32", "OutlineInfoIndexCount", OutlineInfoIndexCount.ToString());
      return OutlineInfoIndexCount;
    }

    private Int32 ReadOutlineInfoIndexSize() {
      Int32 OutlineInfoIndexSize = this.Reader.ReadInt32();

      if(OutlineInfoIndexSize < 0 || OutlineInfoIndexSize > this.Size - this.Position - 4)
        throw new Exception("OutlineInfoIndexSize", new Exception(OutlineInfoIndexSize.ToString()));

      if(OutlineInfoIndexSize == 0)
        this.AddItem(4, "INT32", "OutlineInfoIndexSize", "0 byte");
      else
        this.AddItem(4, "INT32", "OutlineInfoIndexSize", OutlineInfoIndexSize.ToString("#,#") + " bytes");

      return OutlineInfoIndexSize;
    }

    private void ReadOutlineInfoIndexMemory(Int32 OutlineInfoIndexSize, Int32 OutlineTextLength, Int32 OutlineInfoIndexCount) {
      Byte[] OutlineInfoIndexMemory = this.Reader.ReadBytes(OutlineInfoIndexSize);

      this.AddItem(OutlineInfoIndexSize, "BYTE[ ]", "OutlineInfoIndexMemory", "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType.OutlineInfoIndex, new FileViewerTDCompressTextObject(null, FileViewerTDCompressTextType.Index, OutlineInfoIndexMemory, OutlineInfoIndexCount, OutlineTextLength))));
    }


    private Int32 ReadOutlineChildCount() {
      Int32 OutlineChildCount = this.Reader.ReadInt32();

      if(OutlineChildCount < 0)
        throw new Exception("OutlineChildCount", new Exception(OutlineChildCount.ToString()));

      this.AddItem(4, "INT32", "OutlineChildCount", OutlineChildCount.ToString());
      return OutlineChildCount;
    }


    private Int32 ReadPageLineCount() {
      Int32 PageLineCount = this.Reader.ReadInt32();

      if(PageLineCount < 0 || PageLineCount > 1000)
        throw new Exception("PageLineCount", new Exception(PageLineCount.ToString()));

      this.AddItem(4, "INT32", "PageLineCount", PageLineCount.ToString());
      return PageLineCount;
    }

    private Int32 ReadPageLineHeight() {
      Int32 PageLineHeight = this.Reader.ReadInt32();

      if(PageLineHeight < 1)
        throw new Exception("PageLineHeight", new Exception(PageLineHeight.ToString()));

      this.AddItem(4, "INT32", "PageLineHeight", PageLineHeight.ToString());
      return PageLineHeight;
    }

    private Int32 ReadPageFieldCount() {
      Int32 PageFieldCount = this.Reader.ReadInt32();

      if(PageFieldCount < 0)
        throw new Exception("PageFieldCount", new Exception(PageFieldCount.ToString()));

      this.AddItem(4, "INT32", "PageFieldCount", PageFieldCount.ToString());
      return PageFieldCount;
    }


    private Byte ReadPageFieldType() {
      Byte PageFieldType = this.Reader.ReadByte();

      //Empty = 0, Text = 1, Image = 2, Player = 3

      if(PageFieldType > 3)
        throw new Exception("PageFieldType", new Exception(PageFieldType.ToString()));

      this.AddItem(1, "BYTE", "PageFieldType", PageFieldType.ToString());
      return PageFieldType;
    }

    private Int32 ReadPageFieldWidth(String Name) {
      Int32 PageFieldWidth = this.Reader.ReadInt32();

      if(PageFieldWidth < 1 || PageFieldWidth > 800000)
        throw new Exception(Name, new Exception(PageFieldWidth.ToString()));

      this.AddItem(4, "INT32", Name, PageFieldWidth.ToString());
      return PageFieldWidth;
    }

    private Int32 ReadPageFieldHeight(String Name) {
      Int32 PageFieldHeight = this.Reader.ReadInt32();

      if(PageFieldHeight < 1 || PageFieldHeight > 800000)
        throw new Exception(Name, new Exception(PageFieldHeight.ToString()));

      this.AddItem(4, "INT32", Name, PageFieldHeight.ToString());
      return PageFieldHeight;
    }


    private UInt16 ReadFieldTextFlags() {
      UInt16 FieldTextFlags = this.Reader.ReadUInt16();

      //Alignment_Left = 0x0001, Alignment_Right = 0x0002, Alignment_Center = 0x0004, Alignment_Justified = 0x0008
      //UseLanguage = 0x0010, UseSizeable = 0x0020, UseSeparator = 0x0040

      if(FieldTextFlags > 0x007F)
        throw new Exception("FieldTextFlags", new Exception("0x" + FieldTextFlags.ToString("X4")));

      this.AddItem(2, "UINT16", "FieldTextFlags", "0x" + FieldTextFlags.ToString("X4"));
      return FieldTextFlags;
    }

    private Int32 ReadFieldTextColorCount() {
      Int32 FieldTextColorCount = this.Reader.ReadInt32();

      if(FieldTextColorCount < 1 || FieldTextColorCount > 256)
        throw new Exception("FieldTextColorCount", new Exception(FieldTextColorCount.ToString()));

      this.AddItem(4, "INT32", "FieldTextColorCount", FieldTextColorCount.ToString());
      return FieldTextColorCount;
    }

    private void ReadFieldTextColorMemory(Int32 FieldTextColorCount) {
      UInt32[] FieldTextColorArray = new UInt32[FieldTextColorCount];

      for(Int32 i = 0; i < FieldTextColorCount; i++)
        FieldTextColorArray[i] = this.Reader.ReadUInt32();

      this.AddItem(FieldTextColorCount * 4, "UINT32[ ]", "FieldTextColorMemory", "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType.FieldTextColor, FieldTextColorArray)));
    }

    private Int32 ReadFieldTextFontCount() {
      Int32 FieldTextFontCount = this.Reader.ReadInt32();

      if(FieldTextFontCount < 1 || FieldTextFontCount > 256)
        throw new Exception("FieldTextFontCount", new Exception(FieldTextFontCount.ToString()));

      this.AddItem(4, "INT32", "FieldTextFontCount", FieldTextFontCount.ToString());
      return FieldTextFontCount;
    }

    private void ReadFieldTextFontMemory(Int32 FieldTextFontCount) {
      Int32[] FieldTextFontArray = new Int32[FieldTextFontCount];

      for(Int32 i = 0; i < FieldTextFontCount; i++)
        FieldTextFontArray[i] = this.Reader.ReadInt32();

      this.AddItem(FieldTextFontCount * 4, "INT32[ ]", "FieldTextFontMemory", "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType.FieldTextFont, FieldTextFontArray)));
    }

    private Int32 ReadFieldTextLetterCount() {
      Int32 FieldTextLetterCount = this.Reader.ReadInt32();

      if(FieldTextLetterCount < 1 || FieldTextLetterCount > 65535)
        throw new Exception("FieldTextLetterCount", new Exception(FieldTextLetterCount.ToString()));

      this.AddItem(4, "INT32", "FieldTextLetterCount", FieldTextLetterCount.ToString());
      return FieldTextLetterCount;
    }

    private void ReadFieldTextLetterMemory(Int32 FieldTextLetterCount) {
      Int32[] FieldTextLetterArray = new Int32[FieldTextLetterCount];

      for(Int32 i = 0; i < FieldTextLetterCount; i++)
        FieldTextLetterArray[i] = this.Reader.ReadInt32();

      this.AddItem(FieldTextLetterCount * 4, "INT32[ ]", "FieldTextLetterMemory", "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType.FieldTextLetter, FieldTextLetterArray)));
    }

    private Int32 ReadFieldTextLength() {
      Int32 FieldTextLength = this.Reader.ReadInt32();

      if(FieldTextLength < 1)
        throw new Exception("FieldTextLength", new Exception(FieldTextLength.ToString()));

      this.AddItem(4, "INT32", "FieldTextLength", FieldTextLength.ToString());
      return FieldTextLength;
    }

    private Int32 ReadFieldTextInfoSize(String Name) {
      Int32 FieldTextInfoSize = this.Reader.ReadInt32();

      if(FieldTextInfoSize < 1 || FieldTextInfoSize > this.Size - this.Position - 4)
        throw new Exception(Name, new Exception(FieldTextInfoSize.ToString()));

      this.AddItem(4, "INT32", Name, FieldTextInfoSize.ToString("#,#") + " bytes");
      return FieldTextInfoSize;
    }

    private void ReadFieldTextInfoMemory(String Name, Int32 FieldTextInfoSize, Int32 FieldTextLength, Int32 FieldTextInfoCount, FileViewerTDHelpWriterType Type) {
      Byte[] FieldTextInfoMemory = this.Reader.ReadBytes(FieldTextInfoSize);

      //Info Type: 0x00 Letter, 0x01 Seperator, 0x02 Space, 0x03 Control, 0x04 NewLine (Flag)
      //Info Font: Font index
      //Info Color: Color index

      this.AddItem(FieldTextInfoSize, "BYTE[ ]", Name, "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(Type, new FileViewerTDCompressTextObject(null, FileViewerTDCompressTextType.Index, FieldTextInfoMemory, FieldTextInfoCount, FieldTextLength))));
    }

    private Int32 ReadFieldTextIndexCount() {
      Int32 FieldTextIndexCount = this.Reader.ReadInt32();

      if(FieldTextIndexCount < 0)
        throw new Exception("FieldTextIndexCount", new Exception(FieldTextIndexCount.ToString()));

      this.AddItem(4, "INT32", "FieldTextIndexCount", FieldTextIndexCount.ToString());
      return FieldTextIndexCount;
    }

    private void ReadFieldTextIndexMemory(Int32 FieldTextIndexSize, Int32 FieldTextLength, Int32 FieldTextIndexCount) {
      Byte[] FieldTextIndexMemory = this.Reader.ReadBytes(FieldTextIndexSize);

      this.AddItem(FieldTextIndexSize, "BYTE[ ]", "FieldTextIndexMemory", "", new FileViewerObject(FileViewerObjectType.HelpWriter, new FileViewerTDHelpWriterObject(FileViewerTDHelpWriterType.FieldTextLetterIndex, new FileViewerTDCompressTextObject(null, FileViewerTDCompressTextType.Index, FieldTextIndexMemory, FieldTextIndexCount, FieldTextLength))));
    }

    private Int32 ReadFieldTextSeparatorIndex() {
      Int32 FieldTextSeparatorIndex = this.Reader.ReadInt32();

      if(FieldTextSeparatorIndex < 0)
        throw new Exception("FieldTextSeparatorIndex", new Exception(FieldTextSeparatorIndex.ToString()));

      this.AddItem(4, "INT32", "FieldTextSeparatorIndex", FieldTextSeparatorIndex.ToString());
      return FieldTextSeparatorIndex;
    }


    private UInt16 ReadFieldImageFlags() {
      UInt16 FieldImageFlags = this.Reader.ReadUInt16();

      //UseLanguage = 0x0001, UseFullscreen = 0x0002, UseSizeable = 0x0004, UseExpanded = 0x0008
      //UseEdit = 0x0010, UseDownload = 0x0020, UseJpeg = 0x0040, UsePng = 0x0080

      if(FieldImageFlags > 0x00FF)
        throw new Exception("FieldImageFlags", new Exception("0x" + FieldImageFlags.ToString("X4")));

      this.AddItem(2, "UINT16", "FieldImageFlags", "0x" + FieldImageFlags.ToString("X4"));
      return FieldImageFlags;
    }

    private Int32 ReadFieldImageUrlLength() {
      Int32 FieldImageUrlLength = this.Reader.ReadInt32();

      if(FieldImageUrlLength < 1 || FieldImageUrlLength > 16000)
        throw new Exception("FieldImageUrlLength", new Exception(FieldImageUrlLength.ToString()));

      this.AddItem(4, "INT32", "FieldImageUrlLength", FieldImageUrlLength.ToString());
      return FieldImageUrlLength;
    }

    private String ReadFieldImageUrl(Int32 FieldImageUrlLength) {
      Char[] LetterArray = new Char[FieldImageUrlLength];

      for(Int32 i = 0; i < FieldImageUrlLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String FieldImageUrl = new String(LetterArray);

      if(FieldImageUrl == null || FieldImageUrl == String.Empty)
        throw new Exception("FieldImageUrl", new Exception("Empty"));

      this.AddItem(FieldImageUrlLength * 2, "WCHAR[ ]", "FieldImageUrl", FieldImageUrl);
      return FieldImageUrl;
    }

    private Int32 ReadFieldImageOriginal(String Name) {
      Int32 FieldImageOriginal = this.Reader.ReadInt32();

      if(FieldImageOriginal < 1 || FieldImageOriginal > 16000)
        throw new Exception(Name, new Exception(FieldImageOriginal.ToString()));

      this.AddItem(4, "INT32", Name, FieldImageOriginal.ToString());
      return FieldImageOriginal;
    }

    private Int32 ReadFieldImageDisplayIndex() {
      Int32 FieldImageDisplayIndex = this.Reader.ReadInt32();

      if(FieldImageDisplayIndex < 0)
        throw new Exception("FieldImageDisplayIndex", new Exception(FieldImageDisplayIndex.ToString()));

      this.AddItem(4, "INT32", "FieldImageDisplayIndex", FieldImageDisplayIndex.ToString());
      return FieldImageDisplayIndex;
    }


    private Boolean ReadFieldPlayerUseImage() {
      Byte FieldPlayerUseImage = this.Reader.ReadByte();

      if(FieldPlayerUseImage > 1)
        throw new Exception("FieldPlayerUseImage", new Exception(FieldPlayerUseImage.ToString()));

      this.AddItem(1, "BYTE", "FieldPlayerUseImage", FieldPlayerUseImage.ToString());
      return FieldPlayerUseImage == 0 ? false : true;
    }

    private UInt16 ReadFieldPlayerFlagsV1() {
      UInt16 FieldPlayerFlags = this.Reader.ReadUInt16();

      //UseLanguage = 0x0001, UseFullscreen = 0x0002, UseSizeable = 0x0004, UseExpanded = 0x0008
      //UseDownload = 0x0010, UseRepeat = 0x0020, UsePosition = 0x0040, UseVolume = 0x0080

      if(FieldPlayerFlags > 0x00FF)
        throw new Exception("FieldPlayerFlags", new Exception("0x" + FieldPlayerFlags.ToString("X4")));

      this.AddItem(2, "UINT16", "FieldPlayerFlags", "0x" + FieldPlayerFlags.ToString("X4"));
      return FieldPlayerFlags;
    }

    private UInt16 ReadFieldPlayerFlagsV2() {
      UInt16 FieldPlayerFlags = this.Reader.ReadUInt16();

      //UseLanguage = 0x0001, UseFullscreen = 0x0002, UseSizeable = 0x0004, UseExpanded = 0x0008
      //UseDownload = 0x0010, UseRepeat = 0x0020, UsePosition = 0x0040, UseVolume = 0x0080
      //UseStreaming = 0x0100

      if(FieldPlayerFlags > 0x01FF)
        throw new Exception("FieldPlayerFlags", new Exception("0x" + FieldPlayerFlags.ToString("X4")));

      this.AddItem(2, "UINT16", "FieldPlayerFlags", "0x" + FieldPlayerFlags.ToString("X4"));
      return FieldPlayerFlags;
    }

    private Int32 ReadFieldPlayerUrlLength() {
      Int32 FieldPlayerUrlLength = this.Reader.ReadInt32();

      if(FieldPlayerUrlLength < -1 || FieldPlayerUrlLength > 16000)
        throw new Exception("FieldPlayerUrlLength", new Exception(FieldPlayerUrlLength.ToString()));

      this.AddItem(4, "INT32", "FieldPlayerUrlLength", FieldPlayerUrlLength.ToString());
      return FieldPlayerUrlLength;
    }

    private String ReadFieldPlayerUrl(Int32 FieldPlayerUrlLength) {
      Char[] LetterArray = new Char[FieldPlayerUrlLength];

      for(Int32 i = 0; i < FieldPlayerUrlLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String FieldPlayerUrl = new String(LetterArray);

      if(FieldPlayerUrl == null || FieldPlayerUrl == String.Empty)
        throw new Exception("FieldImageUrl", new Exception("Empty"));

      this.AddItem(FieldPlayerUrlLength * 2, "WCHAR[ ]", "FieldPlayerUrl", FieldPlayerUrl);
      return FieldPlayerUrl;
    }

    private Byte ReadFieldPlayerType() {
      Byte FieldPlayerType = this.Reader.ReadByte();

      //Sound = 0, SoundImage = 1, Video = 2, VideoAudio = 3, Gif = 4, TDAnimation = 5,

      if(FieldPlayerType > 5)
        throw new Exception("FieldPlayerType", new Exception(FieldPlayerType.ToString()));

      this.AddItem(1, "BYTE", "FieldPlayerType", FieldPlayerType.ToString());
      return FieldPlayerType;
    }

    private Int64 ReadFieldPlayerDuration() {
      Int64 FieldPlayerDuration = this.Reader.ReadInt64();

      if(FieldPlayerDuration < 0)
        throw new Exception("FieldPlayerDuration", new Exception(FieldPlayerDuration.ToString()));

      this.AddItem(8, "INT64", "FieldPlayerDuration", FieldPlayerDuration.ToString());
      return FieldPlayerDuration;
    }

    private Int32 ReadFieldPlayerDisplayWidth() {
      Int32 FieldPlayerDisplayWidth = this.Reader.ReadInt32();

      if(FieldPlayerDisplayWidth < 0 || FieldPlayerDisplayWidth > 16000)
        throw new Exception("FieldPlayerDisplayWidth", new Exception(FieldPlayerDisplayWidth.ToString()));

      this.AddItem(4, "INT32", "FieldPlayerDisplayWidth", FieldPlayerDisplayWidth.ToString());
      return FieldPlayerDisplayWidth;
    }

    private Int32 ReadFieldPlayerDisplayHeight() {
      Int32 FieldPlayerDisplayHeight = this.Reader.ReadInt32();

      if(FieldPlayerDisplayHeight < 0 || FieldPlayerDisplayHeight > 16000)
        throw new Exception("FieldPlayerDisplayHeight", new Exception(FieldPlayerDisplayHeight.ToString()));

      this.AddItem(4, "INT32", "FieldPlayerDisplayHeight", FieldPlayerDisplayHeight.ToString());
      return FieldPlayerDisplayHeight;
    }

    private Int32 ReadFieldPlayerIndex() {
      Int32 FieldPlayerIndex = this.Reader.ReadInt32();

      if(FieldPlayerIndex < 0)
        throw new Exception("FieldPlayerIndex", new Exception(FieldPlayerIndex.ToString()));

      this.AddItem(4, "INT32", "FieldPlayerIndex", FieldPlayerIndex.ToString());
      return FieldPlayerIndex;
    }


    private UInt16 ReadFieldVariousFlags() {
      UInt16 FieldVariousFlags = this.Reader.ReadUInt16();

      //UseTipText = 0x0001, UseTipTextLanguage = 0x0002, UseLinkPath = 0x0004, UseLinkPathLanguage = 0x0008
      //UseFieldID = 0x0010, UseFieldIDHorizontal = 0x0020, UseFieldIDVertical = 0x0040, UseHover = 0x0080

      if(FieldVariousFlags > 0x00FF)
        throw new Exception("FieldVariousFlags", new Exception("0x" + FieldVariousFlags.ToString("X4")));

      this.AddItem(2, "UINT16", "FieldVariousFlags", "0x" + FieldVariousFlags.ToString("X4"));
      return FieldVariousFlags;
    }

    private SByte ReadFieldVariousHover(String Name) {
      SByte FieldVariousHover = this.Reader.ReadSByte();

      if(FieldVariousHover < 0)
        throw new Exception(Name, new Exception(FieldVariousHover.ToString()));

      this.AddItem(1, "INT8", Name, FieldVariousHover.ToString());
      return FieldVariousHover;
    }

    private Int32 ReadFieldVariousLength(String Name) {
      Int32 FieldVariousLength = this.Reader.ReadInt32();

      if(FieldVariousLength < 1 || FieldVariousLength > 16000)
        throw new Exception(Name, new Exception(FieldVariousLength.ToString()));

      this.AddItem(4, "INT32", Name, FieldVariousLength.ToString());
      return FieldVariousLength;
    }

    private String ReadFieldVariousText(String Name, Int32 FieldVariousLength) {
      Char[] LetterArray = new Char[FieldVariousLength];

      for(Int32 i = 0; i < FieldVariousLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String FieldVariousText = new String(LetterArray);

      if(FieldVariousText == null || FieldVariousText == String.Empty)
        throw new Exception(Name, new Exception("Empty"));

      this.AddItem(FieldVariousLength * 2, "WCHAR[ ]", Name, FieldVariousText);
      return FieldVariousText;
    }


    private UInt16 ReadFieldFlags() {
      UInt16 FieldFlags = this.Reader.ReadUInt16();

      //AlignmentHorizontal_Left = 0x0000, AlignmentHorizontal_Center = 00001, AlignmentHorizontal_Right = 0x0002
      //AlignmentVertical_Top = 0x0000, AlignmentVertical_Center = 0x0010, AlignmentVertical_Bottom = 0x0020
      //UseCurve = 0x0004, UseShadow = 0x0008, UseBackground = 0x0040, UseBorder = 0x0080
      //UseFrame = 0x0100, UseID = 0x0200, UseVarious = 0x0400, UseImage = 0x0800
      //UseImageAnimate = 0x1000, UseImageForeground = 0x2000, UseImageShadow = 0x4000

      if(FieldFlags > 0x7FFF)
        throw new Exception("FieldFlags", new Exception("0x" + FieldFlags.ToString("X4")));

      this.AddItem(2, "UINT16", "FieldFlags", "0x" + FieldFlags.ToString("X4"));
      return FieldFlags;
    }

    private Int32 ReadFieldDistance(String Name) {
      Int32 FieldDistance = this.Reader.ReadInt32();

      if(FieldDistance < 0 || FieldDistance > 10000)
        throw new Exception(Name, new Exception(FieldDistance.ToString()));

      this.AddItem(4, "INT32", Name, FieldDistance.ToString());
      return FieldDistance;
    }

    private Int32 ReadFieldWidth() {
      Int32 FieldWidth = this.Reader.ReadInt32();

      if(FieldWidth < 3 || FieldWidth > 1000000)
        throw new Exception("FieldWidth", new Exception(FieldWidth.ToString()));

      this.AddItem(4, "INT32", "FieldWidth", FieldWidth.ToString());
      return FieldWidth;
    }

    private Int32 ReadFieldCurve(String Name) {
      Int32 FieldFrame = this.Reader.ReadInt32();

      if(FieldFrame < 0 || FieldFrame > 1000)
        throw new Exception(Name, new Exception(FieldFrame.ToString()));

      this.AddItem(4, "INT32", Name, FieldFrame.ToString());
      return FieldFrame;
    }

    private Int32 ReadFieldFrame(String Name) {
      Int32 FieldFrame = this.Reader.ReadInt32();

      if(FieldFrame < 0 || FieldFrame > 10000)
        throw new Exception(Name, new Exception(FieldFrame.ToString()));

      this.AddItem(4, "INT32", Name, FieldFrame.ToString());
      return FieldFrame;
    }

    private Int32 ReadFieldShadow(String Name) {
      Int32 FieldShadow = this.Reader.ReadInt32();

      if(FieldShadow < 0 || FieldShadow > 1000)
        throw new Exception(Name, new Exception(FieldShadow.ToString()));

      this.AddItem(4, "INT32", Name, FieldShadow.ToString());
      return FieldShadow;
    }

    private UInt32 ReadFieldColor(String Name) {
      UInt32 FieldColor = this.Reader.ReadUInt32();

      if(FieldColor < 0xFF000000)
        throw new Exception(Name, new Exception("0x" + FieldColor.ToString("X8")));

      this.AddItem(4, "UINT32", Name, "0x" + FieldColor.ToString("X8"));
      return FieldColor;
    }

    private Int32 ReadFieldImage(String Name) {
      Int32 FieldImage = this.Reader.ReadInt32();

      if(FieldImage < 0 || FieldImage > 100000)
        throw new Exception(Name, new Exception(FieldImage.ToString()));

      this.AddItem(4, "INT32", Name, FieldImage.ToString());
      return FieldImage;
    }

    private Int32 ReadFieldImageCount() {
      Int32 FieldImageCount = this.Reader.ReadInt32();

      if(FieldImageCount < 1 || FieldImageCount > 999)
        throw new Exception("FieldImageCount", new Exception(FieldImageCount.ToString()));

      this.AddItem(4, "INT32", "FieldImageCount", FieldImageCount.ToString());
      return FieldImageCount;
    }

    private Int32 ReadFieldImageIndex() {
      Int32 FieldImageIndex = this.Reader.ReadInt32();

      if(FieldImageIndex < 0)
        throw new Exception("FieldImageIndex", new Exception(FieldImageIndex.ToString()));

      this.AddItem(4, "INT32", "FieldImageIndex", FieldImageIndex.ToString());
      return FieldImageIndex;
    }

    private Int32 ReadFieldImageTime(String Name) {
      Int32 FieldImageTime = this.Reader.ReadInt32();

      if(FieldImageTime < 0 || FieldImageTime > 3600000)
        throw new Exception(Name, new Exception(FieldImageTime.ToString()));

      this.AddItem(4, "INT32", Name, FieldImageTime.ToString());
      return FieldImageTime;
    }

    private UInt32 ReadFieldImageShadowColor() {
      UInt32 FieldImageShadowColor = this.Reader.ReadUInt32();

      this.AddItem(4, "UINT32", "FieldImageShadowColor", "0x" + FieldImageShadowColor.ToString("X8"));
      return FieldImageShadowColor;
    }

    private Int32 ReadFieldIDLength() {
      Int32 FieldIDLength = this.Reader.ReadInt32();

      if(FieldIDLength < 1 || FieldIDLength > 100)
        throw new Exception("FieldIDLength", new Exception(FieldIDLength.ToString()));

      this.AddItem(4, "INT32", "FieldIDLength", FieldIDLength.ToString());
      return FieldIDLength;
    }

    private String ReadFieldID(Int32 FieldIDLength) {
      Char[] LetterArray = new Char[FieldIDLength];

      for(Int32 i = 0; i < FieldIDLength; i++)
        LetterArray[i] = (Char) this.Reader.ReadUInt16();

      String FieldID = new String(LetterArray);

      if(FieldID == null || FieldID == String.Empty)
        throw new Exception("FieldID", new Exception("Empty"));

      this.AddItem(FieldIDLength * 2, "WCHAR[ ]", "FieldID", FieldID);
      return FieldID;
    }


    private Boolean ReadPageFieldEmptyV1() {
      Int32 FieldEmptyWidth = this.ReadPageFieldWidth("FieldEmptyWidth");
      Int32 FieldEmptyHeight = this.ReadPageFieldHeight("FieldEmptyHeight");
      return false;
    }

    private Boolean ReadPageFieldTextV1(Int32 LanguageCount) {
      Int32 FieldTextCount = 1;

      for(Int32 i = 0; i < FieldTextCount; i++) {
        UInt16 FieldTextFlags = this.ReadFieldTextFlags();

        //UseLanguage
        if(i == 0 && (FieldTextFlags & 0x0010) == 0x0010 && LanguageCount > 1)
          FieldTextCount = LanguageCount;

        Int32 FieldTextWidth = this.ReadPageFieldWidth("FieldTextWidth");
        Int32 FieldTextHeight = this.ReadPageFieldHeight("FieldTextHeight");

        Int32 FieldTextColorCount = this.ReadFieldTextColorCount();
        this.ReadFieldTextColorMemory(FieldTextColorCount);

        Int32 FieldTextFontCount = this.ReadFieldTextFontCount();
        this.ReadFieldTextFontMemory(FieldTextFontCount);

        for(Int32 j = 0; j < FieldTextFontCount; j++) {
          Int32 FieldTextLetterCount = this.ReadFieldTextLetterCount();
          this.ReadFieldTextLetterMemory(FieldTextLetterCount);
        }

        Int32 FieldTextLength = this.ReadFieldTextLength();

        Int32 FieldTextInfoTypeSize = this.ReadFieldTextInfoSize("FieldTextInfoTypeSize");
        this.ReadFieldTextInfoMemory("FieldTextInfoTypeMemory", FieldTextInfoTypeSize, FieldTextLength, 8, FileViewerTDHelpWriterType.FieldTextInfoType);

        if(FieldTextFontCount > 1) {
          Int32 FieldTextInfoFontSize = this.ReadFieldTextInfoSize("FieldTextInfoFontSize");
          this.ReadFieldTextInfoMemory("FieldTextInfoFontMemory", FieldTextInfoFontSize, FieldTextLength, FieldTextFontCount, FileViewerTDHelpWriterType.FieldTextInfoFont);
        }

        if(FieldTextColorCount > 1) {
          Int32 FieldTextInfoColorSize = this.ReadFieldTextInfoSize("FieldTextInfoColorSize");
          this.ReadFieldTextInfoMemory("FieldTextInfoColorMemory", FieldTextInfoColorSize, FieldTextLength, FieldTextColorCount, FileViewerTDHelpWriterType.FieldTextInfoColor);
        }

        Int32 FieldTextIndexCount = this.ReadFieldTextIndexCount();

        if(FieldTextIndexCount > 1) {
          Int32 FieldTextIndexSize = this.ReadFieldTextInfoSize("FieldTextIndexSize");
          this.ReadFieldTextIndexMemory(FieldTextIndexSize, FieldTextLength, FieldTextIndexCount);
        }

        //UseSeparator
        if((FieldTextFlags & 0x0040) == 0x0040) {
          for(Int32 j = 0; j < FieldTextFontCount; j++) {
            Int32 FieldTextSeparatorIndex = this.ReadFieldTextSeparatorIndex();
          }
        }
      }
      return FieldTextCount == 1 ? false : true;
    }

    private Boolean ReadPageFieldImageV1(Int32 LanguageCount) {
      Int32 FieldImageCount = 1;

      for(Int32 i = 0; i < FieldImageCount; i++) {
        UInt16 FieldImageFlags = this.ReadFieldImageFlags();

        //UseLanguage
        if(i == 0 && (FieldImageFlags & 0x0001) == 0x0001 && LanguageCount > 1)
          FieldImageCount = LanguageCount;

        Int32 FieldImageWidth = this.ReadPageFieldWidth("FieldImageWidth");
        Int32 FieldImageHeight = this.ReadPageFieldHeight("FieldImageHeight");

        //UseDownload
        if((FieldImageFlags & 0x0020) == 0x0020) {
          Int32 FieldImageUrlLength = this.ReadFieldImageUrlLength();
          this.ReadFieldImageUrl(FieldImageUrlLength);
          Int32 FieldImageQriginalWidth = this.ReadFieldImageOriginal("FieldImageQriginalWidth");
          Int32 FieldImageQriginalHeight = this.ReadFieldImageOriginal("FieldImageQriginalHeight");
        } else {
          Int32 FieldImageDisplayIndex = this.ReadFieldImageDisplayIndex();
        }
      }
      return FieldImageCount == 1 ? false : true;
    }

    private Boolean ReadPageFieldPlayerV1(Int32 LanguageCount) {
      Boolean UseItemLanguage = false;
      Boolean FieldPlayerUseImage = this.ReadFieldPlayerUseImage();

      if(FieldPlayerUseImage)
        UseItemLanguage = this.ReadPageFieldImageV1(LanguageCount);

      Int32 FieldPlayerCount = 1;

      for(Int32 i = 0; i < FieldPlayerCount; i++) {
        UInt16 FieldPlayerFlags = this.ReadFieldPlayerFlagsV1();

        //UseLanguage
        if(i == 0 && (FieldPlayerFlags & 0x0001) == 0x0001 && LanguageCount > 1)
          FieldPlayerCount = LanguageCount;

        Int32 FieldPlayerWidth = this.ReadPageFieldWidth("FieldPlayerWidth");
        Int32 FieldPlayerHeight = this.ReadPageFieldHeight("FieldPlayerHeight");

        UInt32 FieldPlayerColor = this.ReadFieldColor("FieldPlayerColor");

        //UseDownload
        if((FieldPlayerFlags & 0x0010) == 0x0010) {
          Int32 FieldPlayerUrlLength = this.ReadFieldPlayerUrlLength();
          if(FieldPlayerUrlLength > 0) this.ReadFieldPlayerUrl(FieldPlayerUrlLength);

          Byte FieldPlayerType = this.ReadFieldPlayerType();

          Int32 FieldPlayerDisplayWidth = this.ReadFieldPlayerDisplayWidth();
          Int32 FieldPlayerDisplayHeight = this.ReadFieldPlayerDisplayHeight();

          Int64 FieldPlayerDuration = this.ReadFieldPlayerDuration();
        } else {
          Int32 FieldPlayerIndex = this.ReadFieldPlayerIndex();
        }
      }
      return FieldPlayerCount == 1 ? UseItemLanguage : true;
    }


    private void ReadPageFieldSettingV1(Boolean UseItemLanguage, Int32 LanguageCount) {
      Int32 FieldSettingCount = 1;

      if(UseItemLanguage && LanguageCount > 1) FieldSettingCount = LanguageCount;

      UInt16 FieldFlags = 0;

      for(Int32 i = 0; i < FieldSettingCount; i++) {
        FieldFlags = this.ReadFieldFlags();

        Int32 FieldDistanceLeft = this.ReadFieldDistance("FieldDistanceLeft");
        Int32 FieldDistanceTop = this.ReadFieldDistance("FieldDistanceTop");
        Int32 FieldDistanceRight = this.ReadFieldDistance("FieldDistanceRight");
        Int32 FieldDistanceBottom = this.ReadFieldDistance("FieldDistanceBottom");

        Int32 FieldWidth = this.ReadFieldWidth();

        //UseCurve
        if((FieldFlags & 0x0004) == 0x0004) {
          Int32 FieldCurveX = this.ReadFieldCurve("FieldCurveX");
          Int32 FieldCurveY = this.ReadFieldCurve("FieldCurveY");
        }

        //UseFrame
        if((FieldFlags & 0x0100) == 0x0100) {
          Int32 FieldFrameLeft = this.ReadFieldFrame("FieldFrameLeft");
          Int32 FieldFrameTop = this.ReadFieldFrame("FieldFrameTop");
          Int32 FieldFrameRight = this.ReadFieldFrame("FieldFrameRight");
          Int32 FieldFrameBottom = this.ReadFieldFrame("FieldFrameBottom");
        }

        //UseShadow
        if((FieldFlags & 0x0008) == 0x0008) {
          Int32 FieldShadowX = this.ReadFieldShadow("FieldShadowX");
          Int32 FieldShadowY = this.ReadFieldShadow("FieldShadowY");
          UInt32 FieldShadowColor = this.ReadFieldColor("FieldShadowColor");
        }

        //UseBorder
        if((FieldFlags & 0x0080) == 0x0080) {
          UInt32 FieldBorderColor = this.ReadFieldColor("FieldBorderColor");
        }

        //UseBackground
        if((FieldFlags & 0x0040) == 0x0040) {
          UInt32 FieldBackColor = this.ReadFieldColor("FieldBackColor");
        }

        //UseImage
        if((FieldFlags & 0x0800) == 0x0800) {
          Int32 FieldImageX = this.ReadFieldImage("FieldImageX");
          Int32 FieldImageY = this.ReadFieldImage("FieldImageY");

          Int32 FieldImageCount = this.ReadFieldImageCount();
          Int32 FieldImageIndex = this.ReadFieldImageIndex();

          //UseImageAnimate
          if((FieldFlags & 0x1000) == 0x1000) {
            Int32 FieldImageStartTime = this.ReadFieldImageTime("FieldImageStartTime");
            Int32 FieldImageDisplayTime = this.ReadFieldImageTime("FieldImageDisplayTime");
          }

          //UseImageShadow
          if((FieldFlags & 0x4000) == 0x4000) {
            Int32 FieldImageShadowX = this.ReadFieldShadow("FieldImageShadowX");
            Int32 FieldImageShadowY = this.ReadFieldShadow("FieldImageShadowY");
            UInt32 FieldImageShadowColor = this.ReadFieldImageShadowColor();
          }
        }
      }

      //UseID
      if((FieldFlags & 0x0200) == 0x0200) {
        Int32 FieldIDLength = this.ReadFieldIDLength();

        if(FieldIDLength > 0) {
          String FieldID = this.ReadFieldID(FieldIDLength);
        }
      }

      //UseVarious
      if((FieldFlags & 0x0400) == 0x0400)
        this.ReadPageFieldVariousV1(LanguageCount);
    }

    private void ReadPageFieldVariousV1(Int32 LanguageCount) {
      UInt16 FieldVariousFlags = this.ReadFieldVariousFlags();

      //UseHover
      if((FieldVariousFlags & 0x0080) == 0x0080) {
        SByte FieldVariousHoverX = this.ReadFieldVariousHover("FieldVariousHoverX");
        SByte FieldVariousHoverY = this.ReadFieldVariousHover("FieldVariousHoverY");
      }

      //UseFieldID
      if((FieldVariousFlags & 0x0010) == 0x0010) {
        Int32 FieldVariousIDLength = this.ReadFieldVariousLength("FieldVariousIDLength");
        String FieldVariousID = this.ReadFieldVariousText("FieldVariousID", FieldVariousIDLength);
      }

      //UseTipText
      if((FieldVariousFlags & 0x0001) == 0x0001) {
        Int32 FieldVariousCount = 1;

        //UseTipTextLanguage
        if((FieldVariousFlags & 0x0002) == 0x0002 && LanguageCount > 1)
          FieldVariousCount = LanguageCount;

        for(Int32 i = 0; i < FieldVariousCount; i++) {
          Int32 FieldVariousTipLength = this.ReadFieldVariousLength("FieldVariousTipLength");
          String FieldVariousTip = this.ReadFieldVariousText("FieldVariousTip", FieldVariousTipLength);
        }
      }

      //UseLinkPath
      if((FieldVariousFlags & 0x0004) == 0x0004) {
        Int32 FieldVariousCount = 1;

        //UseLinkPathLanguage
        if((FieldVariousFlags & 0x0008) == 0x0008 && LanguageCount > 1)
          FieldVariousCount = LanguageCount;

        for(Int32 i = 0; i < FieldVariousCount; i++) {
          Int32 FieldVariousLinkLength = this.ReadFieldVariousLength("FieldVariousLinkLength");
          String FieldVariousLink = this.ReadFieldVariousText("FieldVariousLink", FieldVariousLinkLength);
        }
      }
    }


    private Boolean ReadPageFieldEmptyV2() {
      Int32 FieldEmptyWidth = this.ReadPageFieldWidth("FieldEmptyWidth");
      Int32 FieldEmptyHeight = this.ReadPageFieldHeight("FieldEmptyHeight");
      return false;
    }

    private Boolean ReadPageFieldTextV2(Int32 LanguageCount) {
      Int32 FieldTextCount = 1;

      for(Int32 i = 0; i < FieldTextCount; i++) {
        UInt16 FieldTextFlags = this.ReadFieldTextFlags();

        //UseLanguage
        if(i == 0 && (FieldTextFlags & 0x0010) == 0x0010 && LanguageCount > 1)
          FieldTextCount = LanguageCount;

        Int32 FieldTextWidth = this.ReadPageFieldWidth("FieldTextWidth");
        Int32 FieldTextHeight = this.ReadPageFieldHeight("FieldTextHeight");

        Int32 FieldTextColorCount = this.ReadFieldTextColorCount();
        this.ReadFieldTextColorMemory(FieldTextColorCount);

        Int32 FieldTextFontCount = this.ReadFieldTextFontCount();
        this.ReadFieldTextFontMemory(FieldTextFontCount);

        for(Int32 j = 0; j < FieldTextFontCount; j++) {
          Int32 FieldTextLetterCount = this.ReadFieldTextLetterCount();
          this.ReadFieldTextLetterMemory(FieldTextLetterCount);
        }

        Int32 FieldTextLength = this.ReadFieldTextLength();

        Int32 FieldTextInfoTypeSize = this.ReadFieldTextInfoSize("FieldTextInfoTypeSize");
        this.ReadFieldTextInfoMemory("FieldTextInfoTypeMemory", FieldTextInfoTypeSize, FieldTextLength, 4, FileViewerTDHelpWriterType.FieldTextInfoType);

        if(FieldTextFontCount > 1) {
          Int32 FieldTextInfoFontSize = this.ReadFieldTextInfoSize("FieldTextInfoFontSize");
          this.ReadFieldTextInfoMemory("FieldTextInfoFontMemory", FieldTextInfoFontSize, FieldTextLength, FieldTextFontCount, FileViewerTDHelpWriterType.FieldTextInfoFont);
        }

        if(FieldTextColorCount > 1) {
          Int32 FieldTextInfoColorSize = this.ReadFieldTextInfoSize("FieldTextInfoColorSize");
          this.ReadFieldTextInfoMemory("FieldTextInfoColorMemory", FieldTextInfoColorSize, FieldTextLength, FieldTextColorCount, FileViewerTDHelpWriterType.FieldTextInfoColor);
        }

        Int32 FieldTextIndexCount = this.ReadFieldTextIndexCount();

        if(FieldTextIndexCount > 1) {
          Int32 FieldTextIndexSize = this.ReadFieldTextInfoSize("FieldTextIndexSize");
          this.ReadFieldTextIndexMemory(FieldTextIndexSize, FieldTextLength, FieldTextIndexCount);
        }

        //UseSeparator
        if((FieldTextFlags & 0x0040) == 0x0040) {
          for(Int32 j = 0; j < FieldTextFontCount; j++) {
            Int32 FieldTextSeparatorIndex = this.ReadFieldTextSeparatorIndex();
          }
        }
      }
      return FieldTextCount == 1 ? false : true;
    }

    private Boolean ReadPageFieldImageV2(Int32 LanguageCount) {
      Int32 FieldImageCount = 1;

      for(Int32 i = 0; i < FieldImageCount; i++) {
        UInt16 FieldImageFlags = this.ReadFieldImageFlags();

        //UseLanguage
        if(i == 0 && (FieldImageFlags & 0x0001) == 0x0001 && LanguageCount > 1)
          FieldImageCount = LanguageCount;

        Int32 FieldImageWidth = this.ReadPageFieldWidth("FieldImageWidth");
        Int32 FieldImageHeight = this.ReadPageFieldHeight("FieldImageHeight");
        Int32 FieldImageDisplayIndex = this.ReadFieldImageDisplayIndex();

      }
      return FieldImageCount == 1 ? false : true;
    }

    private Boolean ReadPageFieldPlayerV2(Int32 LanguageCount) {
      Boolean UseItemLanguage = false;
      Boolean FieldPlayerUseImage = this.ReadFieldPlayerUseImage();

      if(FieldPlayerUseImage)
        UseItemLanguage = this.ReadPageFieldImageV2(LanguageCount);

      Int32 FieldPlayerCount = 1;

      for(Int32 i = 0; i < FieldPlayerCount; i++) {
        UInt16 FieldPlayerFlags = this.ReadFieldPlayerFlagsV2();

        //UseLanguage
        if(i == 0 && (FieldPlayerFlags & 0x0001) == 0x0001 && LanguageCount > 1)
          FieldPlayerCount = LanguageCount;

        Int32 FieldPlayerWidth = this.ReadPageFieldWidth("FieldPlayerWidth");
        Int32 FieldPlayerHeight = this.ReadPageFieldHeight("FieldPlayerHeight");

        UInt32 FieldPlayerColor = this.ReadFieldColor("FieldPlayerColor");

        //UseDownload, UseStreaming
        if(((FieldPlayerFlags & 0x0010) == 0x0010) || ((FieldPlayerFlags & 0x0100) == 0x0100)) {
          Int32 FieldPlayerUrlLength = this.ReadFieldPlayerUrlLength();
          if(FieldPlayerUrlLength > 0) this.ReadFieldPlayerUrl(FieldPlayerUrlLength);

          Byte FieldPlayerType = this.ReadFieldPlayerType();

          Int32 FieldPlayerDisplayWidth = this.ReadFieldPlayerDisplayWidth();
          Int32 FieldPlayerDisplayHeight = this.ReadFieldPlayerDisplayHeight();

          Int64 FieldPlayerDuration = this.ReadFieldPlayerDuration();
        } else {
          Int32 FieldPlayerIndex = this.ReadFieldPlayerIndex();
        }
      }
      return FieldPlayerCount == 1 ? UseItemLanguage : true;
    }


    private void ReadPageFieldSettingV2(Boolean UseItemLanguage, Int32 LanguageCount) {
      Int32 FieldSettingCount = 1;

      if(UseItemLanguage && LanguageCount > 1) FieldSettingCount = LanguageCount;

      UInt16 FieldFlags = 0;

      for(Int32 i = 0; i < FieldSettingCount; i++) {
        FieldFlags = this.ReadFieldFlags();

        Int32 FieldDistanceLeft = this.ReadFieldDistance("FieldDistanceLeft");
        Int32 FieldDistanceTop = this.ReadFieldDistance("FieldDistanceTop");
        Int32 FieldDistanceRight = this.ReadFieldDistance("FieldDistanceRight");
        Int32 FieldDistanceBottom = this.ReadFieldDistance("FieldDistanceBottom");

        Int32 FieldWidth = this.ReadFieldWidth();

        //UseCurve
        if((FieldFlags & 0x0004) == 0x0004) {
          Int32 FieldCurveX = this.ReadFieldCurve("FieldCurveX");
          Int32 FieldCurveY = this.ReadFieldCurve("FieldCurveY");
        }

        //UseFrame
        if((FieldFlags & 0x0100) == 0x0100) {
          Int32 FieldFrameLeft = this.ReadFieldFrame("FieldFrameLeft");
          Int32 FieldFrameTop = this.ReadFieldFrame("FieldFrameTop");
          Int32 FieldFrameRight = this.ReadFieldFrame("FieldFrameRight");
          Int32 FieldFrameBottom = this.ReadFieldFrame("FieldFrameBottom");
        }

        //UseShadow
        if((FieldFlags & 0x0008) == 0x0008) {
          Int32 FieldShadowX = this.ReadFieldShadow("FieldShadowX");
          Int32 FieldShadowY = this.ReadFieldShadow("FieldShadowY");
          UInt32 FieldShadowColor = this.ReadFieldColor("FieldShadowColor");
        }

        //UseBorder
        if((FieldFlags & 0x0080) == 0x0080) {
          UInt32 FieldBorderColor = this.ReadFieldColor("FieldBorderColor");
        }

        //UseBackground
        if((FieldFlags & 0x0040) == 0x0040) {
          UInt32 FieldBackColor = this.ReadFieldColor("FieldBackColor");
        }

        //UseImage
        if((FieldFlags & 0x0800) == 0x0800) {
          Int32 FieldImageX = this.ReadFieldImage("FieldImageX");
          Int32 FieldImageY = this.ReadFieldImage("FieldImageY");
          Int32 FieldImageIndex = this.ReadFieldImageIndex();

          //UseImageAnimate
          if((FieldFlags & 0x1000) == 0x1000) {
            Int32 FieldImageStartTime = this.ReadFieldImageTime("FieldImageStartTime");
            Int32 FieldImageDisplayTime = this.ReadFieldImageTime("FieldImageDisplayTime");
          }

          //UseImageShadow
          if((FieldFlags & 0x4000) == 0x4000) {
            Int32 FieldImageShadowX = this.ReadFieldShadow("FieldImageShadowX");
            Int32 FieldImageShadowY = this.ReadFieldShadow("FieldImageShadowY");
            UInt32 FieldImageShadowColor = this.ReadFieldImageShadowColor();
          }
        }
      }

      //UseID
      if((FieldFlags & 0x0200) == 0x0200) {
        Int32 FieldIDLength = this.ReadFieldIDLength();

        if(FieldIDLength > 0) {
          String FieldID = this.ReadFieldID(FieldIDLength);
        }
      }

      //UseVarious
      if((FieldFlags & 0x0400) == 0x0400)
        this.ReadPageFieldVariousV2(LanguageCount);
    }

    private void ReadPageFieldVariousV2(Int32 LanguageCount) {
      UInt16 FieldVariousFlags = this.ReadFieldVariousFlags();

      //UseHover
      if((FieldVariousFlags & 0x0080) == 0x0080) {
        SByte FieldVariousHoverX = this.ReadFieldVariousHover("FieldVariousHoverX");
        SByte FieldVariousHoverY = this.ReadFieldVariousHover("FieldVariousHoverY");
      }

      //UseFieldID
      if((FieldVariousFlags & 0x0010) == 0x0010) {
        Int32 FieldVariousIDLength = this.ReadFieldVariousLength("FieldVariousIDLength");
        String FieldVariousID = this.ReadFieldVariousText("FieldVariousID", FieldVariousIDLength);
      }

      //UseTipText
      if((FieldVariousFlags & 0x0001) == 0x0001) {
        Int32 FieldVariousCount = 1;

        //UseTipTextLanguage
        if((FieldVariousFlags & 0x0002) == 0x0002 && LanguageCount > 1)
          FieldVariousCount = LanguageCount;

        for(Int32 i = 0; i < FieldVariousCount; i++) {
          Int32 FieldVariousTipLength = this.ReadFieldVariousLength("FieldVariousTipLength");
          String FieldVariousTip = this.ReadFieldVariousText("FieldVariousTip", FieldVariousTipLength);
        }
      }

      //UseLinkPath
      if((FieldVariousFlags & 0x0004) == 0x0004) {
        Int32 FieldVariousCount = 1;

        //UseLinkPathLanguage
        if((FieldVariousFlags & 0x0008) == 0x0008 && LanguageCount > 1)
          FieldVariousCount = LanguageCount;

        for(Int32 i = 0; i < FieldVariousCount; i++) {
          Int32 FieldVariousLinkLength = this.ReadFieldVariousLength("FieldVariousLinkLength");
          String FieldVariousLink = this.ReadFieldVariousText("FieldVariousLink", FieldVariousLinkLength);
        }
      }
    }


    private void ReadDataV1() {
      UInt16 Flags = this.ReadFlags();

      //UsePreview
      if((Flags & 0x0001) == 0x0001) {
        Byte PreviewImageMode = this.ReadPreviewImageMode();

        Int32 PreviewImageWidth = this.ReadPreviewImageWidth();
        Int32 PreviewImageHeight = this.ReadPreviewImageHeight();

        Int32 PreviewImageSize = this.ReadPreviewImageSize();

        this.ReadPreviewImage(PreviewImageSize);
      }

      //UseIcon
      if((Flags & 0x0002) == 0x0002) {
        Byte IconImageMode = this.ReadIconImageMode();

        Int32 IconImageWidth = this.ReadIconImageWidth();
        Int32 IconImageHeight = this.ReadIconImageHeight();

        Int32 IconImageSize = this.ReadIconImageSize();

        this.ReadIconImage(IconImageSize);
      }

      //UseInfo
      if((Flags & 0x0004) == 0x0004) {
        Int32 InfoProducerLength = this.ReadInfoProducerLength();
        String InfoProducer = this.ReadInfoProducer(InfoProducerLength);

        Int32 InfoInternetLength = this.ReadInfoInternetLength();
        String InfoInternet = this.ReadInfoInternet(InfoInternetLength);

        Int32 InfoMailLength = this.ReadInfoMailLength();
        String InfoMail = this.ReadInfoMail(InfoMailLength);
      }

      //MainSetting
      UInt32 SettingBackColor = this.ReadSettingBackColor();
      UInt32 SettingWindowColor = this.ReadSettingWindowColor();

      Int32 SettingOutlineLeft = this.ReadSettingBorderSize("SettingOutlineLeft");
      Int32 SettingOutlineTop = this.ReadSettingBorderSize("SettingOutlineTop");
      Int32 SettingOutlineRight = this.ReadSettingBorderSize("SettingOutlineRight");
      Int32 SettingOutlineBottom = this.ReadSettingBorderSize("SettingOutlineBottom");

      Int32 SettingPageLeft = this.ReadSettingBorderSize("SettingPageLeft");
      Int32 SettingPageTop = this.ReadSettingBorderSize("SettingPageTop");
      Int32 SettingPageRight = this.ReadSettingBorderSize("SettingPageRight");
      Int32 SettingPageBottom = this.ReadSettingBorderSize("SettingPageBottom");

      //Language
      Int32 LanguageCount = this.ReadLanguageCount();

      for(Int32 i = 0; i < LanguageCount; i++) {
        Byte LanguageLetterLength = this.ReadLanguageLetterLength();
        String LanguageLetter = this.ReadLanguageLetter(LanguageLetterLength);
      }

      //LetterImage
      Int32 LetterImageSize = this.ReadLetterImageSize();
      Int32 FontCount = 0;

      if(LetterImageSize > 0) {
        this.ReadLetterImage(LetterImageSize);

        Boolean LetterUseInfo = this.ReadLetterUseInfo();

        //LetterFont
        FontCount = this.ReadLetterFontCount();

        for(Int32 i = 0; i < FontCount; i++) {
          Int16 FontHeight = this.ReadFontHeight();
          Int16 FontAscent = this.ReadFontAscent();

          //Letter
          Int32 LetterCount = this.ReadLetterCount();

          for(Int32 j = 0; j < LetterCount; j++) {
            Int16 LetterWidthA = this.ReadLetterWidth("LetterWidthA");
            Int16 LetterWidthB = this.ReadLetterWidth("LetterWidthB");
            Int16 LetterWidthC = this.ReadLetterWidth("LetterWidthC");

            Int32 LetterImagePosition = this.ReadLetterImagePosition();

            if(LetterImagePosition >= 0) {
              Int16 LetterImageWidth = this.ReadLetterImageWidth();
              Int16 LetterImageHeight = this.ReadLetterImageHeight();
            }
          }
        }
      }

      //Image for page field
      Int32 FieldImageCount = this.ReadImageCount("FieldImageCount");

      for(Int32 i = 0; i < FieldImageCount; i++) {
        Byte FieldImageMode = this.ReadImageMode("FieldImageMode");

        Int32 FieldImageWidth = this.ReadImageWidth("FieldImageWidth");
        Int32 FieldImageHeight = this.ReadImageHeight("FieldImageHeight");

        Int32 FieldImageSize = this.ReadImageSize("FieldImageSize");

        this.ReadImage("FieldImage", FieldImageSize);
      }

      //Display image for page field content
      Int32 DisplayImageCount = this.ReadImageCount("DisplayImageCount");

      for(Int32 i = 0; i < DisplayImageCount; i++) {
        Byte DisplayImageMode = this.ReadImageMode("DisplayImageMode");

        Int32 DisplayImageWidth = this.ReadImageWidth("DisplayImageWidth");
        Int32 DisplayImageHeight = this.ReadImageHeight("DisplayImageHeight");

        Int32 DisplayImageSize = this.ReadImageSize("DisplayImageSize");

        this.ReadImage("DisplayImage", DisplayImageSize);
      }

      //Player file for page field content
      Int32 PlayerFileCount = this.ReadPlayerFileCount();

      for(Int32 i = 0; i < PlayerFileCount; i++) {
        Byte PlayerFileType = this.ReadPlayerFileType();
        Int64 PlayerDuration = this.ReadPlayerFileDuration();

        Int32 PlayerDisplayWidth = this.ReadPlayerDisplayWidth();
        Int32 PlayerDisplayHeight = this.ReadPlayerDisplayHeight();

        Int32 PlayerFileSize = this.ReadPlayerFileSize();

        this.ReadPlayerFile(PlayerFileType, PlayerFileSize);
      }

      //Expanded image for outline with children
      Int32 ExpandedImageCount = this.ReadImageCount("ExpandedImageCount");

      for(Int32 i = 0; i < ExpandedImageCount; i++) {
        Byte ExpandedImageMode = this.ReadImageMode("ExpandedImageMode");

        Int32 ExpandedImageWidth = this.ReadImageWidth("ExpandedImageWidth");
        Int32 ExpandedImageHeight = this.ReadImageHeight("ExpandedImageHeight");

        Int32 ExpandedImageSize = this.ReadImageSize("ExpandedImageSize");

        this.ReadImage("ExpandedImage", ExpandedImageSize);
      }

      //Outline item
      Int32 OutlineCount = this.ReadOutlineCount();

      for(Int32 i = 0; i < OutlineCount; i++) {
        UInt16 OutlineFlags = this.ReadOutlineFlags();

        Int32 OutlineDistanceLeft = this.ReadOutlineDistance("OutlineDistanceLeft");
        Int32 OutlineDistanceTop = this.ReadOutlineDistance("OutlineDistanceTop");
        Int32 OutlineDistanceRight = this.ReadOutlineDistance("OutlineDistanceRight");
        Int32 OutlineDistanceBottom = this.ReadOutlineDistance("OutlineDistanceBottom");

        Int32 OutlineFrameLeft = this.ReadOutlineFrame("OutlineFrameLeft");
        Int32 OutlineFrameTop = this.ReadOutlineFrame("OutlineFrameTop");
        Int32 OutlineFrameRight = this.ReadOutlineFrame("OutlineFrameRight");
        Int32 OutlineFrameBottom = this.ReadOutlineFrame("OutlineFrameBottom");

        Int32 OutlineExpandedX = this.ReadOutlineExpandedLocation("OutlineExpandedX");
        Int32 OutlineExpandedY = this.ReadOutlineExpandedLocation("OutlineExpandedY");
        Int32 OutlineExpandedCount = this.ReadOutlineExpandedCount();
        Int32 OutlineExpandedIndex = this.ReadOutlineExpandedIndex(ExpandedImageCount);

        //UseRound
        if((OutlineFlags & 0x0001) == 0x0001) {
          Int32 OutlineCurveX = this.ReadOutlineCurveLocation("OutlineCurveX");
          Int32 OutlineCurveY = this.ReadOutlineCurveLocation("OutlineCurveY");
        }

        //UseBackground
        if((OutlineFlags & 0x0008) == 0x0008) {
          UInt32 OutlineColorMiddle = this.ReadOutlineBackColor("OutlineColorMiddle");

          //UseBackgroundStyle
          if((OutlineFlags & 0x0010) == 0x0010) {
            UInt32 OutlineColorBegin = this.ReadOutlineBackColor("OutlineColorBegin");
            UInt32 OutlineColorEnd = this.ReadOutlineBackColor("OutlineColorEnd");
          }
        }

        //UseShadow
        if((OutlineFlags & 0x0002) == 0x0002) {
          Int32 OutlineShadowX = this.ReadOutlineShadowLocation("OutlineShadowX");
          Int32 OutlineShadowY = this.ReadOutlineShadowLocation("OutlineShadowY");
          UInt32 OutlineShadowColor = this.ReadOutlineBackColor("OutlineShadowColor");
        }

        //UseBorder
        if((OutlineFlags & 0x0004) == 0x0004) {
          UInt32 OutlineBorderColor = this.ReadOutlineBackColor("OutlineBorderColor");
        }

        Int32 OutlineTextCount = 1;

        //UseLanguage
        if((OutlineFlags & 0x0020) == 0x0020 && LanguageCount > 1)
          OutlineTextCount = LanguageCount;

        for(Int32 j = 0; j < OutlineTextCount; j++) {
          UInt32 OutlineTextColor = this.ReadOutlineBackColor("OutlineTextColor");

          Int32 OutlineTextWidth = this.ReadOutlineTextWidth();
          Int32 OutlineTextHeight = this.ReadOutlineTextHeight();
          Int32 OutlineTextLength = this.ReadOutlineTextLength();

          Int32 OutlineFontIndex = this.ReadOutlineFontIndex(FontCount);

          Int32 OutlineLetterIndexCount = this.ReadOutlineLetterIndexCount();
          if(OutlineLetterIndexCount > 0) this.ReadOutlineLetterIndexMemory(OutlineLetterIndexCount);

          Int32 OutlineInfoTypeSize = this.ReadOutlineInfoTypeSize();

          //Not used
          //if(OutlineInfoTypeSize > 0) this.ReadOutlineInfoTypeMemoryV1(OutlineInfoTypeSize);

          Int32 OutlineInfoIndexCount = this.ReadOutlineInfoIndexCount();

          if(OutlineInfoIndexCount > 1) {
            Int32 OutlineInfoIndexSize = this.ReadOutlineInfoIndexSize();
            if(OutlineInfoIndexSize > 0) this.ReadOutlineInfoIndexMemory(OutlineInfoIndexSize, OutlineTextLength, OutlineInfoIndexCount);
          }
        }

        //UsePage
        if((OutlineFlags & 0x0040) == 0x0040) {
          //PageLine
          Int32 PageLineCount = this.ReadPageLineCount();

          for(Int32 j = 0; j < PageLineCount; j++) {
            Int32 PageLineHeightCount = 1;

            if(LanguageCount > 1)
              PageLineHeightCount = LanguageCount;

            for(Int32 k = 0; k < PageLineHeightCount; k++) {
              Int32 PageLineHeight = this.ReadPageLineHeight();
            }

            //PageField
            Int32 PageFieldCount = this.ReadPageFieldCount();

            for(Int32 k = 0; k < PageFieldCount; k++) {
              Boolean UseItemLanguage = false;

              Byte PageFieldType = this.ReadPageFieldType();

              switch(PageFieldType) {
                case 0: { //Empty
                  UseItemLanguage = this.ReadPageFieldEmptyV1();
                  break;
                }
                case 1: { //Text
                  UseItemLanguage = this.ReadPageFieldTextV1(LanguageCount);
                  break;
                }
                case 2: { //Image
                  UseItemLanguage = this.ReadPageFieldImageV1(LanguageCount);
                  break;
                }
                case 3: { //Player
                  UseItemLanguage = this.ReadPageFieldPlayerV1(LanguageCount);
                  break;
                }
              }

              this.ReadPageFieldSettingV1(UseItemLanguage, LanguageCount);
            }
          }
        }

        //Children
        Int32 OutlineChildCount = this.ReadOutlineChildCount();

        OutlineCount += OutlineChildCount;
      }

      if(this.Position != this.Size) throw new Exception("Wrong FileSize");
    }

    private void ReadDataV2() {
      UInt16 Flags = this.ReadFlags();

      //UsePreview
      if((Flags & 0x0001) == 0x0001) {
        Byte PreviewImageMode = this.ReadPreviewImageMode();

        Int32 PreviewImageWidth = this.ReadPreviewImageWidth();
        Int32 PreviewImageHeight = this.ReadPreviewImageHeight();

        Int32 PreviewImageSize = this.ReadPreviewImageSize();

        this.ReadPreviewImage(PreviewImageSize);
      }

      //UseIcon
      if((Flags & 0x0002) == 0x0002) {
        Byte IconImageMode = this.ReadIconImageMode();
        Int32 IconImageSize = this.ReadIconImageSize();

        this.ReadIconImage(IconImageSize);
      }

      //UseInfo
      if((Flags & 0x0004) == 0x0004) {
        Int32 InfoProducerLength = this.ReadInfoProducerLength();
        String InfoProducer = this.ReadInfoProducer(InfoProducerLength);

        Int32 InfoInternetLength = this.ReadInfoInternetLength();
        String InfoInternet = this.ReadInfoInternet(InfoInternetLength);

        Int32 InfoMailLength = this.ReadInfoMailLength();
        String InfoMail = this.ReadInfoMail(InfoMailLength);
      }

      //MainSetting
      UInt32 SettingBackColor = this.ReadSettingBackColor();
      UInt32 SettingWindowColor = this.ReadSettingWindowColor();

      Int32 SettingOutlineLeft = this.ReadSettingBorderSize("SettingOutlineLeft");
      Int32 SettingOutlineTop = this.ReadSettingBorderSize("SettingOutlineTop");
      Int32 SettingOutlineRight = this.ReadSettingBorderSize("SettingOutlineRight");
      Int32 SettingOutlineBottom = this.ReadSettingBorderSize("SettingOutlineBottom");

      Int32 SettingPageLeft = this.ReadSettingBorderSize("SettingPageLeft");
      Int32 SettingPageTop = this.ReadSettingBorderSize("SettingPageTop");
      Int32 SettingPageRight = this.ReadSettingBorderSize("SettingPageRight");
      Int32 SettingPageBottom = this.ReadSettingBorderSize("SettingPageBottom");

      Double SettingDocumentFactor = this.ReadSettingDocumentFactor();

      //Language
      Int32 LanguageCount = this.ReadLanguageCount();

      for(Int32 i = 0; i < LanguageCount; i++) {
        Byte LanguageLetterLength = this.ReadLanguageLetterLength();
        String LanguageLetter = this.ReadLanguageLetter(LanguageLetterLength);
      }

      //LetterImage
      Int32 LetterImageSize = this.ReadLetterImageSize();
      Int32 FontCount = 0;

      if(LetterImageSize > 0) {
        this.ReadLetterImage(LetterImageSize);

        //LetterFont
        FontCount = this.ReadLetterFontCount();

        for(Int32 i = 0; i < FontCount; i++) {
          FontStyle FontStyle = (FontStyle) this.ReadFontStyle();
          Double FontSize = this.ReadFontSize();

          Int32 FontEmSize = this.ReadFontEm("FontEmSize", 0, 3000);
          Int32 FontEmAscent = this.ReadFontEm("FontEmAscent", 0, 3000);
          Int32 FontEmDescent = this.ReadFontEm("FontEmDescent", 0, 3000);

          if((FontStyle & System.Drawing.FontStyle.Underline) == System.Drawing.FontStyle.Underline) {
            Int32 FontEmUnderlineSize = this.ReadFontEm("FontEmUnderlineSize", -3000, 3000);
            Int32 FontEmUnderlineY = this.ReadFontEm("FontEmUnderlineY", -3000, 3000);
          }

          if((FontStyle & System.Drawing.FontStyle.Strikeout) == System.Drawing.FontStyle.Strikeout) {
            Int32 FontEmStrikeoutSize = this.ReadFontEm("FontEmStrikeoutSize", -3000, 3000);
            Int32 FontEmStrikeoutY = this.ReadFontEm("FontEmStrikeoutY", -3000, 3000);
          }

          //Letter
          Int32 LetterCount = this.ReadLetterCount();

          for(Int32 j = 0; j < LetterCount; j++) {
            Double LetterWidth = this.ReadLetterWidth();

            Int32 LetterImagePosition = this.ReadLetterImagePosition("LetterImageAPosition");

            if(LetterImagePosition >= 0) {
              Int32 LetterImageWidth = this.ReadLetterImageWidth("LetterImageAWidth");
              Int32 LetterImageHeight = this.ReadLetterImageHeight("LetterImageAHeight");
              Double LetterImageX = this.ReadLetterImageX("LetterImageAX");
              Int32 LetterImageY = this.ReadLetterImageY("LetterImageAY");

              LetterImagePosition = this.ReadLetterImagePosition("LetterImageBPosition");
              LetterImageWidth = this.ReadLetterImageWidth("LetterImageBWidth");
              LetterImageHeight = this.ReadLetterImageHeight("LetterImageBHeight");
              LetterImageX = this.ReadLetterImageX("LetterImageBX");
              LetterImageY = this.ReadLetterImageY("LetterImageBY");

              LetterImagePosition = this.ReadLetterImagePosition("LetterImageCPosition");
              LetterImageWidth = this.ReadLetterImageWidth("LetterImageCWidth");
              LetterImageHeight = this.ReadLetterImageHeight("LetterImageCHeight");
              LetterImageX = this.ReadLetterImageX("LetterImageCX");
              LetterImageY = this.ReadLetterImageY("LetterImageCY");

              LetterImagePosition = this.ReadLetterImagePosition("LetterImageDPosition");
              LetterImageWidth = this.ReadLetterImageWidth("LetterImageDWidth");
              LetterImageHeight = this.ReadLetterImageHeight("LetterImageDHeight");
              LetterImageX = this.ReadLetterImageX("LetterImageDX");
              LetterImageY = this.ReadLetterImageY("LetterImageDY");
            }
          }
        }
      }

      //Image for page field
      Int32 FieldImageCount = this.ReadImageCount("FieldImageCount");

      for(Int32 i = 0; i < FieldImageCount; i++) {
        Byte FieldImageMode = this.ReadImageMode("FieldImageMode");

        Int32 FieldImageWidth = this.ReadImageWidth("FieldImageWidth");
        Int32 FieldImageHeight = this.ReadImageHeight("FieldImageHeight");

        Int32 FieldImageSize = this.ReadImageSize("FieldImageSize");

        this.ReadImage("FieldImage", FieldImageSize);

        Int32 FieldImageItemWidth = this.ReadImageItemWidth("FieldImageItemWidth");
        Int32 FieldImageItemCount = this.ReadImageItemCount("FieldImageItemCount");
      }

      //Display image for page field content
      Int32 DisplayImageCount = this.ReadImageCount("DisplayImageCount");

      for(Int32 i = 0; i < DisplayImageCount; i++) {
        Byte DisplayImageMode = this.ReadImageMode("DisplayImageMode");

        Int32 DisplayImageWidth = this.ReadImageWidth("DisplayImageWidth");
        Int32 DisplayImageHeight = this.ReadImageHeight("DisplayImageHeight");

        Int32 DisplayImageSize = this.ReadImageSize("DisplayImageSize");

        this.ReadImage("DisplayImage", DisplayImageSize);
      }

      //Download image for page field content
      Int32 DownloadImageCount = this.ReadImageCount("DownloadImageCount");

      for(Int32 i = 0; i < DownloadImageCount; i++) {
        Byte DownloadImageMode = this.ReadImageMode("DownloadImageMode");

        Int32 DownloadImageWidth = this.ReadImageWidth("DownloadImageWidth");
        Int32 DownloadImageHeight = this.ReadImageHeight("DownloadImageHeight");

        Int32 DownloadImagePathLength = this.ReadImagePathLength("DownloadImagePathLength");
        String DownloadImagePath = this.ReadImagePath("DownloadImagePath", DownloadImagePathLength);
      }

      //Player file for page field content
      Int32 PlayerFileCount = this.ReadPlayerFileCount();

      for(Int32 i = 0; i < PlayerFileCount; i++) {
        Byte PlayerFileType = this.ReadPlayerFileType();
        Int64 PlayerDuration = this.ReadPlayerFileDuration();

        Int32 PlayerDisplayWidth = this.ReadPlayerDisplayWidth();
        Int32 PlayerDisplayHeight = this.ReadPlayerDisplayHeight();

        Int32 PlayerFileSize = this.ReadPlayerFileSize();

        this.ReadPlayerFile(PlayerFileType, PlayerFileSize);
      }

      //Expanded image for outline with children
      Int32 ExpandedImageCount = this.ReadImageCount("ExpandedImageCount");

      for(Int32 i = 0; i < ExpandedImageCount; i++) {
        Byte ExpandedImageMode = this.ReadImageMode("ExpandedImageMode");

        Int32 ExpandedImageWidth = this.ReadImageWidth("ExpandedImageWidth");
        Int32 ExpandedImageHeight = this.ReadImageHeight("ExpandedImageHeight");

        Int32 ExpandedImageSize = this.ReadImageSize("ExpandedImageSize");

        this.ReadImage("ExpandedImage", ExpandedImageSize);

        Int32 ExpandedImageItemWidth = this.ReadImageItemWidth("ExpandedImageItemWidth");
        Int32 ExpandedImageItemCount = this.ReadImageItemCount("ExpandedImageItemCount");
      }

      //Outline item
      Int32 OutlineCount = this.ReadOutlineCount();

      for(Int32 i = 0; i < OutlineCount; i++) {
        UInt16 OutlineFlags = this.ReadOutlineFlags();

        Int32 OutlineDistanceLeft = this.ReadOutlineDistance("OutlineDistanceLeft");
        Int32 OutlineDistanceTop = this.ReadOutlineDistance("OutlineDistanceTop");
        Int32 OutlineDistanceRight = this.ReadOutlineDistance("OutlineDistanceRight");
        Int32 OutlineDistanceBottom = this.ReadOutlineDistance("OutlineDistanceBottom");

        Int32 OutlineFrameLeft = this.ReadOutlineFrame("OutlineFrameLeft");
        Int32 OutlineFrameTop = this.ReadOutlineFrame("OutlineFrameTop");
        Int32 OutlineFrameRight = this.ReadOutlineFrame("OutlineFrameRight");
        Int32 OutlineFrameBottom = this.ReadOutlineFrame("OutlineFrameBottom");

        Int32 OutlineExpandedX = this.ReadOutlineExpandedLocation("OutlineExpandedX");
        Int32 OutlineExpandedY = this.ReadOutlineExpandedLocation("OutlineExpandedY");

        Int32 OutlineExpandedIndex = this.ReadOutlineExpandedIndex(ExpandedImageCount);

        //UseRound
        if((OutlineFlags & 0x0001) == 0x0001) {
          Int32 OutlineCurveX = this.ReadOutlineCurveLocation("OutlineCurveX");
          Int32 OutlineCurveY = this.ReadOutlineCurveLocation("OutlineCurveY");
        }

        //UseBackground
        if((OutlineFlags & 0x0008) == 0x0008) {
          UInt32 OutlineColorMiddle = this.ReadOutlineBackColor("OutlineColorMiddle");

          //UseBackgroundStyle
          if((OutlineFlags & 0x0010) == 0x0010) {
            UInt32 OutlineColorBegin = this.ReadOutlineBackColor("OutlineColorBegin");
            UInt32 OutlineColorEnd = this.ReadOutlineBackColor("OutlineColorEnd");
          }
        }

        //UseShadow
        if((OutlineFlags & 0x0002) == 0x0002) {
          Int32 OutlineShadowX = this.ReadOutlineShadowLocation("OutlineShadowX");
          Int32 OutlineShadowY = this.ReadOutlineShadowLocation("OutlineShadowY");
          UInt32 OutlineShadowColor = this.ReadOutlineBackColor("OutlineShadowColor");
        }

        //UseBorder
        if((OutlineFlags & 0x0004) == 0x0004) {
          UInt32 OutlineBorderColor = this.ReadOutlineBackColor("OutlineBorderColor");
        }

        Int32 OutlineTextCount = 1;

        //UseLanguage
        if((OutlineFlags & 0x0020) == 0x0020 && LanguageCount > 1)
          OutlineTextCount = LanguageCount;

        for(Int32 j = 0; j < OutlineTextCount; j++) {
          UInt32 OutlineTextColor = this.ReadOutlineBackColor("OutlineTextColor");
          Int32 OutlineTextLength = this.ReadOutlineTextLength();

          Int32 OutlineFontIndex = this.ReadOutlineFontIndex(FontCount);

          Int32 OutlineLetterIndexCount = this.ReadOutlineLetterIndexCount();
          this.ReadOutlineLetterIndexMemory(OutlineLetterIndexCount);

          Int32 OutlineInfoTypeSize = this.ReadOutlineInfoTypeSize();
          this.ReadOutlineInfoTypeMemoryV2(OutlineInfoTypeSize, OutlineTextLength);

          if(OutlineTextLength > 1) {
            Int32 OutlineInfoIndexSize = this.ReadOutlineInfoIndexSize();
            if(OutlineInfoIndexSize > 0) this.ReadOutlineInfoIndexMemory(OutlineInfoIndexSize, OutlineTextLength, OutlineTextLength);
          }
        }

        //UsePage
        if((OutlineFlags & 0x0040) == 0x0040) {
          //PageLine
          Int32 PageLineCount = this.ReadPageLineCount();

          for(Int32 j = 0; j < PageLineCount; j++) {
            Int32 PageLineHeightCount = 1;

            if(LanguageCount > 1)
              PageLineHeightCount = LanguageCount;

            for(Int32 k = 0; k < PageLineHeightCount; k++) {
              Int32 PageLineHeight = this.ReadPageLineHeight();
            }

            //PageField
            Int32 PageFieldCount = this.ReadPageFieldCount();

            for(Int32 k = 0; k < PageFieldCount; k++) {
              Boolean UseItemLanguage = false;

              Byte PageFieldType = this.ReadPageFieldType();

              switch(PageFieldType) {
                case 0: { //Empty
                  UseItemLanguage = this.ReadPageFieldEmptyV2();
                  break;
                }
                case 1: { //Text
                  UseItemLanguage = this.ReadPageFieldTextV2(LanguageCount);
                  break;
                }
                case 2: { //Image
                  UseItemLanguage = this.ReadPageFieldImageV2(LanguageCount);
                  break;
                }
                case 3: { //Player
                  UseItemLanguage = this.ReadPageFieldPlayerV2(LanguageCount);
                  break;
                }
              }

              this.ReadPageFieldSettingV2(UseItemLanguage, LanguageCount);
            }
          }
        }

        //Children
        Int32 OutlineChildCount = this.ReadOutlineChildCount();

        OutlineCount += OutlineChildCount;
      }

      if(this.Position != this.Size) throw new Exception("Wrong FileSize");
    }


    private void ReadBegin() {
      UInt32 IDNumber = this.ReadIDNumber();

      Int32 FileCount = this.ReadFileCount();
      Int64 FileSize = this.ReadFileSize();
      Int64 FileMaxSize = this.ReadFileMaxSize();

      Byte Version = this.ReadVersion();

      switch(Version) {
        case 1: {
          this.ReadDataV1();
          return;
        }
        case 2: {
          this.ReadDataV2();
          return;
        }
      }
    }

    #endregion


    #region Public Methods

    public List<ListViewItem> Run(IFileViewerProgressBar Parent, String FilePath) {
      this.Parent = Parent;

      try {
        this.Stream = new FileStream(FilePath, FileMode.Open, FileAccess.Read);
        this.Reader = new BinaryReader(this.Stream);

        this.Size = this.Stream.Length;

        this.Parent.BeginProgressBar(this.Size);

        this.AddItemBeginRead();

        this.ReadBegin();
      } catch(Exception e) {
        this.AddItemError(e);

        this.Close();
        return this.ItemList;
      }

      this.AddItemEndOfFile();

      this.Close();
      return this.ItemList;
    }

    public Byte[] LetterMemory(Bitmap Image, Int32 Position, Int32 Width, Int32 Height) {
      //System.Drawing.Bitmap, System.Drawing.Imaging.BitmapData
      //System.Runtime.InteropServices.Marshal

      Int32 Size = Width * Height * 4;
      Byte[] Memory = new Byte[Size];

      BitmapData Data = Image.LockBits(new Rectangle(0, 0, Image.Width, Image.Height), ImageLockMode.ReadOnly, PixelFormat.Format32bppArgb);

      Marshal.Copy(new IntPtr(Data.Scan0.ToInt64() + Position), Memory, 0, Size);

      Image.UnlockBits(Data);
      return Memory;
    }

    public Color LetterColor(Color BackImage, Color LetterImage, Color TextColor) {
      //System.Drawing.Color

      if(LetterImage.R == 0 && LetterImage.G == 0 && LetterImage.B == 0)
        return TextColor;

      if(LetterImage.R == 255 && LetterImage.G == 255 && LetterImage.B == 255)
        return BackImage;

      Int32 Red = (LetterImage.R * BackImage.R + (255 - LetterImage.R) * TextColor.R) / 255;
      Int32 Green = (LetterImage.G * BackImage.G + (255 - LetterImage.G) * TextColor.G) / 255;
      Int32 Blue = (LetterImage.B * BackImage.B + (255 - LetterImage.B) * TextColor.B) / 255;
      return Color.FromArgb(Red, Green, Blue);
    }


    public void CreateSineColor(UInt32[] ColorArray, Int32 Index, Int32 Count, Double Factor, Color Begin, Color End) {
      if(ColorArray == null || ColorArray.Length == 0) return;

      if(ColorArray.Length == 1) {
        ColorArray[0] = (UInt32) Begin.ToArgb();
        return;
      }

      double A = End.A - Begin.A;
      double R = End.R - Begin.R;
      double G = End.G - Begin.G;
      double B = End.B - Begin.B;

      double dCount = (Count - 1) * (Count - 1);

      for(Int32 i = 0; i < Count; i++) {
        double Value = Math.Pow(1.0 - i * i / dCount, Factor);

        ColorArray[Index + Count - 1 - i] = (UInt32) Color.FromArgb(Convert.ToInt32(Begin.A + Value * A), Convert.ToInt32(Begin.R + Value * R), Convert.ToInt32(Begin.G + Value * G), Convert.ToInt32(Begin.B + Value * B)).ToArgb();
      }
    }

    public void CreateCosineColor(UInt32[] ColorArray, Int32 Index, Int32 Count, Double Factor, Color Begin, Color End) {
      if(ColorArray == null || ColorArray.Length == 0) return;

      if(ColorArray.Length == 1) {
        ColorArray[0] = (UInt32) Begin.ToArgb();
        return;
      }

      double A = End.A - Begin.A;
      double R = End.R - Begin.R;
      double G = End.G - Begin.G;
      double B = End.B - Begin.B;

      double dCount = (Count - 1) * (Count - 1);

      for(Int32 i = 0; i < Count; i++) {
        double Value = Math.Pow(1.0 - i * i / dCount, Factor);

        ColorArray[Index + i] = (UInt32) Color.FromArgb(Convert.ToInt32(End.A - Value * A), Convert.ToInt32(End.R - Value * R), Convert.ToInt32(End.G - Value * G), Convert.ToInt32(End.B - Value * B)).ToArgb();
      }
    }

    public UInt32[] CreateRoundColor(Int32 Count, Double Factor, Color Begin, Color Middle, Color End) {
      if(Count < 1) return new UInt32[0];

      UInt32[] ColorArray = new UInt32[Count];

      if(Count < 4) {
        if(Count == 1) {
          ColorArray[0] = (UInt32) Middle.ToArgb();
          return ColorArray;
        }
        if(Count == 2) {
          ColorArray[0] = (UInt32) Middle.ToArgb();
          ColorArray[1] = (UInt32) End.ToArgb();
          return ColorArray;
        }


        ColorArray[0] = (UInt32) Begin.ToArgb();
        ColorArray[1] = (UInt32) Middle.ToArgb();
        ColorArray[2] = (UInt32) End.ToArgb();
        return ColorArray;
      }

      Int32 SizeA = Count / 2;
      Int32 SizeB = Count - SizeA;

      this.CreateSineColor(ColorArray, 0, SizeA, Factor, Begin, Middle);
      this.CreateCosineColor(ColorArray, SizeA, SizeB, Factor, Middle, End);
      return ColorArray;
    }

    #endregion
  }


  public class FileViewerVideoThumbnailImageData {

    #region Public Constances

    public const Int32 HeaderSize = 8 + 4 + 8 + 4 + 4 + 1 + 4 + 8;

    #endregion

    #region Public Fields

    public IFileViewerProgressBar Parent;
    public List<ListViewItem> ItemList;
    public BinaryReader Reader;
    public FileStream Stream;
    public String FilePath;
    public Int64 Position;
    public Int64 Size;

    #endregion


    #region Contructor

    public FileViewerVideoThumbnailImageData() {
      this.ItemList = new List<ListViewItem>();
    }

    #endregion


    #region Private Methods

    private String GetBytesString(Int64 Size) {
      if(Size == 0) return "0 Byte";
      return Size.ToString("#,#") + " Bytes";
    }


    private void AddItem(Int64 Size, String Type, String Name, String Value) {
      this.ItemList.Add(new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value }));

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }

    private void AddItem(Int64 Size, String Type, String Name, String Value, FileViewerObject Tag) {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString(), Size.ToString(), Type, Name, Value });

      Item.BackColor = Color.FromArgb(240, 255, 245);
      Item.Tag = Tag;

      this.ItemList.Add(Item);

      this.Position = this.Stream.Position;

      if(this.Parent.UpdateProgressBar(this.Position)) return;

      throw new Exception("Reading was canceled.");
    }


    private void AddItemError(String Value) {
      ListViewItem Item = new ListViewItem(new String[] { "".PadRight(10), "".PadRight(10), "".PadRight(10), "Error".PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }

    private void AddItemError(Exception e) {
      String Name = String.Empty;
      String Value = String.Empty;

      if(e.InnerException != null && e.InnerException.Message != null) {
        if(e.Message != null)
          Name = e.Message;

        Value = e.InnerException.Message;
      } else {
        if(e.Message != null)
          Value = e.Message;
      }

      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "ERROR".PadRight(10), Name.PadRight(10), Value.PadRight(10) });

      Item.BackColor = Color.FromArgb(255, 240, 240);

      this.ItemList.Add(Item);
    }


    private void AddItemBeginRead() {
      ListViewItem Item = new ListViewItem(new String[] { (this.Size - 8).ToString().PadRight(10), "0".PadRight(10), "".PadRight(10), "BeginRead".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }

    private void AddItemEndOfFile() {
      ListViewItem Item = new ListViewItem(new String[] { this.Position.ToString().PadRight(10), "0".PadRight(10), "".PadRight(10), "EndOfFile".PadRight(10), "".PadRight(10) });

      Item.BackColor = Color.FromArgb(240, 245, 255);

      this.ItemList.Add(Item);
    }


    private void Close() {
      if(this.Stream != null) {
        this.Stream.Close();
        this.Stream = null;
      }

      if(this.Reader != null) {
        this.Reader.Close();
        this.Reader = null;
      }
    }

    #endregion

    #region Private Methods: Format

    private Int64 ReadPreviewPos() {
      this.Stream.Position = this.Size - 8;

      this.Position = this.Stream.Position;

      Int64 PreviewPos = this.Reader.ReadInt64();

      if(PreviewPos < 0 || PreviewPos > this.Size - FileViewerVideoThumbnailImageData.HeaderSize)
        throw new Exception("PreviewPos", new Exception(PreviewPos.ToString()));

      this.AddItem(8, "INT64", "PreviewPos", PreviewPos.ToString());
      return PreviewPos;
    }

    private Int64 ReadPreviewSize(Int64 Position) {
      this.Stream.Position = Position;

      this.Position = Position;

      Int64 PreviewSize = this.Reader.ReadInt64();

      if(Position + PreviewSize != this.Size)
        throw new Exception("PreviewSize", new Exception(PreviewSize.ToString()));

      this.AddItem(8, "INT64", "PreviewSize", this.GetBytesString(PreviewSize));
      return PreviewSize;
    }


    private UInt32 ReadIDNumber() {
      UInt32 IDNumber = this.Reader.ReadUInt32();

      if(IDNumber != 0x44495456U) //VideoThumbnailImageData = VTID
        throw new Exception("IDNumber", new Exception("0x" + IDNumber.ToString("X8")));

      this.AddItem(4, "UINT32", "IDNumber", "0x" + IDNumber.ToString("X8") + " hex");
      return IDNumber;
    }


    private Int64 ReadInfoDuration() {
      Int64 InfoDuration = this.Reader.ReadInt64();

      if(InfoDuration < 0)
        throw new Exception("InfoDuration", new Exception(InfoDuration.ToString()));

      TimeSpan Span = new TimeSpan(InfoDuration);

      this.AddItem(8, "INT64", "InfoDuration", Span.ToString(@"hh\:mm\:ss") + "." + Span.Milliseconds);
      return InfoDuration;
    }

    private Int32 ReadInfoWidth() {
      Int32 InfoWidth = this.Reader.ReadInt32();

      if(InfoWidth < 1 || InfoWidth > 10000)
        throw new Exception("InfoWidth", new Exception(InfoWidth.ToString()));

      this.AddItem(4, "INT32", "InfoWidth", InfoWidth.ToString());
      return InfoWidth;
    }

    private Int32 ReadInfoHeight() {
      Int32 InfoHeight = this.Reader.ReadInt32();

      if(InfoHeight < 1 || InfoHeight > 10000)
        throw new Exception("InfoHeight", new Exception(InfoHeight.ToString()));

      this.AddItem(4, "INT32", "InfoHeight", InfoHeight.ToString());
      return InfoHeight;
    }

    private Byte ReadInfoRotation() {
      Byte InfoRotation = this.Reader.ReadByte();

      if(InfoRotation > 3)
        throw new Exception("InfoRotation", new Exception(InfoRotation.ToString()));

      switch(InfoRotation) {
        case 0: {
          this.AddItem(2, "BYTE", "InfoRotation", "0°");
          break;
        }
        case 1: {
          this.AddItem(2, "BYTE", "InfoRotation", "90°");
          break;
        }
        case 2: {
          this.AddItem(2, "BYTE", "InfoRotation", "180°");
          break;
        }
        case 3: {
          this.AddItem(2, "BYTE", "InfoRotation", "270°");
          break;
        }
      }
      return InfoRotation;
    }


    private Int32 ReadImageCount() {
      Int32 ImageCount = this.Reader.ReadInt32();

      if(ImageCount < 0 || ImageCount > 36)
        throw new Exception("ImageCount", new Exception(ImageCount.ToString()));

      this.AddItem(4, "INT32", "ImageCount", ImageCount.ToString());
      return ImageCount;
    }

    private Int32 ReadImageSize(Int64 DataPosition) {
      Int32 ImageSize = this.Reader.ReadInt32();

      if(ImageSize < 0 || DataPosition + ImageSize > this.Size)
        throw new Exception("ImageSize", new Exception(ImageSize.ToString()));

      this.AddItem(4, "INT32", "ImageSize", this.GetBytesString(ImageSize));
      return ImageSize;
    }

    private Int64 ReadVideoTime(Int64 Duration) {
      Int64 VideoTime = this.Reader.ReadInt64();

      if(VideoTime < 0 || VideoTime > Duration)
        throw new Exception("VideoTime", new Exception(VideoTime.ToString()));

      TimeSpan Span = new TimeSpan(VideoTime);

      this.AddItem(8, "INT64", "VideoTime", Span.ToString(@"hh\:mm\:ss") + "." + Span.Milliseconds);
      return VideoTime;
    }

    private void ReadImage(Int32 ImageSize) {
      if(ImageSize == 0) return;

      try {
        Byte[] ImageArray = this.Reader.ReadBytes(ImageSize);

        Bitmap Image = FileViewerTDPicture.LoadImage(new MemoryStream(ImageArray));

        if(Image == null) Image = new Bitmap(new MemoryStream(ImageArray));

        this.AddItem(ImageSize, "MEMORY", "ImageFile", this.GetBytesString(ImageSize), new FileViewerObject(Image));
      } catch(Exception e) {
        throw new Exception("ImageFile", e);
      }
    }


    private void ReadBegin() {
      Int64 PreviewPos = this.ReadPreviewPos();
      Int64 PreviewSize = this.ReadPreviewSize(PreviewPos);

      UInt32 IDNumber = this.ReadIDNumber();

      Int64 Duration = this.ReadInfoDuration();
      Int64 InfoWidth = this.ReadInfoWidth();
      Int64 InfoHeight = this.ReadInfoHeight();
      Byte InfoRotation = this.ReadInfoRotation();

      Int32 ImageCount = this.ReadImageCount();

      Int64 ImagePosition = this.Position + ImageCount * 12;

      Int32[] ImageSizeArray = new Int32[ImageCount];

      for(Int32 i = 0; i < ImageCount; i++) {
        Int32 ImageSize = this.ReadImageSize(ImagePosition);
        Int64 VideoTime = this.ReadVideoTime(Duration);

        ImageSizeArray[i] = ImageSize;
        ImagePosition += ImageSize;
      }

      for(Int32 i = 0; i < ImageCount; i++) {
        this.ReadImage(ImageSizeArray[i]);
      }

      PreviewPos = this.ReadPreviewPos();
    }

    #endregion


    #region Public Methods

    public List<ListViewItem> Run(IFileViewerProgressBar Parent, String FilePath) {
      this.Parent = Parent;

      try {
        this.Stream = new FileStream(FilePath, FileMode.Open, FileAccess.Read);
        this.Reader = new BinaryReader(this.Stream);

        this.Size = this.Stream.Length;

        if(this.Size < FileViewerVideoThumbnailImageData.HeaderSize)
          throw new Exception("FileToSmall".PadRight(10), new Exception("< " + FileViewerVideoThumbnailImageData.HeaderSize + " Bytes"));

        this.Parent.BeginProgressBar(this.Size);

        this.AddItemBeginRead();

        this.ReadBegin();
      } catch(Exception e) {
        this.AddItemError(e);

        this.Close();
        return this.ItemList;
      }

      this.AddItemEndOfFile();

      this.Close();
      return this.ItemList;
    }

    #endregion
  }

}
